// scripts/copy-tinymce.js
import fs from 'fs-extra';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// مسارات المصدر والهدف
const projectRoot = path.resolve(__dirname, '..');
const sourceDir = path.join(projectRoot, 'node_modules', 'tinymce');
const targetDir = path.join(projectRoot, 'public', 'js', 'tinymce');

// الملفات والمجلدات المطلوب نسخها
const itemsToCopy = [
    'tinymce.min.js',
    'tinymce.js',
    'themes',
    'skins',
    'plugins',
    'models',
    'icons',
    'langs'
];

async function copyTinyMCE() {
    try {
        console.log('🚀 بدء نسخ ملفات TinyMCE...');
        
        // التأكد من وجود مجلد المصدر
        if (!await fs.pathExists(sourceDir)) {
            throw new Error(`مجلد TinyMCE غير موجود: ${sourceDir}`);
        }

        // إنشاء مجلد الهدف إذا لم يكن موجوداً
        await fs.ensureDir(targetDir);
        
        // نسخ كل عنصر
        for (const item of itemsToCopy) {
            const sourcePath = path.join(sourceDir, item);
            const targetPath = path.join(targetDir, item);
            
            if (await fs.pathExists(sourcePath)) {
                await fs.copy(sourcePath, targetPath, {
                    overwrite: true,
                    errorOnExist: false
                });
                console.log(`✅ تم نسخ: ${item}`);
            } else {
                console.warn(`⚠️  لم يتم العثور على: ${item}`);
            }
        }
        
        // إنشاء ملف version للتتبع
        const versionInfo = {
            copied_at: new Date().toISOString(),
            tinymce_version: await getTinyMCEVersion(),
            items_copied: itemsToCopy
        };
        
        await fs.writeJSON(
            path.join(targetDir, 'copy-info.json'), 
            versionInfo, 
            { spaces: 2 }
        );
        
        console.log('🎉 تم نسخ جميع ملفات TinyMCE بنجاح!');
        console.log(`📍 الملفات في: ${targetDir}`);
        
    } catch (error) {
        console.error('❌ خطأ في نسخ ملفات TinyMCE:', error.message);
        process.exit(1);
    }
}

async function getTinyMCEVersion() {
    try {
        const packagePath = path.join(sourceDir, 'package.json');
        if (await fs.pathExists(packagePath)) {
            const packageInfo = await fs.readJSON(packagePath);
            return packageInfo.version || 'unknown';
        }
    } catch (error) {
        console.warn('لا يمكن قراءة إصدار TinyMCE:', error.message);
    }
    return 'unknown';
}

// تشغيل الدالة إذا تم استدعاء الملف مباشرة
if (process.argv[1] === fileURLToPath(import.meta.url)) {
    copyTinyMCE();
}

export default copyTinyMCE;