@props([
    'name' => '',
    'value' => '',
    'height' => 300,
    'required' => false,
    'placeholder' => null,
    'locale' => null,
    'direction' => null,
    'language' => null
])

@php
    // Auto-detect locale and direction from app settings
    $currentLocale = $locale ?: app()->getLocale();
    
    // Enhanced RTL detection
    $isRTL = $direction === 'rtl' || 
             in_array($currentLocale, ['ar', 'he', 'fa', 'ur']) ||
             (function_exists('get_current_direction') && get_current_direction() === 'rtl') ||
             (session('direction') === 'rtl') ||
             (config('app.direction') === 'rtl');
             
    $currentDirection = $direction ?: ($isRTL ? 'rtl' : 'ltr');
    $currentLanguage = $language ?: ($isRTL ? 'ar' : 'en');
    
    // Generate unique ID
    $uniqueId = 'tinymce_' . uniqid() . '_' . str_replace(['[', ']', '.'], '_', $name);
    
    // NO CONTENT PROCESSING - Use raw value
    $rawValue = $value;
    
    // Dynamic translations
    $translations = [
        'loading_editor' => admin_trans('loading_editor'),
        'enter_content' => admin_trans('enter_content'),
        'bold' => admin_trans('bold'),
        'italic' => admin_trans('italic'),
        'underline' => admin_trans('underline'),
        'align_left' => admin_trans('align_left'),
        'align_center' => admin_trans('align_center'),
        'align_right' => admin_trans('align_right'),
        'align_justify' => admin_trans('align_justify'),
        'bullet_list' => admin_trans('bullet_list'),
        'numbered_list' => admin_trans('numbered_list'),
        'link' => admin_trans('link'),
        'code' => admin_trans('code'),
        'fullscreen' => admin_trans('fullscreen'),
        'help' => admin_trans('help'),
    ];
@endphp

<div class="tinymce-wrapper" 
     data-tinymce-wrapper="{{ $uniqueId }}" 
     data-direction="{{ $currentDirection }}"
     data-locale="{{ $currentLocale }}"
     @if($isRTL) dir="rtl" @endif>
    
    <label for="{{ $uniqueId }}" class="tinymce-label">
        @if($slot->isEmpty())
            {{ ucfirst(str_replace('_', ' ', $name)) }}
        @else
            {{ $slot }}
        @endif
        @if($required)
            <span class="text-red-500">*</span>
        @endif
    </label>
    
    <!-- Loading Overlay -->
    <div class="tinymce-loading-container">
        <div class="tinymce-loading-overlay" id="loading-{{ $uniqueId }}">
            <div class="tinymce-loading-content">
                <div class="tinymce-loading-spinner"></div>
                <span class="tinymce-loading-text">{{ $translations['loading_editor'] }}</span>
            </div>
        </div>
        
        <textarea 
            id="{{ $uniqueId }}" 
            name="{{ $name }}" 
            class="tinymce-editor-target"
            style="width: 100%; min-height: {{ $height }}px;"
            @if($required) required @endif
            @if($placeholder) placeholder="{{ $placeholder ?: $translations['enter_content'] }}" @endif
            data-tinymce-height="{{ $height }}"
            data-tinymce-direction="{{ $currentDirection }}"
            data-tinymce-language="{{ $currentLanguage }}"
            data-tinymce-locale="{{ $currentLocale }}"
            data-translations="{{ json_encode($translations) }}"
            data-no-paragraph="true"
        >{!! $rawValue !!}</textarea>
    </div>
    
    @error($name)
        <p class="tinymce-error">{{ $message }}</p>
    @enderror
</div>

@once
    @push('scripts')
        <script src="{{ asset('js/tinymce/tinymce.min.js') }}"></script>
        
        <script>
            // SIMPLIFIED TinyMCE Manager - Light Mode Only with RTL Support
            window.SecureTinyMCE = window.SecureTinyMCE || (function() {
                'use strict';
                
                const instances = new Map();
                const contentStore = new Map();
                let isInitializing = false;
                let initQueue = [];
                
                // Get direction from wrapper element
                function getElementDirection(element) {
                    const wrapper = element.closest('[data-direction]');
                    if (wrapper) {
                        return wrapper.dataset.direction;
                    }
                    return element.dataset.tinymceDirection || 'ltr';
                }
                
                // Basic security cleaning without content manipulation
                function sanitizeContent(content) {
                    if (!content || typeof content !== 'string') return '';
                    
                    // Remove only dangerous elements
                    content = content.replace(/<script[^>]*>.*?<\/script>/gims, '');
                    content = content.replace(/on\w+="[^"]*"/gim, '');
                    content = content.replace(/on\w+='[^']*'/gim, '');
                    content = content.replace(/javascript:/gim, '');
                    content = content.replace(/vbscript:/gim, '');
                    
                    return content.trim();
                }
                
                // Preserve user content exactly
                function preserveUserContent(content) {
                    if (!content || typeof content !== 'string') return '';
                    return sanitizeContent(content);
                }
                
                // Enhanced secure configuration - Light Mode Only
                const getSecureConfig = (element, translations) => {
                    const elementDirection = getElementDirection(element);
                    const language = element.dataset.tinymceLanguage || (elementDirection === 'rtl' ? 'ar' : 'en');
                    const height = parseInt(element.dataset.tinymceHeight) || 300;
                    
                    // Dynamic toolbar based on direction
                    const toolbarConfig = elementDirection === 'rtl' 
                        ? 'undo redo | styles | fontsize | bold italic underline | forecolor backcolor | alignright aligncenter alignleft alignjustify | bullist numlist | link | ltr rtl | code fullscreen help'
                        : 'undo redo | styles | fontsize | bold italic underline | forecolor backcolor | alignleft aligncenter alignright alignjustify | bullist numlist | link | ltr rtl | code fullscreen help';
                    
                    return {
                        license_key: 'gpl',
                        promotion: false,
                        branding: false,
                        notifications: false,
                        menubar: false,
                        statusbar: false,
                        toolbar_sticky: false,
                        inline: false,
                        
                        plugins: [
                            'lists', 'link', 'paste', 'searchreplace', 
                            'visualblocks', 'code', 'fullscreen', 'help', 'wordcount', 'directionality'
                        ],
                        
                        toolbar: toolbarConfig,
                        
                        fontsize_formats: '8px 10px 12px 14px 16px 18px 20px 24px 30px 36px 48px 60px',
                        
                        style_formats: [
                            { title: translations.heading_main || 'Main Heading', block: 'h1' },
                            { title: translations.heading_sub || 'Sub Heading', block: 'h2' },
                            { title: translations.heading_small || 'Small Heading', block: 'h3' },
                            { title: translations.paragraph || 'Paragraph', block: 'p' },
                            { title: translations.highlighted_text || 'Highlighted Text', inline: 'strong' }
                        ],
                        
                        // Prevent automatic paragraph wrapping
                        forced_root_block: false,
                        force_p_newlines: false,
                        force_br_newlines: true,
                        convert_newlines_to_brs: false,
                        remove_script_host: true,
                        convert_urls: false,
                        remove_linebreaks: false,
                        entity_encoding: 'html',
                        
                        verify_html: false,
                        cleanup: false,
                        apply_source_formatting: false,
                        indent: false,
                        
                        valid_elements: '*[*]',
                        invalid_elements: 'script,object,embed,iframe,form,input,textarea,button,select,option',
                        extended_valid_elements: 'span[*],div[*],p[*],br,strong,em,u,h1,h2,h3,h4,h5,h6,ul,ol,li,a[href|target|title]',
                        
                        paste_data_images: false,
                        paste_as_text: false,
                        paste_remove_styles: false,
                        paste_remove_styles_if_webkit: false,
                        paste_webkit_styles: 'all',
                        paste_retain_style_properties: 'all',
                        
                        paste_preprocess: function (plugin, args) {
                            args.content = preserveUserContent(args.content);
                        },
                        
                        paste_postprocess: function (plugin, args) {
                            const cleanContent = preserveUserContent(args.node.innerHTML);
                            args.node.innerHTML = cleanContent;
                        },
                        
                        // Light Mode content style with RTL support
                        content_style: `
                            body { 
                                font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; 
                                font-size: 14px; 
                                line-height: 1.6; 
                                margin: 20px; 
                                color: #1f2937;
                                background-color: #ffffff;
                                direction: ${elementDirection};
                                text-align: ${elementDirection === 'rtl' ? 'right' : 'left'};
                            }
                            
                            h1, h2, h3, h4, h5, h6 { 
                                margin: 1em 0 0.5em 0; 
                                color: #111827;
                            }
                            
                            p { 
                                margin: 0 0 1em 0; 
                                color: #374151;
                            }
                            
                            ul, ol { 
                                margin: 0 0 1em 0; 
                                padding-${elementDirection === 'rtl' ? 'right' : 'left'}: 2em;
                                color: #374151;
                            }
                            
                            li { 
                                margin-bottom: 0.5em;
                                color: #374151;
                            }
                            
                            a { 
                                color: #3b82f6 !important;
                            }
                        `,
                        
                        browser_spellcheck: true,
                        resize: true,
                        min_height: height,
                        height: height,
                        directionality: elementDirection,
                        language: language,
                        
                        // Light mode only
                        skin: 'oxide',
                        content_css: 'default',
                        
                        setup: function(editor) {
                            setupEditor(editor, element, translations);
                        },
                        
                        init_instance_callback: function(editor) {
                            onEditorInit(editor, element);
                        }
                    };
                };
                
                function setupEditor(editor, element, translations) {
                    const editorId = element.id;
                    const elementDirection = getElementDirection(element);
                    
                    editor.on('init', function() {
                        const editorBody = editor.getBody();
                        const editorContainer = editor.getContainer();
                        
                        // Set direction
                        if (elementDirection === 'rtl') {
                            editorBody.setAttribute('dir', 'rtl');
                            editorBody.style.textAlign = 'right';
                            editorBody.style.direction = 'rtl';
                            
                            if (editorContainer) {
                                editorContainer.classList.add('tox-rtl');
                                editorContainer.setAttribute('dir', 'rtl');
                            }
                        } else {
                            editorBody.setAttribute('dir', 'ltr');
                            editorBody.style.textAlign = 'left';
                            editorBody.style.direction = 'ltr';
                            
                            if (editorContainer) {
                                editorContainer.classList.remove('tox-rtl');
                                editorContainer.setAttribute('dir', 'ltr');
                            }
                        }
                        
                        // Load original content
                        const originalContent = element.value || '';
                        if (originalContent && originalContent.trim()) {
                            editor.setContent(originalContent, {format: 'raw'});
                            contentStore.set(editorId, originalContent);
                            syncToFormField(editorId, originalContent);
                        }
                        
                        hideLoading(editorId);
                        element.classList.add('ready');
                        
                        removeOnlyScriptTags(editorBody);
                    });
                    
                    // Content change handling
                    let saveTimeout;
                    editor.on('change keyup blur setcontent', function() {
                        clearTimeout(saveTimeout);
                        saveTimeout = setTimeout(() => {
                            saveEditorContentRaw(editor);
                        }, 200);
                    });
                    
                    editor.on('blur', function() {
                        saveEditorContentRaw(editor);
                    });
                    
                    editor.on('BeforeSetContent', function(e) {
                        e.content = sanitizeContent(e.content);
                    });
                    
                    editor.on('SetContent', function(e) {
                        const editorBody = editor.getBody();
                        removeOnlyScriptTags(editorBody);
                    });
                    
                    editor.on('init', function() {
                        hideNotifications(editor);
                    });
                    
                    // Direction monitoring
                    const directionObserver = new MutationObserver(function(mutations) {
                        let needsUpdate = false;
                        
                        mutations.forEach(function(mutation) {
                            if (mutation.type === 'attributes' && 
                                mutation.attributeName === 'data-direction') {
                                needsUpdate = true;
                            }
                        });
                        
                        if (needsUpdate) {
                            setTimeout(() => {
                                if (editor && !editor.destroyed) {
                                    updateEditorDirection(editor, element);
                                }
                            }, 100);
                        }
                    });
                    
                    const wrapper = element.closest('[data-direction]');
                    if (wrapper) {
                        directionObserver.observe(wrapper, {
                            attributes: true,
                            attributeFilter: ['data-direction']
                        });
                    }
                }
                
                function updateEditorDirection(editor, element) {
                    if (!editor || editor.destroyed) return;
                    
                    const newDirection = getElementDirection(element);
                    const editorBody = editor.getBody();
                    const editorContainer = editor.getContainer();
                    
                    if (editorBody) {
                        editorBody.setAttribute('dir', newDirection);
                        editorBody.style.direction = newDirection;
                        editorBody.style.textAlign = newDirection === 'rtl' ? 'right' : 'left';
                    }
                    
                    if (editorContainer) {
                        if (newDirection === 'rtl') {
                            editorContainer.classList.add('tox-rtl');
                            editorContainer.setAttribute('dir', 'rtl');
                        } else {
                            editorContainer.classList.remove('tox-rtl');
                            editorContainer.setAttribute('dir', 'ltr');
                        }
                    }
                }
                
                function removeOnlyScriptTags(element) {
                    if (!element) return;
                    
                    const scripts = element.querySelectorAll('script');
                    scripts.forEach(script => script.remove());
                    
                    const elements = element.querySelectorAll('*');
                    elements.forEach(el => {
                        const attributes = el.attributes;
                        for (let i = attributes.length - 1; i >= 0; i--) {
                            const attr = attributes[i];
                            if (attr.name.startsWith('on') && attr.name.length > 2) {
                                el.removeAttribute(attr.name);
                            }
                        }
                    });
                }
                
                function saveEditorContentRaw(editor) {
                    if (!editor || editor.destroyed) return;
                    
                    try {
                        const rawContent = editor.getContent({format: 'raw'});
                        const secureContent = sanitizeContent(rawContent);
                        
                        contentStore.set(editor.id, secureContent);
                        
                        const textarea = document.getElementById(editor.id);
                        if (textarea) {
                            textarea.value = secureContent;
                            
                            const changeEvent = new Event('change', { bubbles: true });
                            textarea.dispatchEvent(changeEvent);
                        }
                        
                        syncToFormField(editor.id, secureContent);
                        
                    } catch (error) {
                        console.error('Error saving editor content:', editor.id, error);
                    }
                }
                
                function syncToFormField(editorId, content) {
                    const hiddenField = document.querySelector(`input[name="${editorId}"]`);
                    if (hiddenField) {
                        hiddenField.value = content;
                    }
                    
                    const translationFields = document.querySelectorAll(`input[name*="${editorId}"], textarea[name*="${editorId}"]`);
                    translationFields.forEach(field => {
                        if (field.id !== editorId) {
                            field.value = content;
                        }
                    });
                }
                
                function hideNotifications(editor) {
                    setTimeout(() => {
                        const container = editor.editorContainer;
                        if (container) {
                            const notifications = container.querySelectorAll('.tox-notification, .tox-promotion, .tox-promotion-link');
                            notifications.forEach(notification => {
                                notification.style.display = 'none';
                                notification.remove();
                            });
                        }
                    }, 100);
                }
                
                function onEditorInit(editor, element) {
                    instances.set(editor.id, editor);
                    
                    const savedContent = contentStore.get(editor.id);
                    if (savedContent) {
                        editor.setContent(savedContent, {format: 'raw'});
                        syncToFormField(editor.id, savedContent);
                    }
                    
                    hideLoading(editor.id);
                }
                
                function showLoading(editorId) {
                    const overlay = document.getElementById(`loading-${editorId}`);
                    if (overlay) {
                        overlay.classList.remove('hidden');
                    }
                }
                
                function hideLoading(editorId) {
                    const overlay = document.getElementById(`loading-${editorId}`);
                    if (overlay) {
                        requestAnimationFrame(() => {
                            overlay.classList.add('hidden');
                        });
                    }
                }
                
                async function initSingleEditor(textarea) {
                    if (!textarea || textarea.dataset.tinymceInitialized === 'true') {
                        return;
                    }
                    
                    const editorId = textarea.id;
                    if (instances.has(editorId)) {
                        return;
                    }
                    
                    showLoading(editorId);
                    
                    try {
                        let translations = {};
                        try {
                            translations = JSON.parse(textarea.dataset.translations || '{}');
                        } catch (e) {
                            console.warn('Failed to parse translations for', editorId);
                        }
                        
                        const config = getSecureConfig(textarea, translations);
                        config.selector = '#' + editorId;
                        
                        await tinymce.init(config);
                        textarea.dataset.tinymceInitialized = 'true';
                        
                    } catch (error) {
                        console.error('Error initializing editor:', editorId, error);
                        hideLoading(editorId);
                    }
                }
                
                async function initAllEditors() {
                    if (isInitializing) {
                        return;
                    }
                    
                    isInitializing = true;
                    
                    try {
                        const textareas = document.querySelectorAll('.tinymce-editor-target:not([data-tinymce-initialized])');
                        
                        for (let i = 0; i < textareas.length; i++) {
                            await initSingleEditor(textareas[i]);
                            if (i < textareas.length - 1) {
                                await new Promise(resolve => setTimeout(resolve, 100));
                            }
                        }
                        
                    } catch (error) {
                        console.error('Error initializing editors:', error);
                    } finally {
                        isInitializing = false;
                        
                        while (initQueue.length > 0) {
                            const callback = initQueue.shift();
                            if (typeof callback === 'function') {
                                callback();
                            }
                        }
                    }
                }
                
                function saveAll() {
                    instances.forEach((editor, editorId) => {
                        if (editor && !editor.destroyed) {
                            saveEditorContentRaw(editor);
                        }
                    });
                }
                
                function destroySingle(editorId) {
                    const editor = instances.get(editorId);
                    if (editor && !editor.destroyed) {
                        try {
                            saveEditorContentRaw(editor);
                            
                            const textarea = document.getElementById(editorId);
                            if (textarea) {
                                textarea.dataset.tinymceInitialized = 'false';
                                textarea.classList.remove('ready');
                            }
                            
                            editor.destroy();
                            instances.delete(editorId);
                            
                        } catch (error) {
                            console.error('Error destroying editor:', editorId, error);
                        }
                    }
                }
                
                function destroyAll() {
                    const editorIds = Array.from(instances.keys());
                    editorIds.forEach(editorId => {
                        destroySingle(editorId);
                    });
                    
                    document.querySelectorAll('.tinymce-editor-target[data-tinymce-initialized]').forEach(textarea => {
                        textarea.dataset.tinymceInitialized = 'false';
                        textarea.classList.remove('ready');
                    });
                }
                
                async function reinitialize() {
                    saveAll();
                    destroyAll();
                    await new Promise(resolve => setTimeout(resolve, 300));
                    await initAllEditors();
                }
                
                function bindEvents() {
                    document.addEventListener('submit', function(e) {
                        saveAll();
                    });
                    
                    let tabChangeTimeout;
                    document.addEventListener('click', function(e) {
                        const tabButton = e.target.closest('[x-on\\:click*="activeTab"], button[\\@click*="activeTab"], .nav-link, [role="tab"], button[type="button"][class*="tab"]');
                        
                        if (tabButton) {
                            saveAll();
                            
                            if (tabChangeTimeout) {
                                clearTimeout(tabChangeTimeout);
                            }
                            
                            tabChangeTimeout = setTimeout(() => {
                                initAllEditors();
                            }, 300);
                        }
                    });
                    
                    // DOM mutation observer
                    const observer = new MutationObserver(function(mutations) {
                        let hasNewEditors = false;
                        
                        mutations.forEach(function(mutation) {
                            mutation.addedNodes.forEach(function(node) {
                                if (node.nodeType === 1) {
                                    if (node.classList && node.classList.contains('tinymce-editor-target') ||
                                        node.querySelector && node.querySelector('.tinymce-editor-target:not([data-tinymce-initialized])')) {
                                        hasNewEditors = true;
                                    }
                                }
                            });
                        });
                        
                        if (hasNewEditors) {
                            setTimeout(() => {
                                if (isInitializing) {
                                    initQueue.push(() => initAllEditors());
                                } else {
                                    initAllEditors();
                                }
                            }, 200);
                        }
                    });
                    
                    observer.observe(document.body, {
                        childList: true,
                        subtree: true
                    });
                    
                    window.addEventListener('beforeunload', function() {
                        saveAll();
                    });
                    
                    // Periodic save every 30 seconds
                    setInterval(() => {
                        saveAll();
                    }, 30000);
                }
                
                function initOnLoad() {
                    if (typeof tinymce === 'undefined') {
                        console.error('TinyMCE not loaded');
                        return;
                    }
                    
                    setTimeout(() => {
                        initAllEditors();
                    }, 100);
                }
                
                // Initialize events
                bindEvents();
                
                // Initialize on load
                if (document.readyState === 'loading') {
                    document.addEventListener('DOMContentLoaded', initOnLoad);
                } else {
                    initOnLoad();
                }
                
                // Global functions for external use
                window.saveAllEditors = saveAll;
                window.reinitializeEditors = reinitialize;
                
                return {
                    init: initAllEditors,
                    destroy: destroyAll,
                    save: saveAll,
                    reinit: reinitialize,
                    getInstance: (id) => instances.get(id),
                    getContent: (id) => contentStore.get(id),
                    destroySingle: destroySingle,
                    initSingle: initSingleEditor,
                    sanitize: sanitizeContent
                };
            })();
        </script>
    @endpush
@endonce
