@extends('admin.layouts.app')

@section('title', admin_trans('occasions'))
@section('page-title', admin_trans('occasions'))

@section('content')

<div class="space-y-6" x-data="occasionsPage()">

    <!-- Header: Search & Add Button -->
    <div class="card">
        <div class="flex flex-col sm:flex-row gap-4 items-start sm:items-center justify-between">
            
            <!-- Search -->
            <div class="flex-1 w-full max-w-lg">
                <form method="GET" action="{{ route('admin.occasions.index') }}">
                    <!-- Keep other filters -->
                    @if(request('locale_id'))<input type="hidden" name="locale_id" value="{{ request('locale_id') }}">@endif
                    @if(request('status'))<input type="hidden" name="status" value="{{ request('status') }}">@endif
                    @if(request('featured'))<input type="hidden" name="featured" value="{{ request('featured') }}">@endif
                    
                    <x-search-input 
                        name="search" 
                        :value="request('search')" 
                        :placeholder="admin_trans('search_occasions')" 
                        class="w-full"
                    />
                </form>
            </div>

            
                    <!-- Filters Row -->
        <div class="flex flex-wrap items-center gap-4 "">
            
            <!-- Language Filter -->
            @if($currentCountry && $availableLocales->count() > 1)
                <form method="GET" action="{{ route('admin.occasions.index') }}" class="flex items-center gap-2">
                    @if(request('search'))<input type="hidden" name="search" value="{{ request('search') }}">@endif
                    @if(request('status'))<input type="hidden" name="status" value="{{ request('status') }}">@endif
                    @if(request('featured'))<input type="hidden" name="featured" value="{{ request('featured') }}">@endif
                    
                    <label class="text-sm font-medium text-tertiary">{{ admin_trans('language') }}:</label>
                    <select name="locale_id" onchange="this.form.submit()" class="input text-sm py-2 px-3">
                        <option value="">{{ admin_trans('all_languages') }}</option>
                        @foreach($availableLocales as $locale)
                            <option value="{{ $locale->id }}" {{ request('locale_id') == $locale->id ? 'selected' : '' }}>
                                {{ $locale->language->name ?? $locale->name }}
                            </option>
                        @endforeach
                    </select>
                </form>
            @endif

            <!-- Status Filter -->
            <form method="GET" action="{{ route('admin.occasions.index') }}" class="flex items-center gap-2">
                @if(request('search'))<input type="hidden" name="search" value="{{ request('search') }}">@endif
                @if(request('locale_id'))<input type="hidden" name="locale_id" value="{{ request('locale_id') }}">@endif
                @if(request('featured'))<input type="hidden" name="featured" value="{{ request('featured') }}">@endif
                
                <label class="text-sm font-medium text-tertiary">{{ admin_trans('status') }}:</label>
                <select name="status" onchange="this.form.submit()" class="input text-sm py-2 px-3">
                    <option value="">{{ admin_trans('all_statuses') }}</option>
                    <option value="active" {{ request('status') === 'active' ? 'selected' : '' }}>{{ admin_trans('active') }}</option>
                    <option value="draft" {{ request('status') === 'draft' ? 'selected' : '' }}>{{ admin_trans('draft') }}</option>
                </select>
            </form>

            <!-- Featured Filter -->
            <form method="GET" action="{{ route('admin.occasions.index') }}" class="flex items-center gap-2">
                @if(request('search'))<input type="hidden" name="search" value="{{ request('search') }}">@endif
                @if(request('locale_id'))<input type="hidden" name="locale_id" value="{{ request('locale_id') }}">@endif
                @if(request('status'))<input type="hidden" name="status" value="{{ request('status') }}">@endif
                
                <label class="text-sm font-medium text-tertiary">{{ admin_trans('featured') }}:</label>
                <select name="featured" onchange="this.form.submit()" class="input text-sm py-2 px-3">
                    <option value="">{{ admin_trans('all') }}</option>
                    <option value="1" {{ request('featured') === '1' ? 'selected' : '' }}>{{ admin_trans('featured') }}</option>
                    <option value="0" {{ request('featured') === '0' ? 'selected' : '' }}>{{ admin_trans('not_featured') }}</option>
                </select>
            </form>

            <!-- Clear Filters -->
            @if(request()->hasAny(['search', 'locale_id', 'status', 'featured']))
                <x-button variant="outline" size="sm" icon="x-circle" href="{{ route('admin.occasions.index') }}">
                    {{ admin_trans('clear_filters') }}
                </x-button>
            @endif
        </div>
            
            <!-- Add Button -->
            <x-button variant="primary" icon="plus" @click="$dispatch('open-modal', 'create-occasion-modal')">
                {{ admin_trans('add_occasion') }}
            </x-button>
        </div>
    </div>

    <!-- Bulk Actions -->
    <div x-show="selectedItems.length > 0" x-cloak>
        <x-alert variant="info">
            <div class="flex items-center justify-between w-full">
                <span x-text="`${selectedItems.length} {{ admin_trans('items_selected') }}`"></span>
                <div class="flex gap-2">
                    <x-button size="sm" variant="outline" @click="performBulkAction('activate')">{{ admin_trans('activate') }}</x-button>
                    <x-button size="sm" variant="outline" @click="performBulkAction('deactivate')">{{ admin_trans('deactivate') }}</x-button>
                    <x-button size="sm" variant="outline" @click="performBulkAction('feature')">{{ admin_trans('feature') }}</x-button>
                    <x-button size="sm" variant="outline" @click="performBulkAction('unfeature')">{{ admin_trans('unfeature') }}</x-button>
                    <x-button size="sm" variant="danger" @click="performBulkAction('delete')">{{ admin_trans('delete') }}</x-button>
                    <x-button size="sm" variant="outline" @click="clearSelection()">{{ admin_trans('clear') }}</x-button>
                </div>
            </div>
        </x-alert>
    </div>

    <!-- Occasions Table -->
    <x-card>
        @if ($occasions->isEmpty())
            @if($currentCountry)
                <x-empty-state icon="calendar" :title="admin_trans('no_occasions_yet')" :description="admin_trans('start_by_adding_first_occasion')">
                    <x-button variant="primary" icon="plus" @click="$dispatch('open-modal', 'create-occasion-modal')">
                        {{ admin_trans('add_occasion') }}
                    </x-button>
                </x-empty-state>
            @else
                <x-empty-state icon="globe" :title="admin_trans('select_country_first')" :description="admin_trans('select_country_to_manage_occasions')">
                    <p class="text-sm text-tertiary mt-2">{{ admin_trans('use_country_filter_navbar') }}</p>
                </x-empty-state>
            @endif
        @else
            <x-enhanced-table 
                :headers="[
                    'select' => ['text' => '', 'align' => 'center', 'width' => '50px'],
                    'order' => ['text' => admin_trans('order'), 'align' => 'center', 'width' => '80px'],
                    'icon' => ['text' => admin_trans('icon'), 'align' => 'center', 'width' => '60px'],
                    'name' => ['text' => admin_trans('name'), 'align' => 'left', 'width' => '200px'],
                    'languages' => ['text' => admin_trans('languages'), 'align' => 'center', 'width' => '120px'],
                    'featured' => ['text' => admin_trans('featured'), 'align' => 'center', 'width' => '80px'],
                    'status' => ['text' => admin_trans('status'), 'align' => 'center', 'width' => '80px'],
                    'actions' => ['text' => admin_trans('actions'), 'align' => 'center', 'width' => '100px']
                ]"
                :responsive="true"
                :striped="true"
            >
                @foreach ($occasions as $occasion)
                    <tr class="occasion-row hover:bg-primary-light transition-colors" data-id="{{ $occasion->id }}">

                        <!-- Select -->
                        <x-table-cell type="actions">
                            <input type="checkbox" value="{{ $occasion->id }}" 
                                @change="toggleItem({{ $occasion->id }})" 
                                :checked="selectedItems.includes({{ $occasion->id }})" 
                                class="checkbox">
                        </x-table-cell>

                        <!-- Order -->
                        <x-table-cell type="number">
                            <div class="flex items-center justify-center gap-2">
                                <button class="drag-handle cursor-move p-1 rounded hover:bg-primary-light">
                                    <i data-lucide="grip-vertical" class="w-3 h-3 text-tertiary"></i>
                                </button>
                                <span class="text-sm font-medium text-tertiary">{{ $occasion->order ?? 0 }}</span>
                            </div>
                        </x-table-cell>

                        <!-- Icon -->
                        <x-table-cell type="actions">
                            @if ($occasion->getFirstMediaUrl('icon'))
                                <img src="{{ $occasion->getFirstMediaUrl('icon') }}" alt="Icon" class="w-8 h-8 object-cover rounded-lg border mx-auto">
                            @else
                                <div class="w-8 h-8 bg-gray-200 rounded-lg border flex items-center justify-center mx-auto">
                                    <i data-lucide="calendar" class="w-4 h-4 text-gray-400"></i>
                                </div>
                            @endif
                        </x-table-cell>

                        <!-- Name & Description -->
                        <x-table-cell type="text">
                            @php
                                $translation = $occasion->translations->first();
                            @endphp
                            <div>
                                <span class="text-sm font-medium text-primary">
                                    {{ $translation->name ?? 'Occasion #' . $occasion->id }}
                                </span>
                                @if ($translation && $translation->description)
                                    <div class="text-xs text-tertiary mt-1 line-clamp-2">
                                        {!! strip_tags(Str::limit($translation->description, 60)) !!}
                                    </div>
                                @endif
                            </div>
                        </x-table-cell>

                        <!-- Languages -->
                        <x-table-cell type="badge">
                            <div class="flex items-center justify-center gap-1 flex-wrap">
                                @foreach($occasion->translations->take(3) as $trans)
                                    <x-badge variant="info" class="text-xs">
                                        {{ $trans->locale->code ?? 'N/A' }}
                                    </x-badge>
                                @endforeach
                                @if($occasion->translations->count() > 3)
                                    <x-badge variant="secondary" class="text-xs">
                                        +{{ $occasion->translations->count() - 3 }}
                                    </x-badge>
                                @endif
                            </div>
                        </x-table-cell>

                        <!-- Featured -->
                        <x-table-cell type="actions">
                            <form method="POST" action="{{ route('admin.occasions.toggle-featured', $occasion) }}">
                                @csrf @method('PATCH')
                                <button type="submit" class="p-1 rounded-lg transition-colors {{ $occasion->is_featured ? 'text-yellow-500 hover:bg-yellow-50' : 'text-gray-400 hover:bg-gray-50' }}">
                                    <i data-lucide="star" class="w-4 h-4 {{ $occasion->is_featured ? 'fill-current' : '' }}"></i>
                                </button>
                            </form>
                        </x-table-cell>

                        <!-- Status -->
                        <x-table-cell type="actions">
                            <form method="POST" action="{{ route('admin.occasions.toggle-status', $occasion) }}">
                                @csrf @method('PATCH')
                                <x-toggle name="status_toggle_{{ $occasion->id }}" :checked="$occasion->status === 'active'" size="sm" variant="success" onchange="this.form.submit()" />
                            </form>
                        </x-table-cell>

                        <!-- Actions -->
                        <x-table-cell type="actions">
                            <div class="flex items-center justify-center gap-3">
                                <button onclick="editOccasion({{ $occasion->id }})" class="p-1.5 rounded-lg hover:bg-blue-50 text-blue-600" title="{{ admin_trans('edit') }}">
                                    <i data-lucide="edit-3" class="w-4 h-4"></i>
                                </button>
                                <button onclick="confirmDelete({{ $occasion->id }}, '{{ addslashes($translation->name ?? 'Occasion #' . $occasion->id) }}')" class="p-1.5 rounded-lg hover:bg-red-50 text-red-600" title="{{ admin_trans('delete') }}">
                                    <i data-lucide="trash-2" class="w-4 h-4"></i>
                                </button>
                            </div>
                        </x-table-cell>
                    </tr>
                @endforeach
            </x-enhanced-table>

            @if ($occasions->hasPages())
                <div class="mt-6">
                    <x-pagination :paginator="$occasions" />
                </div>
            @endif
        @endif
    </x-card>
</div>

<!-- Create Modal -->
<x-modal name="create-occasion-modal" maxWidth="2xl">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('add_occasion') }}</h3>
            <button @click="$dispatch('close-modal', 'create-occasion-modal')" class="p-2 rounded-lg hover:bg-primary-light">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        @if($currentCountry && $availableLocales->count() > 0)
            <form method="POST" action="{{ route('admin.occasions.store') }}" enctype="multipart/form-data" class="space-y-6">
                @csrf
                
                <!-- Auto-select current country and language -->
                <input type="hidden" name="countries[]" value="{{ $currentCountry->id }}">
                @if(request('locale_id'))
                    <input type="hidden" name="selected_locale_id" value="{{ request('locale_id') }}">
                @endif
                
                <!-- Current Context -->
                <x-alert variant="info" icon="info">
                    <div class="flex items-center gap-3">
                        <span class="text-2xl mb-2">{{ $currentCountry->flag_emoji }}</span>
                        <div>
                            <h4 class="font-semibold">{{ admin_trans('creating_occasion_for') }}</h4>
                            <p class="text-sm opacity-75">{{ $currentCountry->getName() }}</p>
                            @if(request('locale_id'))
                                @php
                                    $selectedLocale = $availableLocales->find(request('locale_id'));
                                @endphp
                                @if($selectedLocale)
                                    <p class="text-sm opacity-75">{{ admin_trans('language') }}: {{ $selectedLocale->language->name ?? $selectedLocale->name }}</p>
                                @endif
                            @endif
                        </div>
                    </div>
                </x-alert>

                <!-- Multi-language Tabs -->
                <x-tabs defaultTab="{{ request('locale_id') ?: $availableLocales->first()->id }}">
                    <div class="flex flex-wrap gap-2 border-b mb-4" style="border-color: var(--color-border);">
                        @foreach($availableLocales as $locale)
                            <button type="button" @click="activeTab = '{{ $locale->id }}'; $event.preventDefault();"
                                :class="activeTab === '{{ $locale->id }}' ? 'text-primary bg-primary-light' : 'text-tertiary hover:text-primary'"
                                class="flex items-center gap-2 px-4 py-2 border-b transition-colors font-medium">
                                <i data-lucide="globe" class="w-4 h-4"></i>
                                {{ $locale->language->name ?? $locale->name }}
                                @if($locale->direction === 'rtl')
                                    <span class="text-xs">(ع)</span>
                                @endif
                            </button>
                        @endforeach
                    </div>

                    @foreach($availableLocales as $locale)
                        <x-tab-content :tab="$locale->id">
                            <div class="space-y-4" @if($locale->direction === 'rtl') dir="rtl" @endif>
                                <x-input 
                                    :label="admin_trans('occasion_name')" 
                                    name="translations[{{ $locale->id }}][name]" 
                                    placeholder="{{ admin_trans('enter_occasion_name') }}" 
                                    :required="$loop->first" 
                                    value="{{ old('translations.' . $locale->id . '.name') }}"
                                />
                                
                                <x-tinymce
                                    name="translations[{{ $locale->id }}][description]"
                                    :height="200"
                                    placeholder="{{ admin_trans('enter_description') }}"
                                    :value="old('translations.' . $locale->id . '.description')"
                                    :required="false"
                                    :locale="$locale->code"
                                    :direction="$locale->direction"
                                    :language="$locale->direction === 'rtl' ? 'ar' : 'en'"
                                >
                                    {{ admin_trans('description') }}
                                </x-tinymce>
                            </div>
                        </x-tab-content>
                    @endforeach
                </x-tabs>

                <!-- Media & Settings -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <x-file-upload :label="admin_trans('occasion_icon')" name="icon" accept="image/*" maxSize="2MB" />
                    <x-file-upload :label="admin_trans('occasion_banner')" name="banner" accept="image/*" maxSize="5MB" />
                </div>

                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <x-input :label="admin_trans('order')" name="order" type="number" value="{{ old('order', 0) }}" />
                    <x-select :label="admin_trans('status')" name="status" :required="true">
                        <option value="draft" {{ old('status') === 'draft' ? 'selected' : '' }}>{{ admin_trans('draft') }}</option>
                        <option value="active" {{ old('status', 'active') === 'active' ? 'selected' : '' }}>{{ admin_trans('active') }}</option>
                    </x-select>
                </div>

                <x-toggle name="is_featured" :label="admin_trans('featured_occasion')" value="1" :checked="old('is_featured')" />

                <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                    <x-button variant="outline" type="button" @click="$dispatch('close-modal', 'create-occasion-modal')">{{ admin_trans('cancel') }}</x-button>
                    <x-button variant="primary" type="submit" icon="check">{{ admin_trans('create') }}</x-button>
                </div>
            </form>
        @else
            <x-empty-state icon="globe" :title="admin_trans('select_country_first')" :description="admin_trans('use_country_filter_navbar')">
                <x-button variant="outline" @click="$dispatch('close-modal', 'create-occasion-modal')">{{ admin_trans('close') }}</x-button>
            </x-empty-state>
        @endif
    </div>
</x-modal>

<!-- Edit Modal - FIXED VERSION WITH PROPER TINYMCE DATA LOADING -->
<x-modal name="edit-occasion-modal" maxWidth="2xl">
    <div class="p-6" x-data="editOccasionModal()">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('edit_occasion') }}</h3>
            <button @click="$dispatch('close-modal', 'edit-occasion-modal')" class="p-2 rounded-lg hover:bg-primary-light">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <div x-show="loading">
            <x-loading text="{{ admin_trans('loading') }}..." />
        </div>

        <div x-show="!loading && occasionData" x-cloak>
            <form :action="`/admin/occasions/${occasionData?.id}`" method="POST" enctype="multipart/form-data" class="space-y-6">
                @csrf @method('PUT')
                
                <input type="hidden" name="countries[]" value="{{ $currentCountry->id ?? '' }}">
                
                @if($availableLocales->count() > 0)
                    <x-tabs defaultTab="{{ $availableLocales->first()->id }}">
                        <div class="flex flex-wrap gap-2 border-b mb-4" style="border-color: var(--color-border);">
                            @foreach($availableLocales as $locale)
                                <button type="button" @click="activeTab = '{{ $locale->id }}'; $event.preventDefault();"
                                    :class="activeTab === '{{ $locale->id }}' ? 'text-primary bg-primary-light' : 'text-tertiary hover:text-primary'"
                                    class="flex items-center gap-2 px-4 py-2 border-b transition-colors font-medium">
                                    <i data-lucide="globe" class="w-4 h-4"></i>
                                    {{ $locale->language->name ?? $locale->name }}
                                    @if($locale->direction === 'rtl')
                                        <span class="text-xs">(ع)</span>
                                    @endif
                                </button>
                            @endforeach
                        </div>

                        @foreach($availableLocales as $locale)
                            @php
                                $isRTL = $locale->direction === 'rtl';
                            @endphp
                            <x-tab-content :tab="$locale->id">
                                <div class="space-y-4" @if($isRTL) dir="rtl" @endif>
                                    <x-input 
                                        :label="admin_trans('occasion_name')" 
                                        :name="'translations[' . $locale->id . '][name]'" 
                                        placeholder="{{ admin_trans('enter_occasion_name') }}" 
                                        :required="$loop->first"
                                        x-bind:value="getTranslationValue({{ $locale->id }}, 'name')"
                                        @input="setTranslationValue({{ $locale->id }}, 'name', $event.target.value)"
                                    />
                                    
                                    {{-- FIXED: TinyMCE with proper data binding --}}
                                    <div class="tinymce-edit-container" data-locale-id="{{ $locale->id }}">
                                        <x-tinymce
                                            name="translations[{{ $locale->id }}][description]"
                                            :height="200"
                                            placeholder="{{ admin_trans('enter_description') }}"
                                            :value="''"
                                            :required="false"
                                            :locale="$locale->code"
                                            :direction="$locale->direction"
                                            :language="$locale->direction === 'rtl' ? 'ar' : 'en'"
                                        >
                                            {{ admin_trans('description') }}
                                        </x-tinymce>
                                    </div>
                                </div>
                            </x-tab-content>
                        @endforeach
                    </x-tabs>
                @endif

                <!-- Current Media Display -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                    <div>
                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('current_icon') }}</label>
                        <div x-show="occasionData?.icon_url" class="mb-3">
                            <div class="relative inline-block">
                                <img :src="occasionData?.icon_url" class="w-20 h-20 object-cover rounded-lg border">
                                <button 
                                    type="button" 
                                    @click="removeCurrentMedia('icon')"
                                    class="absolute top-1 right-1 w-6 h-6 rounded-full flex items-center justify-center text-white text-sm hover:bg-red-600" 
                                    style="background-color: var(--color-danger-500);"
                                >
                                    ×
                                </button>
                            </div>
                        </div>
                        <div x-show="!occasionData?.icon_url" class="mb-3">
                            <p class="text-sm text-tertiary">{{ admin_trans('no_icon') }}</p>
                        </div>

                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('new_icon') }}</label>
                        <x-file-upload :label="''" name="icon" accept="image/*" maxSize="2MB" />
                    </div>
                    
                    <div>
                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('current_banner') }}</label>
                        <div x-show="occasionData?.banner_url" class="mb-3">
                            <div class="relative inline-block">
                                <img :src="occasionData?.banner_url" class="w-32 h-16 object-cover rounded-lg border">
                                <button 
                                    type="button" 
                                    @click="removeCurrentMedia('banner')"
                                    class="absolute top-1 right-1 w-6 h-6 rounded-full flex items-center justify-center text-white text-sm hover:bg-red-600" 
                                    style="background-color: var(--color-danger-500);"
                                >
                                    ×
                                </button>
                            </div>
                        </div>
                        <div x-show="!occasionData?.banner_url" class="mb-3">
                            <p class="text-sm text-tertiary">{{ admin_trans('no_banner') }}</p>
                        </div>

                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('new_banner') }}</label>
                        <x-file-upload :label="''" name="banner" accept="image/*" maxSize="5MB" />
                    </div>
                </div>

                <!-- Settings -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                    <x-input 
                        :label="admin_trans('order')" 
                        name="order" 
                        type="number" 
                        x-bind:value="occasionData?.order || 0"
                        @input="occasionData.order = $event.target.value"
                    />

                    <x-select :label="admin_trans('status')" name="status" :required="true">
                        <option value="draft" x-bind:selected="occasionData?.status === 'draft'">{{ admin_trans('draft') }}</option>
                        <option value="active" x-bind:selected="occasionData?.status === 'active'">{{ admin_trans('active') }}</option>
                    </x-select>
                </div>

                <div class="flex items-center gap-3">
                    <input 
                        type="checkbox" 
                        name="is_featured" 
                        value="1"
                        class="checkbox"
                        x-bind:checked="occasionData?.is_featured"
                        @change="occasionData.is_featured = $event.target.checked"
                    >
                    <label class="text-sm font-medium text-primary">
                        {{ admin_trans('featured_occasion') }}
                    </label>
                </div>

                <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                    <x-button variant="outline" type="button" @click="$dispatch('close-modal', 'edit-occasion-modal')">{{ admin_trans('cancel') }}</x-button>
                    <x-button variant="primary" type="submit" icon="check">{{ admin_trans('update') }}</x-button>
                </div>
            </form>
        </div>
    </div>
</x-modal>

<!-- Delete Confirmation -->
<x-confirm-modal name="delete-occasion-modal" :title="admin_trans('confirm_delete')" :message="admin_trans('delete_occasion_message')" :confirmText="admin_trans('yes_delete')" :cancelText="admin_trans('cancel')" confirmVariant="danger" icon="trash-2" iconColor="danger" />

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script>
function occasionsPage() {
    return {
        selectedItems: [],

        init() {
            try {
                // Check if the function exists before calling it
                if (typeof this.loadCurrentFilter === 'function') {
                    this.loadCurrentFilter();
                }
                
                // Listen to country filter changes
                this.listenToCountryFilter();
                
                // Initialize sortable on page load
                this.$nextTick(() => {
                    this.initSortable();
                });
            } catch (error) {
                console.log('Init error (safely handled):', error);
                // Initialize sortable anyway
                this.$nextTick(() => {
                    this.initSortable();
                });
            }
        },

        // Add the missing function to prevent errors
        loadCurrentFilter() {
            // This function exists to prevent errors
            return Promise.resolve();
        },

        listenToCountryFilter() {
            // Listen to country filter changes
            window.addEventListener('country-filter-changed', () => {
                if (window.toast) {
                    toast.info('{{ admin_trans("refreshing_occasions") }}...');
                }
                setTimeout(() => window.location.reload(), 500);
            });
        },

        initSortable() {
            const tableBody = document.querySelector('.occasion-row')?.parentElement;
            if (tableBody) {
                new Sortable(tableBody, {
                    handle: '.drag-handle',
                    animation: 150,
                    onEnd: async (evt) => {
                        const rows = Array.from(tableBody.children);
                        
                        // Create orders array in the format the controller expects
                        // orders[0] = id_of_first_item, orders[1] = id_of_second_item, etc.
                        const orders = rows.map(row => parseInt(row.dataset.id));

                        // Update the order numbers in the UI immediately
                        rows.forEach((row, index) => {
                            const orderSpan = row.querySelector('.text-sm.font-medium.text-tertiary');
                            if (orderSpan) {
                                orderSpan.textContent = index;
                            }
                        });

                        try {
                            const response = await fetch('/admin/occasions/update-order', {
                                method: 'POST',
                                headers: {
                                    'Content-Type': 'application/json',
                                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                                },
                                body: JSON.stringify({ orders })
                            });

                            const result = await response.json();
                            
                            if (result.success) {
                                if (window.toast) {
                                    toast.success('{{ admin_trans("order_updated_successfully") }}');
                                }
                            } else {
                                console.error('Order update failed:', result);
                                if (window.toast) {
                                    toast.error('{{ admin_trans("error_updating_order") }}');
                                }
                                // Reload to get correct order from database
                                window.location.reload();
                            }
                        } catch (error) {
                            console.error('Order update error:', error);
                            if (window.toast) {
                                toast.error('{{ admin_trans("error_updating_order") }}');
                            }
                            window.location.reload();
                        }
                    }
                });
            }
        },

        toggleItem(id) {
            const index = this.selectedItems.indexOf(id);
            if (index > -1) {
                this.selectedItems.splice(index, 1);
            } else {
                this.selectedItems.push(id);
            }
        },

        clearSelection() {
            this.selectedItems = [];
            document.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
        },

        async performBulkAction(action) {
            if (this.selectedItems.length === 0) {
                if (window.toast) {
                    toast.warning('{{ admin_trans("please_select_items") }}');
                }
                return;
            }

            if (action === 'delete' && !confirm('{{ admin_trans("confirm_bulk_delete") }}')) return;

            try {
                const formData = new FormData();
                formData.append('_token', document.querySelector('meta[name="csrf-token"]').content);
                formData.append('action', action);
                this.selectedItems.forEach(id => formData.append('selected[]', id));

                const response = await fetch('/admin/occasions/bulk-action', {
                    method: 'POST',
                    body: formData
                });

                const result = await response.json();
                if (result.success) {
                    if (window.toast) {
                        toast.success(result.message);
                    }
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    if (window.toast) {
                        toast.error(result.message);
                    }
                }
            } catch (error) {
                console.error('Bulk action error:', error);
                if (window.toast) {
                    toast.error('{{ admin_trans("error_bulk_action") }}');
                }
            }
        }
    }
}

// FIXED: Enhanced Edit Occasion Modal with proper TinyMCE data loading
function editOccasionModal() {
    return {
        loading: false,
        occasionData: null,

        async loadOccasion(id) {
            this.loading = true;
            try {
                const response = await fetch(`/admin/occasions/${id}/edit`);
                const result = await response.json();
                
                if (result.success) {
                    this.occasionData = result.data;
                    
                    console.log('📊 Occasion data loaded:', this.occasionData);
                    
                    // Wait for modal to be fully rendered
                    await this.$nextTick();
                    
                    // Update the form fields after a short delay
                    setTimeout(() => {
                        this.updateAllFieldsWithData();
                    }, 500);
                }
            } catch (error) {
                console.error('Error loading occasion:', error);
                if (window.toast) {
                    toast.error('{{ admin_trans("error_loading_occasion") }}');
                }
            } finally {
                this.loading = false;
            }
        },

        getTranslationValue(localeId, field) {
            const translation = this.occasionData?.translations?.find(t => t.locale_id == localeId);
            return translation ? translation[field] || '' : '';
        },

        setTranslationValue(localeId, field, value) {
            if (!this.occasionData || !this.occasionData.translations) return;
            let translation = this.occasionData.translations.find(t => t.locale_id == localeId);
            if (!translation) {
                translation = { locale_id: localeId };
                this.occasionData.translations.push(translation);
            }
            translation[field] = value;
        },

        // FIXED: Enhanced data update method
        updateAllFieldsWithData() {
            if (!this.occasionData || !this.occasionData.translations) {
                console.warn('⚠️ No occasion data or translations available');
                return;
            }
            
            console.log('🔄 Updating all fields with occasion data');
            
            // Update TinyMCE editors for each locale
            this.occasionData.translations.forEach((translation) => {
                const localeId = translation.locale_id;
                const editorName = `translations[${localeId}][description]`;
                
                console.log(`🔍 Updating editor for locale ${localeId}:`, translation.description);
                
                // Find all textareas with this name
                setTimeout(() => {
                    const textareas = document.querySelectorAll(`textarea[name="${editorName}"]`);
                    
                    textareas.forEach(textarea => {
                        if (translation.description) {
                            // Set the textarea value directly
                            textarea.value = translation.description;
                            textarea.dataset.originalContent = translation.description;
                            
                            console.log(`✅ Set content for ${textarea.id}:`, translation.description.substring(0, 50) + '...');
                            
                            // Update TinyMCE if available
                            if (window.SecureTinyMCE) {
                                try {
                                    const editorInstance = window.SecureTinyMCE.getInstance(textarea.id);
                                    if (editorInstance && !editorInstance.destroyed) {
                                        editorInstance.setContent(translation.description, {format: 'raw'});
                                        console.log(`🎯 TinyMCE updated for ${textarea.id}`);
                                    } else {
                                        // Editor not ready yet, try alternative method
                                        setTimeout(() => {
                                            const retryEditor = window.SecureTinyMCE.getInstance(textarea.id);
                                            if (retryEditor && !retryEditor.destroyed) {
                                                retryEditor.setContent(translation.description, {format: 'raw'});
                                                console.log(`🔄 TinyMCE updated on retry for ${textarea.id}`);
                                            }
                                        }, 500);
                                    }
                                } catch (error) {
                                    console.warn(`⚠️ Error updating TinyMCE for ${textarea.id}:`, error);
                                }
                            }
                            
                            // Alternative: Direct TinyMCE access
                            if (typeof tinymce !== 'undefined') {
                                try {
                                    const editor = tinymce.get(textarea.id);
                                    if (editor) {
                                        editor.setContent(translation.description, {format: 'raw'});
                                        console.log(`🎯 Direct TinyMCE update for ${textarea.id}`);
                                    }
                                } catch (error) {
                                    console.warn(`⚠️ Direct TinyMCE error for ${textarea.id}:`, error);
                                }
                            }
                        }
                    });
                }, 300);
                
                // Also try with a longer delay for stubborn editors
                setTimeout(() => {
                    const textareas = document.querySelectorAll(`textarea[name="${editorName}"]`);
                    textareas.forEach(textarea => {
                        if (translation.description && textarea.value !== translation.description) {
                            textarea.value = translation.description;
                            
                            // Force TinyMCE update
                            if (typeof tinymce !== 'undefined') {
                                const editor = tinymce.get(textarea.id);
                                if (editor && editor.getContent() !== translation.description) {
                                    editor.setContent(translation.description, {format: 'raw'});
                                    console.log(`🔄 Force updated TinyMCE for ${textarea.id}`);
                                }
                            }
                        }
                    });
                }, 1000);
            });
        },

        async removeCurrentMedia(type) {
            if (!this.occasionData || !this.occasionData.id) return;
            
            try {
                const response = await fetch(`/admin/occasions/${this.occasionData.id}/media/${type}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });

                if (response.ok) {
                    this.occasionData[`${type}_url`] = null;
                    if (window.toast) {
                        toast.success('{{ admin_trans("media_removed_successfully") }}');
                    }
                } else {
                    if (window.toast) {
                        toast.error('{{ admin_trans("error_removing_media") }}');
                    }
                }
            } catch (error) {
                console.error('Error removing media:', error);
                if (window.toast) {
                    toast.error('{{ admin_trans("error_removing_media") }}');
                }
            }
        }
    }
}

// FIXED: Enhanced edit occasion function
function editOccasion(id) {
    console.log('🔧 Opening edit modal for occasion:', id);
    
    // Open the modal
    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'edit-occasion-modal' }));
    
    // Load occasion data with proper timing
    setTimeout(() => {
        const modal = document.querySelector('[x-data*="editOccasionModal"]');
        if (modal && modal._x_dataStack && modal._x_dataStack[0]) {
            console.log('📂 Loading occasion data...');
            modal._x_dataStack[0].loadOccasion(id);
        } else {
            console.warn('⚠️ Modal component not found, retrying...');
            // Retry after a longer delay
            setTimeout(() => {
                const retryModal = document.querySelector('[x-data*="editOccasionModal"]');
                if (retryModal && retryModal._x_dataStack && retryModal._x_dataStack[0]) {
                    retryModal._x_dataStack[0].loadOccasion(id);
                }
            }, 500);
        }
    }, 100);
}

function confirmDelete(id, name) {
    window.addEventListener('confirm-action', function handler(e) {
        if (e.detail === 'delete-occasion-modal') {
            const form = document.createElement('form');
            form.method = 'POST';
            form.action = `/admin/occasions/${id}`;
            form.innerHTML = '@csrf @method("DELETE")';
            document.body.appendChild(form);
            form.submit();
            window.removeEventListener('confirm-action', handler);
        }
    });
    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'delete-occasion-modal' }));
}

// Initialize functions when DOM loads
document.addEventListener('DOMContentLoaded', function() {
    // Initialize Lucide icons
    if (typeof lucide !== 'undefined') {
        lucide.createIcons();
    }
});
</script>
@endpush