@extends('admin.layouts.app')

@section('title', admin_trans('create_notification'))
@section('page-title', admin_trans('create_notification'))

@section('content')

<div class="space-y-6">
    
    <!-- Back Button -->
    <div class="flex items-center gap-3">
        <x-button 
            variant="outline" 
            icon="arrow-left"
            href="{{ route('admin.notifications.index') }}"
        >
            {{ admin_trans('back_to_notifications') }}
        </x-button>
    </div>

    <!-- Create Notification Form -->
    <form method="POST" action="{{ route('admin.notifications.store') }}" 
          class="space-y-6" x-data="createNotificationForm()">
        @csrf

        <!-- Basic Information -->
        <div class="card">
            <h3 class="text-lg font-bold text-primary mb-6">{{ admin_trans('basic_information') }}</h3>
            
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                <x-select 
                    :label="admin_trans('notification_type')"
                    name="type"
                    :required="true"
                    x-model="formData.type"
                >
                    <option value="">{{ admin_trans('select_type') }}</option>
                    <option value="system">{{ admin_trans('system') }}</option>
                    <option value="promotional">{{ admin_trans('promotional') }}</option>
                    <option value="offer">{{ admin_trans('offer') }}</option>
                    <option value="store">{{ admin_trans('store') }}</option>
                    <option value="brand">{{ admin_trans('brand') }}</option>
                    <option value="user">{{ admin_trans('user') }}</option>
                </x-select>

                <x-select 
                    :label="admin_trans('priority')"
                    name="priority"
                    :required="true"
                    x-model="formData.priority"
                >
                    <option value="">{{ admin_trans('select_priority') }}</option>
                    <option value="1">{{ admin_trans('low') }} (1)</option>
                    <option value="3">{{ admin_trans('normal') }} (3)</option>
                    <option value="5">{{ admin_trans('medium') }} (5)</option>
                    <option value="8">{{ admin_trans('high') }} (8)</option>
                    <option value="10">{{ admin_trans('urgent') }} (10)</option>
                </x-select>
            </div>

            <x-input 
                :label="admin_trans('notification_title')"
                name="title"
                :required="true"
                :placeholder="admin_trans('enter_notification_title')"
                x-model="formData.title"
            />

            <x-textarea 
                :label="admin_trans('notification_message')"
                name="message"
                :required="true"
                rows="4"
                :placeholder="admin_trans('enter_notification_message')"
                x-model="formData.message"
            />
        </div>

        <!-- Channels Selection -->
        <div class="card">
            <h3 class="text-lg font-bold text-primary mb-6">{{ admin_trans('delivery_channels') }}</h3>
            
            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <x-toggle 
                    name="channels[]"
                    value="database"
                    :label="admin_trans('database_notification')"
                    :description="admin_trans('store_in_database')"
                    checked
                />
                
                <x-toggle 
                    name="channels[]"
                    value="push"
                    :label="admin_trans('push_notification')"
                    :description="admin_trans('send_push_to_devices')"
                />
                
                <x-toggle 
                    name="channels[]"
                    value="email"
                    :label="admin_trans('email_notification')"
                    :description="admin_trans('send_via_email')"
                />
            </div>
        </div>

        <!-- Targeting -->
        <div class="card">
            <h3 class="text-lg font-bold text-primary mb-6">{{ admin_trans('target_audience') }}</h3>
            
            <div class="space-y-4">
                <!-- Target Type Selector -->
                <div class="grid grid-cols-2 md:grid-cols-4 gap-3">
                    <button type="button" @click="addTarget('all')"
                            class="p-4 rounded-xl border-2 transition-all hover:border-primary hover:bg-primary-light text-center">
                        <i data-lucide="users" class="w-6 h-6 mx-auto mb-2 text-primary"></i>
                        <div class="text-sm font-medium text-primary">{{ admin_trans('all_users') }}</div>
                    </button>
                    
                    <button type="button" @click="addTarget('country')"
                            class="p-4 rounded-xl border-2 transition-all hover:border-primary hover:bg-primary-light text-center">
                        <i data-lucide="globe" class="w-6 h-6 mx-auto mb-2 text-primary"></i>
                        <div class="text-sm font-medium text-primary">{{ admin_trans('by_country') }}</div>
                    </button>
                    
                    <button type="button" @click="addTarget('language')"
                            class="p-4 rounded-xl border-2 transition-all hover:border-primary hover:bg-primary-light text-center">
                        <i data-lucide="languages" class="w-6 h-6 mx-auto mb-2 text-primary"></i>
                        <div class="text-sm font-medium text-primary">{{ admin_trans('by_language') }}</div>
                    </button>
                    
                    <button type="button" @click="addTarget('role')"
                            class="p-4 rounded-xl border-2 transition-all hover:border-primary hover:bg-primary-light text-center">
                        <i data-lucide="user-check" class="w-6 h-6 mx-auto mb-2 text-primary"></i>
                        <div class="text-sm font-medium text-primary">{{ admin_trans('by_role') }}</div>
                    </button>
                </div>

                <!-- Advanced Targeting -->
                <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                    <button type="button" @click="addTarget('store_followers')"
                            class="p-3 rounded-lg border transition-all hover:border-primary hover:bg-primary-light flex items-center gap-3">
                        <i data-lucide="store" class="w-5 h-5 text-primary"></i>
                        <div class="text-left">
                            <div class="text-sm font-medium text-primary">{{ admin_trans('store_followers') }}</div>
                            <div class="text-xs text-tertiary">{{ admin_trans('users_following_stores') }}</div>
                        </div>
                    </button>
                    
                    <button type="button" @click="addTarget('brand_followers')"
                            class="p-3 rounded-lg border transition-all hover:border-primary hover:bg-primary-light flex items-center gap-3">
                        <i data-lucide="sparkles" class="w-5 h-5 text-primary"></i>
                        <div class="text-left">
                            <div class="text-sm font-medium text-primary">{{ admin_trans('brand_followers') }}</div>
                            <div class="text-xs text-tertiary">{{ admin_trans('users_following_brands') }}</div>
                        </div>
                    </button>
                </div>

                <!-- Selected Targets -->
                <div x-show="targets.length > 0" class="space-y-3">
                    <h4 class="font-semibold text-primary">{{ admin_trans('selected_targets') }}:</h4>

                    <!-- Badges Container -->
                    <div class="flex flex-wrap gap-2">
                        <template x-for="(target, index) in targets" :key="index">
                            <div class="inline-flex items-center gap-2 px-4 py-2.5 rounded-xl border-2 text-sm font-medium transition-all"
                                 style="background: var(--color-primary); border-color: var(--color-primary); color: white;">
                                <i :data-lucide="getTargetIcon(target.type)" class="w-4 h-4" style="color: white;"></i>
                                <span x-text="getTargetLabel(target)" style="color: white;"></span>
                                <button type="button" @click="removeTarget(index)"
                                        class="ml-2 p-1 rounded-full transition-all hover:bg-white/20"
                                        style="color: white;">
                                    <i data-lucide="x" class="w-4 h-4" style="color: white;"></i>
                                </button>

                                <!-- Hidden inputs for form submission -->
                                <input type="hidden" :name="`targets[${index}][type]`" :value="target.type">
                                <input type="hidden" :name="`targets[${index}][id]`" :value="target.id || ''">
                            </div>
                        </template>
                    </div>
                </div>
            </div>
        </div>

        <!-- Additional Data (Optional) -->
        <div class="card">
            <h3 class="text-lg font-bold text-primary mb-6">{{ admin_trans('additional_data') }}</h3>
            
            <div x-data="{ showAdvanced: false }">
                <button type="button" @click="showAdvanced = !showAdvanced"
                        class="flex items-center gap-2 text-sm text-primary hover:text-primary-dark transition-colors mb-4">
                    <i data-lucide="settings" class="w-4 h-4"></i>
                    {{ admin_trans('advanced_options') }}
                    <i data-lucide="chevron-down" class="w-4 h-4 transition-transform" :class="showAdvanced ? 'rotate-180' : ''"></i>
                </button>

                <div x-show="showAdvanced" x-transition class="space-y-4">
                    <x-input 
                        :label="admin_trans('action_url')"
                        name="data[action_url]"
                        type="url"
                        :placeholder="admin_trans('url_to_open_on_tap')"
                    />
                    
                    <x-input 
                        :label="admin_trans('image_url')"
                        name="data[image_url]"
                        type="url"
                        :placeholder="admin_trans('notification_image_url')"
                    />
                    
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                        <x-input 
                            :label="admin_trans('custom_sound')"
                            name="data[sound]"
                            :placeholder="admin_trans('notification_sound')"
                        />
                        
                        <x-input 
                            :label="admin_trans('badge_count')"
                            name="data[badge]"
                            type="number"
                            :placeholder="admin_trans('app_badge_number')"
                        />
                    </div>
                </div>
            </div>
        </div>

        <!-- Form Actions -->
        <div class="card">
            <div class="flex gap-3 justify-end">
                <x-button 
                    variant="outline" 
                    type="button"
                    href="{{ route('admin.notifications.index') }}"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                
                <x-button 
                    variant="secondary" 
                    type="button"
                    @click="saveAsDraft()"
                >
                    {{ admin_trans('save_draft') }}
                </x-button>
                
                <x-button 
                    variant="primary" 
                    type="submit"
                    icon="send"
                >
                    {{ admin_trans('send_notification') }}
                </x-button>
            </div>
        </div>
    </form>
</div>

<!-- Target Selection Modals -->

<!-- Country Selection Modal -->
<x-modal name="country-target-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('select_countries') }}</h3>
            <button @click="$dispatch('close-modal', 'country-target-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <div class="space-y-3 max-h-96 overflow-y-auto">
            @foreach($countries as $country)
                <label class="flex items-center gap-3 p-3 rounded-lg hover:bg-primary-light cursor-pointer">
                    <input type="checkbox" value="{{ $country->id }}" class="checkbox country-checkbox">
                    <span class="text-lg">{{ $country->flag_emoji ?? '🏳️' }}</span>
                    <span class="flex-1 text-sm font-medium text-primary">{{ $country->name }}</span>
                    <span class="text-xs text-tertiary">{{ $country->code }}</span>
                </label>
            @endforeach
        </div>

        <div class="flex gap-3 justify-end pt-4 border-t mt-6" style="border-color: var(--color-border);">
            <x-button variant="outline" @click="$dispatch('close-modal', 'country-target-modal')">
                {{ admin_trans('cancel') }}
            </x-button>
            <x-button variant="primary" @click="addCountryTargets()">
                {{ admin_trans('add_selected') }}
            </x-button>
        </div>
    </div>
</x-modal>

<!-- Language Selection Modal -->
<x-modal name="language-target-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('select_languages') }}</h3>
            <button @click="$dispatch('close-modal', 'language-target-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <div class="space-y-3">
            @foreach($languages as $language)
                <label class="flex items-center gap-3 p-3 rounded-lg hover:bg-primary-light cursor-pointer">
                    <input type="checkbox" value="{{ $language->id }}" class="checkbox language-checkbox">
                    <div class="w-8 h-8 rounded-lg bg-primary text-white text-xs font-bold flex items-center justify-center">
                        {{ strtoupper(substr($language->code, 0, 2)) }}
                    </div>
                    <div class="flex-1">
                        <div class="text-sm font-medium text-primary">{{ $language->name }}</div>
                        <div class="text-xs text-tertiary">{{ $language->native_name }}</div>
                    </div>
                </label>
            @endforeach
        </div>

        <div class="flex gap-3 justify-end pt-4 border-t mt-6" style="border-color: var(--color-border);">
            <x-button variant="outline" @click="$dispatch('close-modal', 'language-target-modal')">
                {{ admin_trans('cancel') }}
            </x-button>
            <x-button variant="primary" @click="addLanguageTargets()">
                {{ admin_trans('add_selected') }}
            </x-button>
        </div>
    </div>
</x-modal>

<!-- Store Selection Modal -->
<x-modal name="store-target-modal" maxWidth="lg">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('select_stores') }}</h3>
            <button @click="$dispatch('close-modal', 'store-target-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <div class="space-y-3 max-h-96 overflow-y-auto">
            @foreach($stores as $store)
                <label class="flex items-center gap-3 p-3 rounded-lg hover:bg-primary-light cursor-pointer">
                    <input type="checkbox" value="{{ $store->id }}" class="checkbox store-checkbox">
                    <div class="w-10 h-10 rounded-lg bg-primary text-white text-sm font-bold flex items-center justify-center">
                        {{ substr($store->store->getName(), 0, 2) }}
                    </div>
                    <div class="flex-1">
                        <div class="text-sm font-medium text-primary">{{ $store->store->getName() }}</div>
                        <div class="text-xs text-tertiary">{{ $store->country->name }}</div>
                    </div>
                </label>
            @endforeach
        </div>

        <div class="flex gap-3 justify-end pt-4 border-t mt-6" style="border-color: var(--color-border);">
            <x-button variant="outline" @click="$dispatch('close-modal', 'store-target-modal')">
                {{ admin_trans('cancel') }}
            </x-button>
            <x-button variant="primary" @click="addStoreTargets()">
                {{ admin_trans('add_selected') }}
            </x-button>
        </div>
    </div>
</x-modal>

<!-- Brand Selection Modal -->
<x-modal name="brand-target-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('select_brands') }}</h3>
            <button @click="$dispatch('close-modal', 'brand-target-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <div class="space-y-3 max-h-96 overflow-y-auto">
            @foreach($brands as $brand)
                <label class="flex items-center gap-3 p-3 rounded-lg hover:bg-primary-light cursor-pointer">
                    <input type="checkbox" value="{{ $brand->id }}" class="checkbox brand-checkbox">
                    @if($brand->getFirstMediaUrl('logo'))
                        <img src="{{ $brand->getFirstMediaUrl('logo') }}" class="w-10 h-10 rounded-lg object-cover">
                    @else
                        <div class="w-10 h-10 rounded-lg bg-secondary text-white text-sm font-bold flex items-center justify-center">
                            {{ substr($brand->getName(), 0, 2) }}
                        </div>
                    @endif
                    <span class="flex-1 text-sm font-medium text-primary">{{ $brand->getName() }}</span>
                </label>
            @endforeach
        </div>

        <div class="flex gap-3 justify-end pt-4 border-t mt-6" style="border-color: var(--color-border);">
            <x-button variant="outline" @click="$dispatch('close-modal', 'brand-target-modal')">
                {{ admin_trans('cancel') }}
            </x-button>
            <x-button variant="primary" @click="addBrandTargets()">
                {{ admin_trans('add_selected') }}
            </x-button>
        </div>
    </div>
</x-modal>

@endsection

@push('scripts')
<script>
// Alpine.js Create Notification Form Component
function createNotificationForm() {
    return {
        formData: {
            type: '',
            priority: '',
            title: '',
            message: ''
        },
        targets: [],

        addTarget(type) {
            if (type === 'all') {
                this.targets = [{ type: 'all', label: '{{ admin_trans("all_users") }}' }];
                // Re-initialize icons
                this.$nextTick(() => {
                    if (typeof lucide !== 'undefined') {
                        lucide.createIcons();
                    }
                });
                return;
            }

            if (type === 'role') {
                // Add admin and user roles
                const adminExists = this.targets.find(t => t.type === 'role' && t.id === 'admin');
                const userExists = this.targets.find(t => t.type === 'role' && t.id === 'user');

                if (!adminExists) {
                    this.targets.push({ type: 'role', id: 'admin', label: '{{ admin_trans("admin_users") }}' });
                }
                if (!userExists) {
                    this.targets.push({ type: 'role', id: 'user', label: '{{ admin_trans("regular_users") }}' });
                }

                // Re-initialize icons
                this.$nextTick(() => {
                    if (typeof lucide !== 'undefined') {
                        lucide.createIcons();
                    }
                });
                return;
            }

            // Open modal for selection-based targets
            switch (type) {
                case 'country':
                    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'country-target-modal' }));
                    break;
                case 'language':
                    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'language-target-modal' }));
                    break;
                case 'store_followers':
                    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'store-target-modal' }));
                    break;
                case 'brand_followers':
                    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'brand-target-modal' }));
                    break;
            }
        },

        removeTarget(index) {
            this.targets.splice(index, 1);
            // Re-initialize icons after removal
            this.$nextTick(() => {
                if (typeof lucide !== 'undefined') {
                    lucide.createIcons();
                }
            });
        },

        getTargetIcon(type) {
            const icons = {
                'all': 'users',
                'country': 'globe',
                'language': 'languages',
                'role': 'user-check',
                'store_followers': 'store',
                'brand_followers': 'sparkles'
            };
            return icons[type] || 'target';
        },

        getTargetLabel(target) {
            if (target.label) return target.label;
            return target.type + (target.id ? ` (${target.id})` : '');
        },

        saveAsDraft() {
            // Add hidden input for draft status
            const form = this.$el;
            const draftInput = document.createElement('input');
            draftInput.type = 'hidden';
            draftInput.name = 'status';
            draftInput.value = 'draft';
            form.appendChild(draftInput);
            form.submit();
        }
    }
}

// Add selected countries
function addCountryTargets() {
    const selectedCountries = Array.from(document.querySelectorAll('.country-checkbox:checked'));
    const component = Alpine.$data(document.querySelector('[x-data*="createNotificationForm"]'));

    selectedCountries.forEach(checkbox => {
        const countryId = checkbox.value;
        const countryName = checkbox.closest('label').querySelector('.text-primary').textContent;

        // Check if already exists
        const exists = component.targets.find(t => t.type === 'country' && t.id === countryId);
        if (!exists) {
            component.targets.push({
                type: 'country',
                id: countryId,
                label: `{{ admin_trans("country") }}: ${countryName}`
            });
        }
    });

    // Clear selections and close modal
    document.querySelectorAll('.country-checkbox').forEach(cb => cb.checked = false);
    window.dispatchEvent(new CustomEvent('close-modal', { detail: 'country-target-modal' }));

    // Re-initialize icons
    setTimeout(() => {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }, 100);
}

// Add selected languages
function addLanguageTargets() {
    const selectedLanguages = Array.from(document.querySelectorAll('.language-checkbox:checked'));
    const component = Alpine.$data(document.querySelector('[x-data*="createNotificationForm"]'));

    selectedLanguages.forEach(checkbox => {
        const languageId = checkbox.value;
        const languageName = checkbox.closest('label').querySelector('.text-primary').textContent;

        const exists = component.targets.find(t => t.type === 'language' && t.id === languageId);
        if (!exists) {
            component.targets.push({
                type: 'language',
                id: languageId,
                label: `{{ admin_trans("language") }}: ${languageName}`
            });
        }
    });

    document.querySelectorAll('.language-checkbox').forEach(cb => cb.checked = false);
    window.dispatchEvent(new CustomEvent('close-modal', { detail: 'language-target-modal' }));

    // Re-initialize icons
    setTimeout(() => {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }, 100);
}

// Add selected stores
function addStoreTargets() {
    const selectedStores = Array.from(document.querySelectorAll('.store-checkbox:checked'));
    const component = Alpine.$data(document.querySelector('[x-data*="createNotificationForm"]'));

    selectedStores.forEach(checkbox => {
        const storeId = checkbox.value;
        const storeName = checkbox.closest('label').querySelector('.text-primary').textContent;

        const exists = component.targets.find(t => t.type === 'store_followers' && t.id === storeId);
        if (!exists) {
            component.targets.push({
                type: 'store_followers',
                id: storeId,
                label: `{{ admin_trans("store_followers") }}: ${storeName}`
            });
        }
    });

    document.querySelectorAll('.store-checkbox').forEach(cb => cb.checked = false);
    window.dispatchEvent(new CustomEvent('close-modal', { detail: 'store-target-modal' }));

    // Re-initialize icons
    setTimeout(() => {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }, 100);
}

// Add selected brands
function addBrandTargets() {
    const selectedBrands = Array.from(document.querySelectorAll('.brand-checkbox:checked'));
    const component = Alpine.$data(document.querySelector('[x-data*="createNotificationForm"]'));

    selectedBrands.forEach(checkbox => {
        const brandId = checkbox.value;
        const brandName = checkbox.closest('label').querySelector('.text-primary').textContent;

        const exists = component.targets.find(t => t.type === 'brand_followers' && t.id === brandId);
        if (!exists) {
            component.targets.push({
                type: 'brand_followers',
                id: brandId,
                label: `{{ admin_trans("brand_followers") }}: ${brandName}`
            });
        }
    });

    document.querySelectorAll('.brand-checkbox').forEach(cb => cb.checked = false);
    window.dispatchEvent(new CustomEvent('close-modal', { detail: 'brand-target-modal' }));

    // Re-initialize icons
    setTimeout(() => {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }, 100);
}

// Form validation
document.addEventListener('DOMContentLoaded', function() {
    const form = document.querySelector('form');
    form.addEventListener('submit', function(e) {
        const component = Alpine.$data(document.querySelector('[x-data*="createNotificationForm"]'));
        
        if (component.targets.length === 0) {
            e.preventDefault();
            toast.error('{{ admin_trans("please_select_target_audience") }}');
            return false;
        }

        // Check if at least one channel is selected
        const channels = document.querySelectorAll('input[name="channels[]"]:checked');
        if (channels.length === 0) {
            e.preventDefault();
            toast.error('{{ admin_trans("please_select_delivery_channel") }}');
            return false;
        }
    });
});
</script>
@endpush