@extends('admin.layouts.app')

@section('title', admin_trans('locales'))
@section('page-title', admin_trans('locales'))

@section('content')

<div class="space-y-6">
    
    <!-- Search Bar + Add Button -->
    <div class="card">
        <div class="flex items-center gap-3">
            <form method="GET" action="{{ route('admin.locales.index') }}" id="search-form" class="flex-1">
                <x-search-input 
                    name="search"
                    :value="request('search')"
                    :placeholder="admin_trans('search_locales')"
                    oninput="clearTimeout(window.searchTimeout); window.searchTimeout = setTimeout(() => this.form.submit(), 500)"
                    class="w-full"
                />
            </form>
            
            <x-button
                variant="secondary"
                icon="zap"
                onclick="confirmAutoCreate()"
            >
                {{ admin_trans('auto_create') }}
            </x-button>

            <x-button
                variant="primary"
                icon="plus"
                @click="$dispatch('open-modal', 'create-locale-modal')"
            >
                {{ admin_trans('add_locale') }}
            </x-button>
        </div>
    </div>

    <!-- Quick Info Card -->
    <div class="card">
        <div class="flex items-center justify-between">
            <div class="flex items-center gap-4">
                <div class="w-12 h-12 rounded-xl flex items-center justify-center gradient-primary">
                    <i data-lucide="globe-2" class="w-6 h-6 text-white"></i>
                </div>
                <div>
                    <h3 class="text-lg font-bold text-primary">{{ admin_trans('locales_management') }}</h3>
                    <p class="text-sm text-secondary">{{ admin_trans('combine_languages_with_countries') }}</p>
                </div>
            </div>
            <div class="flex items-center gap-4 text-sm">
                <div class="text-center">
                    <p class="font-bold text-primary">{{ $locales->total() }}</p>
                    <p class="text-tertiary">{{ admin_trans('total_locales') }}</p>
                </div>
                <div class="text-center">
                    <p class="font-bold text-success-500">{{ $locales->where('status', 'active')->count() }}</p>
                    <p class="text-tertiary">{{ admin_trans('active') }}</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Locales Table Card -->
    <div class="card">
        
        @if($locales->isEmpty())
            @if(request('search'))
                <!-- No Search Results -->
                <x-empty-state 
                    icon="search-x"
                    :title="admin_trans('no_results_found')"
                    :description="admin_trans('try_different_search')"
                >
                    <x-button 
                        variant="outline"
                        icon="x"
                        onclick="window.location.href='{{ route('admin.locales.index') }}'"
                    >
                        {{ admin_trans('clear') }}
                    </x-button>
                </x-empty-state>
            @else
                <!-- No Locales Yet -->
                <x-empty-state 
                    icon="globe-2"
                    :title="admin_trans('no_locales_yet')"
                    :description="admin_trans('start_by_adding_first_locale')"
                >
                    <x-button 
                        variant="primary"
                        icon="plus"
                        @click="$dispatch('open-modal', 'create-locale-modal')"
                    >
                        {{ admin_trans('add_locale') }}
                    </x-button>
                </x-empty-state>
            @endif
        @else

<!-- Locales Table -->
<x-enhanced-table 
    :headers="[
        'order' => [
            'text' => admin_trans('order'),
            'align' => 'center',
            'width' => '80px'
        ],
        'code' => [
            'text' => admin_trans('code'),
            'align' => 'center',
            'width' => '100px'
        ],
        'name' => [
            'text' => admin_trans('name'),
            'align' => 'auto',
            'width' => '250px'
        ],
        'language' => [
            'text' => admin_trans('language'),
            'align' => 'center',
            'width' => '150px'
        ],
        'country' => [
            'text' => admin_trans('country'),
            'align' => 'center',
            'width' => '150px'
        ],
        'status' => [
            'text' => admin_trans('status'),
            'align' => 'center',
            'width' => '100px'
        ],
        'default' => [
            'text' => admin_trans('default'),
            'align' => 'center',
            'width' => '100px'
        ],
        'actions' => [
            'text' => admin_trans('actions'),
            'align' => 'center',
            'width' => '120px'
        ]
    ]"
    :responsive="true"
>
    @foreach($locales as $locale)
    <tr class="locale-row hover:bg-primary-light transition-colors" data-id="{{ $locale->id }}">
        
        <!-- Order (Drag Handle) -->
        <x-table-cell type="number">
            <div class="flex items-center justify-center gap-2">
                <button class="drag-handle cursor-move p-1 rounded hover:bg-primary-light">
                    <i data-lucide="grip-vertical" class="w-3 h-3 text-tertiary"></i>
                </button>
                <span class="text-sm font-medium text-tertiary">{{ $locale->order ?? 0 }}</span>
            </div>
        </x-table-cell>
        
        <!-- Code -->
        <x-table-cell type="center">
            <x-badge variant="secondary" class="font-mono">{{ $locale->code }}</x-badge>
        </x-table-cell>
        
        <!-- Name -->
        <x-table-cell type="text">
            <span class="text-sm font-medium text-primary">{{ $locale->name ?: $locale->code }}</span>
        </x-table-cell>
        
        <!-- Language -->
        <x-table-cell type="center">
            <div class="flex items-center justify-center gap-2">
                <div class="w-6 h-6 rounded flex items-center justify-center text-xs font-bold text-white"
                     style="background: var(--color-secondary-500);">
                    {{ strtoupper(substr($locale->language->code, 0, 2)) }}
                </div>
                <div class="text-left">
                    <div class="text-sm font-medium text-primary">{{ $locale->language->name }}</div>
                    <div class="text-xs text-tertiary">{{ $locale->language->native_name }}</div>
                </div>
            </div>
        </x-table-cell>
        
        <!-- Country -->
        <x-table-cell type="center">
            <div class="flex items-center justify-center gap-2">
                @if($locale->country->flag_url)
                    <img src="{{ $locale->country->flag_url }}" 
                         alt="{{ $locale->country->name }} Flag" 
                         class="w-6 h-4 object-cover rounded border">
                @else
                    <div class="w-6 h-4 bg-gray-200 rounded border flex items-center justify-center">
                        <span class="text-xs">{{ $locale->country->flag_emoji ?? '🏳️' }}</span>
                    </div>
                @endif
                <div class="text-left">
                    <div class="text-sm font-medium text-primary">{{ $locale->country->name ?? $locale->country->code }}</div>
                    <div class="text-xs text-tertiary">{{ $locale->country->code }}</div>
                </div>
            </div>
        </x-table-cell>
        
        <!-- Status with Toggle -->
        <x-table-cell type="actions">
            <form method="POST" action="{{ route('admin.locales.toggle-status', $locale) }}">
                @csrf
                @method('PATCH')
                <x-toggle 
                    name="status_toggle_{{ $locale->id }}"
                    :checked="$locale->status === 'active'"
                    size="sm"
                    variant="success"
                    onchange="this.form.submit()"
                />
            </form>
        </x-table-cell>

        <!-- Is Default -->
        <x-table-cell type="center">
            @if($locale->is_default)
                <x-badge variant="primary">
                    {{ admin_trans('default') }}
                </x-badge>
            @else
                <form method="POST" action="{{ route('admin.locales.set-default', $locale) }}">
                    @csrf
                    @method('PATCH')
                    <button type="submit" class="text-xs text-tertiary hover:text-primary transition-colors">
                        {{ admin_trans('set_default') }}
                    </button>
                </form>
            @endif
        </x-table-cell>
        
        <!-- Actions -->
        <x-table-cell type="actions">
            <div class="flex items-center justify-center gap-2">
                <!-- Edit Button -->
                <button 
                    class="p-2 rounded-lg hover:bg-blue-50 text-blue-600 transition-colors"
                    onclick="editLocale({{ json_encode($locale) }})"
                    title="{{ admin_trans('edit') }}"
                >
                    <i data-lucide="edit-3" class="w-4 h-4"></i>
                </button>
                
                @if(!$locale->is_default)
                    <!-- Delete Button -->
                    <button 
                        class="p-2 rounded-lg hover:bg-red-50 text-red-600 transition-colors"
                        onclick="confirmDelete({{ $locale->id }}, '{{ $locale->name ?: $locale->code }}')"
                        title="{{ admin_trans('delete') }}"
                    >
                        <i data-lucide="trash-2" class="w-4 h-4"></i>
                    </button>
                @endif
            </div>
        </x-table-cell>
    </tr>
    @endforeach
</x-enhanced-table>

            <!-- Footer Info & Pagination -->
            <div class="mt-6 flex flex-col sm:flex-row items-center justify-between gap-4 border-t pt-4" style="border-color: var(--color-border);">
                <div class="flex items-center gap-2 text-sm text-tertiary">
                    <i data-lucide="info" class="w-4 h-4"></i>
                    <span>{{ admin_trans('drag_rows_to_reorder') }}</span>
                </div>
                
                @if($locales->hasPages())
                    <x-pagination :paginator="$locales" />
                @endif
            </div>
        @endif

    </div>
</div>

<!-- Create Locale Modal -->
<x-modal name="create-locale-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('add_locale') }}</h3>
            <button @click="$dispatch('close-modal', 'create-locale-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <form method="POST" action="{{ route('admin.locales.store') }}" class="space-y-4">
            @csrf

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-select 
                    :label="admin_trans('language')"
                    name="language_id"
                    :required="true"
                    id="create_language_id"
                >
                    <option value="">{{ admin_trans('select_language') }}</option>
                    @foreach($languages as $language)
                        <option value="{{ $language->id }}">{{ $language->name }} ({{ $language->native_name }})</option>
                    @endforeach
                </x-select>

                <x-select 
                    :label="admin_trans('country')"
                    name="country_id"
                    :required="true"
                    id="create_country_id"
                >
                    <option value="">{{ admin_trans('select_country') }}</option>
                    @foreach($countries as $country)
                        <option value="{{ $country->id }}">{{ $country->name ?? $country->code }} ({{ $country->code }})</option>
                    @endforeach
                </x-select>
            </div>

            <x-input 
                :label="admin_trans('custom_name')"
                name="name"
                placeholder="{{ admin_trans('auto_generated_if_empty') }}"
                :hint="admin_trans('optional_custom_name')"
            />

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('order')"
                    name="order"
                    type="number"
                    value="0"
                    :hint="admin_trans('display_order')"
                />

                <x-select 
                    :label="admin_trans('status')"
                    name="status"
                    :required="true"
                >
                    <option value="draft">{{ admin_trans('draft') }}</option>
                    <option value="active" selected>{{ admin_trans('active') }}</option>
                </x-select>
            </div>

            <div class="flex items-center gap-3">
                <input 
                    type="checkbox" 
                    name="is_default" 
                    id="create_is_default"
                    value="1"
                    class="checkbox"
                >
                <label for="create_is_default" class="text-sm font-medium text-primary cursor-pointer">
                    {{ admin_trans('set_as_default') }}
                </label>
            </div>

            <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                <x-button 
                    variant="outline" 
                    type="button"
                    @click="$dispatch('close-modal', 'create-locale-modal')"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                <x-button variant="primary" type="submit" icon="check">
                    {{ admin_trans('create') }}
                </x-button>
            </div>
        </form>
    </div>
</x-modal>

<!-- Edit Locale Modal -->
<x-modal name="edit-locale-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('edit_locale') }}</h3>
            <button @click="$dispatch('close-modal', 'edit-locale-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <form id="edit-locale-form" method="POST" class="space-y-4">
            @csrf
            @method('PUT')

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-select 
                    :label="admin_trans('language')"
                    name="language_id"
                    :required="true"
                    id="edit_language_id"
                >
                    @foreach($languages as $language)
                        <option value="{{ $language->id }}">{{ $language->name }} ({{ $language->native_name }})</option>
                    @endforeach
                </x-select>

                <x-select 
                    :label="admin_trans('country')"
                    name="country_id"
                    :required="true"
                    id="edit_country_id"
                >
                    @foreach($countries as $country)
                        <option value="{{ $country->id }}">{{ $country->name ?? $country->code }} ({{ $country->code }})</option>
                    @endforeach
                </x-select>
            </div>

            <x-input 
                :label="admin_trans('custom_name')"
                name="name"
                id="edit_name"
            />

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('order')"
                    name="order"
                    type="number"
                    id="edit_order"
                />

                <x-select 
                    :label="admin_trans('status')"
                    name="status"
                    id="edit_status"
                    :required="true"
                >
                    <option value="draft">{{ admin_trans('draft') }}</option>
                    <option value="active">{{ admin_trans('active') }}</option>
                </x-select>
            </div>

            <div class="flex items-center gap-3">
                <input 
                    type="checkbox" 
                    name="is_default" 
                    id="edit_is_default"
                    value="1"
                    class="checkbox"
                >
                <label for="edit_is_default" class="text-sm font-medium text-primary cursor-pointer">
                    {{ admin_trans('set_as_default') }}
                </label>
            </div>

            <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                <x-button 
                    variant="outline" 
                    type="button"
                    @click="$dispatch('close-modal', 'edit-locale-modal')"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                <x-button variant="primary" type="submit" icon="check">
                    {{ admin_trans('update') }}
                </x-button>
            </div>
        </form>
    </div>
</x-modal>

<!-- Delete Confirmation Modal -->
<form id="delete-locale-form" method="POST">
    @csrf
    @method('DELETE')
</form>

<x-confirm-modal
    name="delete-locale-modal"
    :title="admin_trans('confirm_delete')"
    message=""
    :confirmText="admin_trans('yes_delete')"
    :cancelText="admin_trans('cancel')"
    confirmVariant="danger"
    icon="trash-2"
    iconColor="danger"
    x-data="{ message: '' }"
/>

<!-- Auto Create Confirmation Modal -->
<form id="auto-create-locales-form" method="POST" action="{{ route('admin.locales.auto-create') }}">
    @csrf
</form>

<x-confirm-modal
    name="auto-create-modal"
    :title="admin_trans('auto_create_locales')"
    :message="admin_trans('auto_create_locales_message')"
    :confirmText="admin_trans('yes_create')"
    :cancelText="admin_trans('cancel')"
    confirmVariant="primary"
    icon="zap"
    iconColor="primary"
/>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script>

// Edit Locale Function
function editLocale(locale) {
    const form = document.getElementById('edit-locale-form');
    form.action = `/admin/locales/${locale.id}`;
    
    document.getElementById('edit_language_id').value = locale.language_id || '';
    document.getElementById('edit_country_id').value = locale.country_id || '';
    document.getElementById('edit_name').value = locale.name || '';
    document.getElementById('edit_order').value = locale.order || 0;
    document.getElementById('edit_status').value = locale.status || 'draft';
    document.getElementById('edit_is_default').checked = locale.is_default || false;
    
    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'edit-locale-modal' }));
}

// Confirm Delete Function
function confirmDelete(id, name) {
    const form = document.getElementById('delete-locale-form');
    form.action = `/admin/locales/${id}`;

    // Submit form on confirm
    window.addEventListener('confirm-action', function handler(e) {
        if (e.detail === 'delete-locale-modal') {
            form.submit();
            window.removeEventListener('confirm-action', handler);
        }
    });

    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'delete-locale-modal' }));
}

// Confirm Auto Create Function
function confirmAutoCreate() {
    const form = document.getElementById('auto-create-locales-form');

    // Submit form on confirm
    window.addEventListener('confirm-action', function handler(e) {
        if (e.detail === 'auto-create-modal') {
            form.submit();
            window.removeEventListener('confirm-action', handler);
        }
    });

    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'auto-create-modal' }));
}

// Sortable.js for Drag & Drop
document.addEventListener('DOMContentLoaded', function() {
    const tbody = document.querySelector('.locale-row')?.parentElement;
    
    if (tbody) {
        new Sortable(tbody, {
            handle: '.drag-handle',
            animation: 150,
            ghostClass: 'opacity-50',
            onEnd: updateOrder
        });
    }
});

// Update Order via AJAX
function updateOrder() {
    const rows = document.querySelectorAll('.locale-row');
    const orders = Array.from(rows).map(row => row.dataset.id);
    
    fetch('/admin/locales/update-order', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify({ orders })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            toast.success('{{ admin_trans('order_updated_successfully') }}');
            setTimeout(() => location.reload(), 1000);
        }
    })
    .catch(() => {
        toast.error('{{ admin_trans('error_updating_order') }}');
    });
}
</script>
@endpush