@extends('admin.layouts.app')

@section('title', admin_trans('countries'))
@section('page-title', admin_trans('countries'))

@section('content')

<div class="space-y-6">
    
    <!-- Search Bar + Add Button -->
    <div class="card">
        <div class="flex items-center gap-3">
            <form method="GET" action="{{ route('admin.countries.index') }}" id="search-form" class="flex-1">
                <x-search-input 
                    name="search"
                    :value="request('search')"
                    :placeholder="admin_trans('search_countries')"
                    oninput="clearTimeout(window.searchTimeout); window.searchTimeout = setTimeout(() => this.form.submit(), 500)"
                    class="w-full"
                />
            </form>
            
            <x-button 
                variant="primary" 
                icon="plus"
                @click="$dispatch('open-modal', 'create-country-modal')"
            >
                {{ admin_trans('add_country') }}
            </x-button>
        </div>
    </div>

    <!-- Countries Table Card -->
    <div class="card">
        
        @if($countries->isEmpty())
            @if(request('search'))
                <!-- No Search Results -->
                <x-empty-state 
                    icon="search-x"
                    :title="admin_trans('no_results_found')"
                    :description="admin_trans('try_different_search')"
                >
                    <x-button 
                        variant="outline"
                        icon="x"
                        onclick="window.location.href='{{ route('admin.countries.index') }}'"
                    >
                        {{ admin_trans('clear') }}
                    </x-button>
                </x-empty-state>
            @else
                <!-- No Countries Yet -->
                <x-empty-state 
                    icon="globe"
                    :title="admin_trans('no_countries_yet')"
                    :description="admin_trans('start_by_adding_first_country')"
                >
                    <x-button 
                        variant="primary"
                        icon="plus"
                        @click="$dispatch('open-modal', 'create-country-modal')"
                    >
                        {{ admin_trans('add_country') }}
                    </x-button>
                </x-empty-state>
            @endif
        @else

<!-- Countries Table -->
<x-enhanced-table 
    :headers="[
        'order' => [
            'text' => admin_trans('order'),
            'align' => 'center',
            'width' => '80px'
        ],
        'flag' => [
            'text' => admin_trans('flag'),
            'align' => 'center',
            'width' => '60px'
        ],
        'name' => [
            'text' => admin_trans('name'),
            'align' => 'auto',
            'width' => '200px'
        ],
        'code' => [
            'text' => admin_trans('codes'),
            'align' => 'center',
            'width' => '100px'
        ],
        'currency' => [
            'text' => admin_trans('currency'),
            'align' => 'center',
            'width' => '100px'
        ],
        'dial_code' => [
            'text' => admin_trans('dial_code'),
            'align' => 'center',
            'width' => '80px'
        ],
        'status' => [
            'text' => admin_trans('status'),
            'align' => 'center',
            'width' => '100px'
        ],
        'default' => [
            'text' => admin_trans('default'),
            'align' => 'center',
            'width' => '100px'
        ],
        'actions' => [
            'text' => admin_trans('actions'),
            'align' => 'center',
            'width' => '160px'
        ]
    ]"
    :responsive="true"
>
    @foreach($countries as $country)
    <tr class="country-row hover:bg-primary-light transition-colors" data-id="{{ $country->id }}">
        
        <!-- Order (Drag Handle) -->
        <x-table-cell type="number">
            <div class="flex items-center justify-center gap-2">
                <button class="drag-handle cursor-move p-1 rounded hover:bg-primary-light">
                    <i data-lucide="grip-vertical" class="w-3 h-3 text-tertiary"></i>
                </button>
                <span class="text-sm font-medium text-tertiary">{{ $country->order ?? 0 }}</span>
            </div>
        </x-table-cell>
        
        <!-- Flag -->
        <x-table-cell type="center">
            <div class="flex items-center justify-center">
                @if($country->flag_url)
                    <img src="{{ $country->flag_url }}" 
                         alt="{{ $country->name }} Flag" 
                         class="w-8 h-6 object-cover rounded border"
                         onerror="this.style.display='none'; this.nextElementSibling.style.display='block';">
                    <div class="hidden w-8 h-6 bg-gray-200 rounded border flex items-center justify-center">
                        <span class="text-xs">{{ $country->flag_emoji ?? '🏳️' }}</span>
                    </div>
                @else
                    <div class="w-8 h-6 bg-gray-200 rounded border flex items-center justify-center">
                        <span class="text-xs">{{ $country->flag_emoji ?? '🏳️' }}</span>
                    </div>
                @endif
            </div>
        </x-table-cell>
        
        <!-- Name -->
        <x-table-cell type="text">
            <div>
                <span class="text-sm font-medium text-primary">{{ $country->name ?? $country->code }}</span>
                @if($country->name && $country->name !== $country->code)
                    <div class="text-xs text-tertiary">{{ $country->code }}</div>
                @endif
            </div>
        </x-table-cell>
        
        <!-- Codes -->
        <x-table-cell type="center">
            <div class="space-y-1">
                @if($country->iso_code)
                    <x-badge variant="secondary" class="text-xs">{{ $country->iso_code }}</x-badge>
                @endif
                @if($country->iso2 && $country->iso2 !== $country->code)
                    <x-badge variant="info" class="text-xs">{{ $country->iso2 }}</x-badge>
                @endif
            </div>
        </x-table-cell>
        
        <!-- Currency -->
        <x-table-cell type="center">
            @if($country->currency_code || $country->currency)
                <div class="text-sm">
                    <div class="font-medium text-primary">{{ $country->currency_code ?? $country->currency }}</div>
                    @if($country->currency_symbol)
                        <div class="text-xs text-tertiary">{{ $country->currency_symbol }}</div>
                    @endif
                </div>
            @else
                <span class="text-xs text-tertiary">-</span>
            @endif
        </x-table-cell>
        
        <!-- Dial Code -->
        <x-table-cell type="center">
            @if($country->dial_code)
                <x-badge variant="info">{{ $country->dial_code }}</x-badge>
            @else
                <span class="text-xs text-tertiary">-</span>
            @endif
        </x-table-cell>
        
        <!-- Status with Toggle -->
        <x-table-cell type="actions">
            <form method="POST" action="{{ route('admin.countries.toggle-status', $country) }}">
                @csrf
                @method('PATCH')
                <x-toggle 
                    name="status_toggle_{{ $country->id }}"
                    :checked="$country->status === 'active'"
                    size="sm"
                    variant="success"
                    onchange="this.form.submit()"
                />
            </form>
        </x-table-cell>

        <!-- Is Default -->
        <x-table-cell type="center">
            @if($country->is_default)
                <x-badge variant="primary">
                    {{ admin_trans('default') }}
                </x-badge>
            @else
                <form method="POST" action="{{ route('admin.countries.set-default', $country) }}">
                    @csrf
                    @method('PATCH')
                    <button type="submit" class="text-xs text-tertiary hover:text-primary transition-colors">
                        {{ admin_trans('set_default') }}
                    </button>
                </form>
            @endif
        </x-table-cell>
        
        <!-- Actions -->
        <x-table-cell type="actions">
            <div class="flex items-center justify-center gap-2">
                <!-- API Sync Button -->
                @if($country->iso_code)
                    <form method="POST" action="{{ route('admin.countries.update-flag', $country) }}" class="inline">
                        @csrf
                        @method('PATCH')
                        <button 
                            type="submit"
                            class="p-2 rounded-lg hover:bg-green-50 text-green-600 transition-colors"
                            title="{{ admin_trans('sync_from_api') }}"
                        >
                            <i data-lucide="refresh-cw" class="w-4 h-4"></i>
                        </button>
                    </form>
                @endif
                
                <!-- Edit Button -->
                <button 
                    class="p-2 rounded-lg hover:bg-blue-50 text-blue-600 transition-colors"
                    onclick="editCountry({{ json_encode($country) }})"
                    title="{{ admin_trans('edit') }}"
                >
                    <i data-lucide="edit-3" class="w-4 h-4"></i>
                </button>
                
                <!-- Translations Button -->
                <a href="{{ route('admin.countries.translations', $country) }}"
                   class="p-2 rounded-lg hover:bg-purple-50 text-purple-600 transition-colors"
                   title="{{ admin_trans('translations') }}">
                    <i data-lucide="languages" class="w-4 h-4"></i>
                </a>
                
                @if(!$country->is_default)
                    <!-- Delete Button -->
                    <button 
                        class="p-2 rounded-lg hover:bg-red-50 text-red-600 transition-colors"
                        onclick="confirmDelete({{ $country->id }}, '{{ $country->name ?? $country->code }}')"
                        title="{{ admin_trans('delete') }}"
                    >
                        <i data-lucide="trash-2" class="w-4 h-4"></i>
                    </button>
                @endif
            </div>
        </x-table-cell>
    </tr>
    @endforeach
</x-enhanced-table>

            <!-- Footer Info & Pagination -->
            <div class="mt-6 flex flex-col sm:flex-row items-center justify-between gap-4 border-t pt-4" style="border-color: var(--color-border);">
                <div class="flex items-center gap-2 text-sm text-tertiary">
                    <i data-lucide="info" class="w-4 h-4"></i>
                    <span>{{ admin_trans('drag_rows_to_reorder') }}</span>
                </div>
                
                @if($countries->hasPages())
                    <x-pagination :paginator="$countries" />
                @endif
            </div>
        @endif

    </div>
</div>

<!-- Create Country Modal -->
<x-modal name="create-country-modal" maxWidth="lg">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('add_country') }}</h3>
            <button @click="$dispatch('close-modal', 'create-country-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <form method="POST" action="{{ route('admin.countries.store') }}" class="space-y-4" x-data="countryForm()" onsubmit="debugForm(event)">

            <!-- ISO Code + API Fetch -->
            <div class="flex gap-3">
                <div class="flex-1">
                    <x-input 
                        :label="admin_trans('iso_code')"
                        name="iso_code"
                        placeholder="EGY, USA, SAU..."
                        :hint="admin_trans('iso_3_letter_code')"
                        ::value="isoCode"
                        @input="isoCode = $event.target.value"
                    />
                </div>
                <div class="flex items-end">
                    <x-button 
                        type="button"
                        variant="secondary"
                        icon="download"
                        @click="fetchFromApi()"
                        :loading="false"
                        x-bind:disabled="!isoCode || fetching"
                        x-text="fetching ? '{{ admin_trans('fetching') }}' : '{{ admin_trans('fetch_from_api') }}'"
                    ></x-button>
                </div>
            </div>

            @csrf
            <input type="hidden" name="fetch_from_api" x-bind:value="fetchFromApiFlag ? '1' : '0'">

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('country_name')"
                    name="name"
                    placeholder="Egypt, United States..."
                    :required="true"
                    ::value="formData.name"
                />

                <x-input 
                    :label="admin_trans('country_code')"
                    name="code"
                    placeholder="EG, US, SA..."
                    :required="true"
                    :hint="admin_trans('iso_2_letter_code')"
                    ::value="formData.code"
                />
            </div>

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('iso2_code')"
                    name="iso2"
                    placeholder="EG, US..."
                    :hint="admin_trans('iso_2_letter_code')"
                    ::value="formData.iso2"
                />

                <x-input 
                    :label="admin_trans('dial_code')"
                    name="dial_code"
                    placeholder="+20, +1, +966..."
                    ::value="formData.dial_code"
                />
            </div>

            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <x-input 
                    :label="admin_trans('currency')"
                    name="currency"
                    placeholder="EGP, USD..."
                    ::value="formData.currency"
                />

                <x-input 
                    :label="admin_trans('currency_code')"
                    name="currency_code"
                    placeholder="EGP, USD..."
                    ::value="formData.currency_code"
                />

                <x-input 
                    :label="admin_trans('currency_symbol')"
                    name="currency_symbol"
                    placeholder="E£, $, ر.س..."
                    ::value="formData.currency_symbol"
                />
            </div>

            <x-input 
                :label="admin_trans('flag_url')"
                name="flag_url"
                type="url"
                placeholder="https://flagcdn.com/w320/eg.png"
                :hint="admin_trans('flag_image_url')"
                ::value="formData.flag_url"
            />

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('order')"
                    name="order"
                    type="number"
                    value="0"
                    :hint="admin_trans('display_order')"
                />

                <x-select 
                    :label="admin_trans('status')"
                    name="status"
                    :required="true"
                >
                    <option value="draft">{{ admin_trans('draft') }}</option>
                    <option value="active" selected>{{ admin_trans('active') }}</option>
                </x-select>
            </div>

            <div class="flex items-center gap-3">
                <input 
                    type="checkbox" 
                    name="is_default" 
                    id="create_is_default"
                    value="1"
                    class="checkbox"
                >
                <label for="create_is_default" class="text-sm font-medium text-primary cursor-pointer">
                    {{ admin_trans('set_as_default') }}
                </label>
            </div>

            <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                <x-button 
                    variant="outline" 
                    type="button"
                    @click="$dispatch('close-modal', 'create-country-modal')"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                <x-button variant="primary" type="submit" icon="check">
                    {{ admin_trans('create') }}
                </x-button>
            </div>
        </form>
    </div>
</x-modal>

<!-- Edit Country Modal -->
<x-modal name="edit-country-modal" maxWidth="lg">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('edit_country') }}</h3>
            <button @click="$dispatch('close-modal', 'edit-country-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <form id="edit-country-form" method="POST" class="space-y-4">
            @csrf
            @method('PUT')

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('country_name')"
                    name="name"
                    id="edit_name"
                    :required="true"
                />

                <x-input 
                    :label="admin_trans('country_code')"
                    name="code"
                    id="edit_code"
                    :required="true"
                />
            </div>

            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <x-input 
                    :label="admin_trans('iso_code')"
                    name="iso_code"
                    id="edit_iso_code"
                />

                <x-input 
                    :label="admin_trans('iso2_code')"
                    name="iso2"
                    id="edit_iso2"
                />

                <x-input 
                    :label="admin_trans('dial_code')"
                    name="dial_code"
                    id="edit_dial_code"
                />
            </div>

            <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                <x-input 
                    :label="admin_trans('currency')"
                    name="currency"
                    id="edit_currency"
                />

                <x-input 
                    :label="admin_trans('currency_code')"
                    name="currency_code"
                    id="edit_currency_code"
                />

                <x-input 
                    :label="admin_trans('currency_symbol')"
                    name="currency_symbol"
                    id="edit_currency_symbol"
                />
            </div>

            <x-input 
                :label="admin_trans('flag_url')"
                name="flag_url"
                type="url"
                id="edit_flag_url"
            />

            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('order')"
                    name="order"
                    type="number"
                    id="edit_order"
                />

                <x-select 
                    :label="admin_trans('status')"
                    name="status"
                    id="edit_status"
                    :required="true"
                >
                    <option value="draft">{{ admin_trans('draft') }}</option>
                    <option value="active">{{ admin_trans('active') }}</option>
                </x-select>
            </div>

            <div class="flex items-center gap-3">
                <input 
                    type="checkbox" 
                    name="is_default" 
                    id="edit_is_default"
                    value="1"
                    class="checkbox"
                >
                <label for="edit_is_default" class="text-sm font-medium text-primary cursor-pointer">
                    {{ admin_trans('set_as_default') }}
                </label>
            </div>

            <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                <x-button 
                    variant="outline" 
                    type="button"
                    @click="$dispatch('close-modal', 'edit-country-modal')"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                <x-button variant="primary" type="submit" icon="check">
                    {{ admin_trans('update') }}
                </x-button>
            </div>
        </form>
    </div>
</x-modal>

<!-- Delete Confirmation Modal -->
<form id="delete-country-form" method="POST">
    @csrf
    @method('DELETE')
</form>

<x-confirm-modal 
    name="delete-country-modal"
    :title="admin_trans('confirm_delete')"
    message=""
    :confirmText="admin_trans('yes_delete')"
    :cancelText="admin_trans('cancel')"
    confirmVariant="danger"
    icon="trash-2"
    iconColor="danger"
    x-data="{ message: '' }"
/>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/sortablejs@1.15.0/Sortable.min.js"></script>
<script>

// Debugging function for form submission
function debugForm(event) {
    console.log('Form is being submitted...');
    const formData = new FormData(event.target);
    console.log('Form data:');
    for (let [key, value] of formData.entries()) {
        console.log(`${key}: ${value}`);
    }
    
    // Check if required fields are filled
    const name = formData.get('name');
    const code = formData.get('code');
    
    if (!name || !code) {
        console.error('Missing required fields:', {
            name: name || 'MISSING',
            code: code || 'MISSING'
        });
    }
}

// Alpine.js Country Form Component
function countryForm() {
    return {
        isoCode: '',
        fetching: false,
        fetchFromApiFlag: false,
        formData: {
            name: '',
            code: '',
            iso2: '',
            dial_code: '',
            currency: '',
            currency_code: '',
            currency_symbol: '',
            flag_url: ''
        },

        async fetchFromApi() {
            if (!this.isoCode) {
                console.warn('No ISO code provided');
                return;
            }
            
            console.log('Fetching data for ISO code:', this.isoCode);
            this.fetching = true;
            
            try {
                const response = await fetch(`/admin/countries/fetch-api`, {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    },
                    body: JSON.stringify({ iso_code: this.isoCode })
                });

                console.log('API Response status:', response.status);
                const result = await response.json();
                console.log('API Response data:', result);
                
                if (result.success && result.data) {
                    // Fill form with API data
                    Object.assign(this.formData, result.data);
                    this.fetchFromApiFlag = true;
                    
                    console.log('Updated formData:', this.formData);
                    
                    // Update actual form inputs
                    setTimeout(() => {
                        const form = document.querySelector('[x-data*="countryForm"]');
                        if (form) {
                            Object.keys(this.formData).forEach(key => {
                                const input = form.querySelector(`[name="${key}"]`);
                                if (input) {
                                    console.log(`Setting ${key} to:`, this.formData[key]);
                                    input.value = this.formData[key] || '';
                                }
                            });
                            
                            // Update ISO code input too
                            const isoInput = form.querySelector('[name="iso_code"]');
                            if (isoInput) {
                                isoInput.value = this.isoCode;
                                console.log('Set ISO code input to:', this.isoCode);
                            }
                        }
                    }, 100);
                    
                    toast.success('{{ admin_trans("data_fetched_successfully") }}');
                } else {
                    console.error('API Error:', result.message);
                    toast.error(result.message || '{{ admin_trans("country_not_found") }}');
                }
            } catch (error) {
                console.error('Fetch error:', error);
                toast.error('{{ admin_trans("error_fetching_data") }}');
            } finally {
                this.fetching = false;
            }
        }
    }
}

// Edit Country Function
function editCountry(country) {
    const form = document.getElementById('edit-country-form');
    form.action = `/admin/countries/${country.id}`;
    
    document.getElementById('edit_name').value = country.name || '';
    document.getElementById('edit_code').value = country.code || '';
    document.getElementById('edit_iso_code').value = country.iso_code || '';
    document.getElementById('edit_iso2').value = country.iso2 || '';
    document.getElementById('edit_dial_code').value = country.dial_code || '';
    document.getElementById('edit_currency').value = country.currency || '';
    document.getElementById('edit_currency_code').value = country.currency_code || '';
    document.getElementById('edit_currency_symbol').value = country.currency_symbol || '';
    document.getElementById('edit_flag_url').value = country.flag_url || '';
    document.getElementById('edit_order').value = country.order || 0;
    document.getElementById('edit_status').value = country.status || 'draft';
    document.getElementById('edit_is_default').checked = country.is_default || false;
    
    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'edit-country-modal' }));
}

// Confirm Delete Function
function confirmDelete(id, name) {
    const form = document.getElementById('delete-country-form');
    form.action = `/admin/countries/${id}`;
    
    // Submit form on confirm
    window.addEventListener('confirm-action', function handler(e) {
        if (e.detail === 'delete-country-modal') {
            form.submit();
            window.removeEventListener('confirm-action', handler);
        }
    });
    
    window.dispatchEvent(new CustomEvent('open-modal', { detail: 'delete-country-modal' }));
}

// Sortable.js for Drag & Drop
document.addEventListener('DOMContentLoaded', function() {
    const tbody = document.querySelector('.country-row')?.parentElement;
    
    if (tbody) {
        new Sortable(tbody, {
            handle: '.drag-handle',
            animation: 150,
            ghostClass: 'opacity-50',
            onEnd: updateOrder
        });
    }
});

// Update Order via AJAX
function updateOrder() {
    const rows = document.querySelectorAll('.country-row');
    const orders = Array.from(rows).map(row => row.dataset.id);
    
    fetch('/admin/countries/update-order', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
        },
        body: JSON.stringify({ orders })
    })
    .then(response => response.json())
    .then(data => {
        if (data.success) {
            toast.success('{{ admin_trans('order_updated_successfully') }}');
            setTimeout(() => location.reload(), 1000);
        }
    })
    .catch(() => {
        toast.error('{{ admin_trans('error_updating_order') }}');
    });
}
</script>
@endpush