/**
 * Toast Notification System - Final Version
 * English: Left position | Arabic: Center position
 */

export class Toast {
    constructor() {
        this.container = null;
        this.toasts = [];
        this.init();
    }

    init() {
        // Create container if it doesn't exist
        if (!document.getElementById('toast-container')) {
            this.container = document.createElement('div');
            this.container.id = 'toast-container';
            
            // Positioning: English = right, Arabic = left
            const isRTL = document.documentElement.getAttribute('dir') === 'rtl';
            this.container.className = `fixed top-4 z-[9999] flex flex-col gap-3 pointer-events-none ${
                isRTL ? 'left-4' : 'right-4'
            } max-w-sm`;
            
            // Ensure proper spacing from screen edges
            this.container.style.maxWidth = 'calc(100vw - 2rem)';
            this.container.style.width = 'auto';
            
            document.body.appendChild(this.container);
        } else {
            this.container = document.getElementById('toast-container');
        }
    }

    show(options = {}) {
        const {
            type = 'info',
            title = '',
            message = '',
            duration = 5000,
            closeable = true,
        } = options;

        const id = `toast-${Date.now()}-${Math.random()}`;
        const toast = this.createToast(id, type, title, message, closeable);

        this.container.appendChild(toast);
        this.toasts.push(id);

        // Trigger animation
        requestAnimationFrame(() => {
            toast.classList.add('toast-show');
        });

        // Auto remove
        if (duration > 0) {
            setTimeout(() => {
                this.remove(id);
            }, duration);
        }

        return id;
    }

    createToast(id, type, title, message, closeable) {
        const toast = document.createElement('div');
        toast.id = id;
        toast.className = 'toast pointer-events-auto';
        toast.setAttribute('role', 'alert');

        const config = this.getTypeConfig(type);
        const isRTL = document.documentElement.getAttribute('dir') === 'rtl';

        toast.innerHTML = `
            <div class="flex items-start gap-3 p-4 rounded-xl shadow-lg border backdrop-blur-sm w-full max-w-md" 
                 style="background: var(--color-bg-primary); border-color: ${config.borderColor}; min-width: 320px;">
                
                <!-- Icon -->
                <div class="flex-shrink-0 w-10 h-10 rounded-lg flex items-center justify-center" 
                     style="background: ${config.bgColor};">
                    ${config.icon}
                </div>

                <!-- Content -->
                <div class="flex-1 min-w-0" style="direction: ${isRTL ? 'rtl' : 'ltr'};">
                    ${title ? `<h4 class="font-semibold text-sm mb-0.5" style="color: var(--color-text-primary);">${title}</h4>` : ''}
                    ${message ? `<p class="text-sm" style="color: var(--color-text-secondary);">${message}</p>` : ''}
                </div>

                <!-- Close Button -->
                ${closeable ? `
                    <button 
                        onclick="window.toastManager.remove('${id}')"
                        class="flex-shrink-0 p-1 rounded-lg transition-colors hover:bg-gray-100 dark:hover:bg-gray-800"
                        style="color: var(--color-text-tertiary);"
                    >
                        <svg class="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M6 18L18 6M6 6l12 12"/>
                        </svg>
                    </button>
                ` : ''}
            </div>
        `;

        return toast;
    }

    getTypeConfig(type) {
        const configs = {
            success: {
                bgColor: 'var(--color-success-50)',
                borderColor: 'var(--color-success-500)',
                icon: `<svg class="w-6 h-6" style="color: var(--color-success-500);" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>`
            },
            error: {
                bgColor: 'var(--color-danger-50)',
                borderColor: 'var(--color-danger-500)',
                icon: `<svg class="w-6 h-6" style="color: var(--color-danger-500);" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M10 14l2-2m0 0l2-2m-2 2l-2-2m2 2l2 2m7-2a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>`
            },
            warning: {
                bgColor: 'var(--color-warning-50)',
                borderColor: 'var(--color-warning-500)',
                icon: `<svg class="w-6 h-6" style="color: var(--color-warning-500);" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-3L13.732 4c-.77-1.333-2.694-1.333-3.464 0L3.34 16c-.77 1.333.192 3 1.732 3z"/>
                </svg>`
            },
            info: {
                bgColor: 'var(--color-info-50)',
                borderColor: 'var(--color-info-500)',
                icon: `<svg class="w-6 h-6" style="color: var(--color-info-500);" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path stroke-linecap="round" stroke-linejoin="round" stroke-width="2" d="M13 16h-1v-4h-1m1-4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z"/>
                </svg>`
            }
        };

        return configs[type] || configs.info;
    }

    remove(id) {
        const toast = document.getElementById(id);
        if (toast) {
            toast.classList.remove('toast-show');
            toast.classList.add('toast-hide');
            
            setTimeout(() => {
                toast.remove();
                this.toasts = this.toasts.filter(t => t !== id);
            }, 300);
        }
    }

    success(message, title = 'نجح!') {
        return this.show({ type: 'success', title, message });
    }

    error(message, title = 'خطأ!') {
        return this.show({ type: 'error', title, message });
    }

    warning(message, title = 'تحذير!') {
        return this.show({ type: 'warning', title, message });
    }

    info(message, title = 'معلومة') {
        return this.show({ type: 'info', title, message });
    }

    clearAll() {
        this.toasts.forEach(id => this.remove(id));
    }
}

// Add CSS for toast animations with proper positioning
const style = document.createElement('style');
style.textContent = `
    #toast-container {
        position: fixed !important;
        z-index: 9999 !important;
    }

    /* Arabic: Left positioning */
    [dir="rtl"] #toast-container {
        left: 1rem !important;
        right: auto !important;
        transform: none !important;
    }

    /* English: Right positioning */
    [dir="ltr"] #toast-container {
        right: 1rem !important;
        left: auto !important;
        transform: none !important;
    }

    .toast {
        opacity: 0;
        transform: translateY(-20px) scale(0.95);
        transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
        width: 100%;
        max-width: 400px;
    }

    .toast-show {
        opacity: 1;
        transform: translateY(0) scale(1);
    }

    .toast-hide {
        opacity: 0;
        transform: translateY(-20px) scale(0.95);
    }

    /* Responsive breakpoints */
    @media (max-width: 640px) {
        [dir="rtl"] #toast-container {
            left: 0.5rem !important;
            right: auto !important;
            transform: none !important;
            max-width: calc(100vw - 1rem) !important;
        }
        
        [dir="ltr"] #toast-container {
            right: 0.5rem !important;
            left: auto !important;
            transform: none !important;
            max-width: calc(100vw - 1rem) !important;
        }
        
        .toast {
            min-width: auto !important;
            max-width: 100% !important;
        }
    }
`;
document.head.appendChild(style);

// Initialize global toast manager
if (typeof window !== 'undefined') {
    window.toastManager = new Toast();
    
    // Expose helper methods
    window.toast = {
        success: (message, title) => window.toastManager.success(message, title),
        error: (message, title) => window.toastManager.error(message, title),
        warning: (message, title) => window.toastManager.warning(message, title),
        info: (message, title) => window.toastManager.info(message, title),
    };
}