<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Spatie\MediaLibrary\HasMedia;
use Spatie\MediaLibrary\InteractsWithMedia;

class Store extends Model implements HasMedia
{
    use HasFactory, InteractsWithMedia;

    protected $fillable = [
        'user_id',
    ];

    // Relations
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function translations()
    {
        return $this->hasMany(StoreTranslation::class);
    }

    public function countries()
    {
        return $this->belongsToMany(Country::class, 'country_store')
                    ->using(CountryStore::class)
                    ->withPivot(['user_id', 'status', 'is_featured', 'order'])
                    ->withTimestamps();
    }

    public function countryStores()
    {
        return $this->hasMany(CountryStore::class, 'store_id', 'id');
    }

    // ✅ FIX: Offers relation through CountryStore with explicit foreign keys
    public function offers()
    {
        return $this->hasManyThrough(
            Offer::class,           // النموذج النهائي
            CountryStore::class,    // النموذج الوسيط
            'store_id',             // Foreign key في country_store
            'country_store_id',     // Foreign key في offers
            'id',                   // Local key في stores
            'id'                    // Local key في country_store
        );
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->whereHas('countryStores', function ($q) {
            $q->where('status', 'active');
        });
    }

    public function scopeFeatured($query)
    {
        return $query->whereHas('countryStores', function ($q) {
            $q->where('is_featured', true);
        });
    }

    public function scopeByCountry($query, $countryId)
    {
        return $query->whereHas('countries', function ($q) use ($countryId) {
            $q->where('countries.id', $countryId);
        });
    }

    // Helper Methods
    public function getTranslation($localeId)
    {
        return $this->translations()->where('locale_id', $localeId)->first();
    }

    public function getName($localeId = null)
    {
        if ($localeId) {
            $translation = $this->getTranslation($localeId);
            return $translation ? $translation->name : 'Store #' . $this->id;
        }
        return 'Store #' . $this->id;
    }

    public function getDescription($localeId = null)
    {
        if ($localeId) {
            $translation = $this->getTranslation($localeId);
            return $translation ? $translation->description : null;
        }
        return null;
    }

    public function getCountryStore($countryId)
    {
        return $this->countryStores()->where('country_id', $countryId)->first();
    }

    // Media Collections
    public function registerMediaCollections(): void
    {
        $this->addMediaCollection('logo')
              ->singleFile()
              ->acceptsMimeTypes(['image/jpeg', 'image/png', 'image/webp']);

        $this->addMediaCollection('gallery')
              ->acceptsMimeTypes(['image/jpeg', 'image/png', 'image/webp']);
    }
}