<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Locale extends Model
{
    use HasFactory;

    protected $fillable = [
        'language_id',
        'country_id',
        'code',
        'name',
        'order',
        'is_default',
        'status',
    ];

    protected $casts = [
        'is_default' => 'boolean',
        'order' => 'integer',
    ];

    // Relations
    public function language()
    {
        return $this->belongsTo(Language::class);
    }

    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function carouselSlideTranslations()
    {
        return $this->hasMany(CarouselSlideTranslation::class);
    }

    public function offerTranslations()
    {
        return $this->hasMany(OfferTranslation::class);
    }

    public function pageTranslations()
    {
        return $this->hasMany(PageTranslation::class);
    }

    public function storeTranslations()
    {
        return $this->hasMany(StoreTranslation::class);
    }

    public function userLocales()
    {
        return $this->hasMany(UserLocale::class);
    }

    public function occasionTranslations()  
    {
        return $this->hasMany(OccasionTranslation::class);
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeDefault($query)
    {
        return $query->where('is_default', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('order')->orderBy('id');
    }

    public function scopeByCode($query, $code)
    {
        return $query->where('code', $code);
    }

    public function scopeByLanguage($query, $languageId)
    {
        return $query->where('language_id', $languageId);
    }

    public function scopeByCountry($query, $countryId)
    {
        return $query->where('country_id', $countryId);
    }

    // Accessors
    public function getIsActiveAttribute()
    {
        return $this->status === 'active';
    }

    public function getDisplayNameAttribute()
    {
        return $this->name ?: $this->code;
    }

    public function getFullNameAttribute()
    {
        return $this->language->name . ' (' . $this->country->code . ')';
    }

    /**
     * Get localized country name based on language
     */
    public function getLocalizedCountryNameAttribute()
    {
        // Get country translation for this locale's language
        $translation = $this->country->getTranslation($this->language_id);
        
        if ($translation) {
            return $translation->name;
        }
        
        // Fallback to country's default name
        return $this->country->name ?? $this->country->code;
    }

    /**
     * Get formatted locale display with localized country name
     */
    public function getLocalizedDisplayNameAttribute()
    {
        $countryName = $this->getLocalizedCountryNameAttribute();
        return $this->language->native_name . ' (' . $countryName . ')';
    }

    /**
     * Helper method to get locale data with proper country names
     */
    public function toLocalizedArray()
    {
        return [
            'id' => $this->id,
            'code' => $this->code,
            'name' => $this->name,
            'is_default' => $this->is_default,
            'language' => [
                'id' => $this->language->id,
                'name' => $this->language->name,
                'native_name' => $this->language->native_name,
                'code' => $this->language->code,
            ],
            'country' => [
                'id' => $this->country->id,
                'name' => $this->getLocalizedCountryNameAttribute(),
                'original_name' => $this->country->name,
                'code' => $this->country->code,
                'flag_emoji' => $this->country->flag_emoji,
            ],
            'display_name' => $this->getLocalizedDisplayNameAttribute(),
        ];
    }
}