<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;

class Ban extends Model
{
    use HasFactory, SoftDeletes;

    protected $fillable = [
        'bannable_type',
        'bannable_id',
        'created_by_type',
        'created_by_id',
        'comment',
        'expired_at',
    ];

    protected $casts = [
        'expired_at' => 'datetime',
    ];

    // Relations
    public function bannable()
    {
        return $this->morphTo();
    }

    public function createdBy()
    {
        return $this->morphTo('created_by');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where(function ($q) {
            $q->whereNull('expired_at')
              ->orWhere('expired_at', '>', now());
        });
    }

    public function scopeExpired($query)
    {
        return $query->where('expired_at', '<=', now());
    }

    public function scopeForModel($query, $model)
    {
        return $query->where([
            'bannable_type' => get_class($model),
            'bannable_id' => $model->id,
        ]);
    }

    // Accessors
    public function getIsActiveAttribute()
    {
        return is_null($this->expired_at) || $this->expired_at->isFuture();
    }

    public function getIsExpiredAttribute()
    {
        return $this->expired_at && $this->expired_at->isPast();
    }

    // Helper Methods
    public static function banModel($model, $createdBy = null, $comment = null, $expiresAt = null)
    {
        return static::create([
            'bannable_type' => get_class($model),
            'bannable_id' => $model->id,
            'created_by_type' => $createdBy ? get_class($createdBy) : null,
            'created_by_id' => $createdBy?->id,
            'comment' => $comment,
            'expired_at' => $expiresAt,
        ]);
    }

    public static function unbanModel($model)
    {
        return static::where([
            'bannable_type' => get_class($model),
            'bannable_id' => $model->id,
        ])->active()->delete();
    }

    public static function isBanned($model)
    {
        return static::where([
            'bannable_type' => get_class($model),
            'bannable_id' => $model->id,
        ])->active()->exists();
    }
}