<?php

namespace App\Http\Middleware;

use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\RateLimiter;
use Symfony\Component\HttpFoundation\Response;

class SubscriptionRateLimiter
{
    /**
     * Handle an incoming request.
     *
     * @param  \Closure(\Illuminate\Http\Request): (\Symfony\Component\HttpFoundation\Response)  $next
     */
    public function handle(Request $request, Closure $next, string $action = 'general'): Response
    {
        $user = $request->user();
        
        if (!$user) {
            return response()->json([
                'success' => false,
                'message' => 'Authentication required'
            ], 401);
        }

        // Different rate limits for different actions
        $limits = $this->getRateLimits($action);
        
        $key = $this->getRateLimitKey($user, $action);
        
        if (RateLimiter::tooManyAttempts($key, $limits['max_attempts'])) {
            $seconds = RateLimiter::availableIn($key);
            
            return response()->json([
                'success' => false,
                'message' => 'Too many attempts. Please try again in ' . $seconds . ' seconds.',
                'retry_after' => $seconds,
            ], 429);
        }

        RateLimiter::hit($key, $limits['decay_seconds']);

        $response = $next($request);

        // Add rate limit headers
        $response->headers->add([
            'X-RateLimit-Limit' => $limits['max_attempts'],
            'X-RateLimit-Remaining' => RateLimiter::remaining($key, $limits['max_attempts']),
            'X-RateLimit-Reset' => now()->addSeconds(RateLimiter::availableIn($key))->timestamp,
        ]);

        return $response;
    }

    /**
     * Get rate limit configuration for different actions
     */
    private function getRateLimits(string $action): array
    {
        return match($action) {
            'subscribe' => [
                'max_attempts' => 20,      // 20 subscriptions per minute
                'decay_seconds' => 60,
            ],
            'unsubscribe' => [
                'max_attempts' => 30,      // 30 unsubscriptions per minute
                'decay_seconds' => 60,
            ],
            'bulk' => [
                'max_attempts' => 5,       // 5 bulk operations per minute
                'decay_seconds' => 60,
            ],
            'check' => [
                'max_attempts' => 100,     // 100 status checks per minute
                'decay_seconds' => 60,
            ],
            default => [
                'max_attempts' => 60,      // General rate limit
                'decay_seconds' => 60,
            ],
        };
    }

    /**
     * Generate rate limit key
     */
    private function getRateLimitKey($user, string $action): string
    {
        return "subscription_rate_limit:{$user->id}:{$action}";
    }
}