<?php

namespace App\Http\Middleware;

use App\Models\Country;
use App\Models\Language;
use App\Models\Locale;
use App\Models\UserLocale;
use Closure;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;

class LocaleApiMiddleware
{
    /**
     * Handle an incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  \Closure  $next
     * @return mixed
     */
    public function handle(Request $request, Closure $next)
    {
        // ========================================
        // 1. أولوية تحديد البلد واللغة (مُحدثة):
        // ========================================
        
        $countryId = null;
        $languageId = null;
        $localeId = null;
        $savedLocaleIds = null; // ✅ تعريف المتغير في البداية
        
        // أولاً: جلب إعدادات اليوزر المحفوظة (إذا كان مسجل دخول)
        $user = $request->user();
        if ($user) {
            // إعدادات اليوزر المحفوظة لها الأولوية الأولى
            $countryId = $user->current_country_id ?? $user->country_id;
            $languageId = $user->language_id;
            
            // جلب الـ Locale المحفوظ للمستخدم بطريقة آمنة
            $savedLocaleIds = UserLocale::getUserLocaleIds($user->id);
            if ($savedLocaleIds) {
                $countryId = $savedLocaleIds['country_id'];
                $languageId = $savedLocaleIds['language_id'];
                $localeId = $savedLocaleIds['locale_id'];
            }
        }
        
        // ثانياً: إذا لم توجد إعدادات محفوظة، جرب من الـ Request
        if (!$countryId) {
            $countryId = $request->input('country_id') ?? $request->header('X-Country-ID');
        }
        
        if (!$languageId) {
            $languageId = $request->input('language_id') ?? $request->header('X-Language-ID');
        }
        
        if (!$localeId) {
            $localeId = $request->input('locale_id') ?? $request->header('X-Locale-ID');
        }
        
        // ثالثاً: إذا ما زال فارغ، استخدم IP Detection أو الافتراضي
        $ipCountry = $this->detectCountryFromIP($request->ip());
        
        if (!$countryId) {
            $country = $this->getDefaultCountry($ipCountry);
            $countryId = $country?->id;
        }
        
        if (!$languageId) {
            $language = $this->getDefaultLanguage();
            $languageId = $language?->id;
        }
        
        // ========================================
        // 2. التحقق من صحة البيانات
        // ========================================
        
        $country = Country::where('id', $countryId)->active()->first();
        $language = Language::where('id', $languageId)->active()->first();
        
        if (!$country) {
            $country = $this->getDefaultCountry();
            $countryId = $country?->id;
        }
        
        if (!$language) {
            $language = $this->getDefaultLanguage();
            $languageId = $language?->id;
        }
        
        // ========================================
        // 3. البحث عن أو إنشاء Locale مناسب
        // ========================================
        
        $locale = null;
        if ($countryId && $languageId) {
            $locale = Locale::where('country_id', $countryId)
                           ->where('language_id', $languageId)
                           ->active()
                           ->first();
            
            // إذا لم يوجد locale محدد، ابحث عن الافتراضي للبلد
            if (!$locale) {
                $locale = Locale::where('country_id', $countryId)
                               ->where('is_default', true)
                               ->active()
                               ->first();
                
                if ($locale) {
                    $languageId = $locale->language_id;
                }
            }
        }
        
        // إذا كان هناك locale ID من الـ request، تحقق منه
        if ($localeId) {
            $requestedLocale = Locale::where('id', $localeId)->active()->first();
            if ($requestedLocale) {
                $locale = $requestedLocale;
                $countryId = $locale->country_id;
                $languageId = $locale->language_id;
            }
        }
        
        // تحديد locale_id النهائي
        $localeId = $locale?->id;
        
        // ========================================
        // 4. حفظ البيانات في الـ Request
        // ========================================
        
        $request->attributes->set('country_id', $countryId);
        $request->attributes->set('language_id', $languageId);
        $request->attributes->set('locale_id', $localeId);
        $request->attributes->set('ip_country', $ipCountry);
        $request->attributes->set('user_has_saved_locale', $user && $savedLocaleIds !== null); // ✅ الآن المتغير معرّف
        
        // تحديد locale التطبيق
        if ($locale && $locale->language) {
            app()->setLocale($locale->language->code);
        }
        
        // Log للـ debugging (يمكن إزالته في الإنتاج)
        Log::debug('API Locale Middleware', [
            'user_id' => $user?->id,
            'has_saved_locale' => $user && $savedLocaleIds !== null,
            'saved_locale_ids' => $savedLocaleIds,
            'final_country_id' => $countryId,
            'final_language_id' => $languageId,
            'final_locale_id' => $localeId,
            'ip_country' => $ipCountry,
            'app_locale' => app()->getLocale(),
        ]);
        
        return $next($request);
    }
    
    /**
     * Detect country from IP address
     */
    private function detectCountryFromIP($ip)
    {
        try {
            if ($ip === '127.0.0.1' || $ip === '::1') {
                return null;
            }
            
            // يمكن تطبيق خدمة GeoIP حقيقية هنا
            // مثال: MaxMind, ipapi.com, etc.
            
            return null;
        } catch (\Exception $e) {
            Log::warning('IP country detection failed', [
                'ip' => $ip,
                'error' => $e->getMessage(),
            ]);
            return null;
        }
    }
    
    /**
     * Get default country
     */
    private function getDefaultCountry($ipCountry = null)
    {
        // محاولة البحث بالـ IP أولاً
        if ($ipCountry) {
            $country = Country::where('iso2', $ipCountry)
                             ->orWhere('code', $ipCountry)
                             ->active()
                             ->first();
            if ($country) {
                return $country;
            }
        }
        
        // البحث عن البلد الافتراضي
        return Country::where('is_default', true)->active()->first() 
            ?? Country::active()->first();
    }
    
    /**
     * Get default language
     */
    private function getDefaultLanguage()
    {
        return Language::where('is_default', true)->active()->first() 
            ?? Language::active()->first();
    }
}