<?php

namespace App\Http\Controllers\Api;

use App\Models\Country;
use App\Models\Language;
use App\Models\Locale;
use App\Models\UserLocale;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Dedoc\Scramble\Attributes\HeaderParameter;
use Dedoc\Scramble\Attributes\BodyParameter;

/**
 * @tags 03. User Locale Settings
 */
class UserLocaleController extends ApiController
{
    /**
     * Get User's Current Locale Settings
     * 
     * Get user's current country and language preferences with available options.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function getCurrentSettings(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $user->load(['country', 'currentCountry', 'language']);

        // جلب الـ Locale الحالي من الـ Request (الآن مبني على إعدادات اليوزر)
        $currentLocale = $this->getCurrentLocale($request);
        
        // جلب الـ Locale المحفوظ للمستخدم
        $savedUserLocale = UserLocale::getUserLocaleWithRelations($user->id);
        $savedLocale = $savedUserLocale?->locale;

        // جلب الدول المتاحة
        $availableCountries = Country::active()
            ->ordered()
            ->with(['translations' => function ($q) use ($currentLocale) {
                $q->where('language_id', $currentLocale['language_id']);
            }])
            ->get()
            ->map(function ($country) use ($currentLocale) {
                return [
                    'id' => $country->id,
                    'name' => $country->getName($currentLocale['language_id']),
                    'code' => $country->code,
                    'flag_emoji' => $country->flag_emoji,
                    'is_default' => $country->is_default,
                ];
            });

        // تحديد مصدر الإعدادات الحالية
        $settingsSource = 'request_params'; // افتراضي
        if ($request->attributes->get('user_has_saved_locale')) {
            $settingsSource = 'saved_preferences';
        } elseif ($user->current_country_id) {
            $settingsSource = 'user_defaults';
        }

        return $this->successResponse([
            'current_settings' => [
                'country_id' => $currentLocale['country_id'],
                'language_id' => $currentLocale['language_id'],
                'locale_id' => $currentLocale['locale_id'],
                'source' => $settingsSource, // 🔥 NEW: يوضح مصدر الإعدادات
            ],
            'user_defaults' => [
                'default_country_id' => $user->country_id,
                'default_language_id' => $user->language_id,
                'current_country_id' => $user->current_country_id,
            ],
            'saved_locale' => $savedLocale ? [
                'id' => $savedLocale->id,
                'country_id' => $savedLocale->country_id,
                'language_id' => $savedLocale->language_id,
                'country_name' => $savedLocale->country ? $savedLocale->country->getName($currentLocale['language_id']) : 'Unknown Country',
                'language_name' => $savedLocale->language ? $savedLocale->language->native_name : 'Unknown Language',
                'locale_code' => $savedLocale->code ?? 'unknown',
            ] : null,
            'available_countries' => $availableCountries,
            'status' => [
                'has_saved_preferences' => !is_null($savedLocale),
                'using_saved_preferences' => $settingsSource === 'saved_preferences',
                'middleware_applied_user_settings' => $request->attributes->get('user_has_saved_locale', false),
            ],
        ], 'Current locale settings retrieved successfully');
    }

    /**
     * Change User Country
     * 
     * Change user's country and get default language for that country.
     * The system will automatically select the default language for the chosen country.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('country_id', 'Country ID to switch to', required: true, type: 'integer', example: 1)]
    #[BodyParameter('save_as_default', 'Save this country as user default', required: false, type: 'boolean', example: true)]
    public function changeCountry(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'country_id' => 'required|exists:countries,id',
            'save_as_default' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $countryId = $request->country_id;
        $saveAsDefault = $request->boolean('save_as_default', false);

        // التحقق من أن البلد فعال
        $country = Country::where('id', $countryId)->active()->first();
        if (!$country) {
            return $this->errorResponse('Country not found or inactive', 404);
        }

        // الحصول على اللغة الافتراضية للبلد
        $defaultLocale = Locale::where('country_id', $countryId)
                              ->where('is_default', true)
                              ->active()
                              ->with(['language', 'country'])
                              ->first();

        if (!$defaultLocale) {
            // إذا لم توجد locale افتراضية، أخذ أول locale متاحة
            $defaultLocale = Locale::where('country_id', $countryId)
                                  ->active()
                                  ->with(['language', 'country'])
                                  ->first();
        }

        if (!$defaultLocale) {
            return $this->errorResponse('No available language for this country', 400);
        }

        return $this->executeTransaction(function () use ($user, $countryId, $defaultLocale, $saveAsDefault) {
            $updateData = [
                'current_country_id' => $countryId,
            ];

            // إذا المستخدم اختار حفظ كإعداد افتراضي
            if ($saveAsDefault) {
                $updateData['country_id'] = $countryId;
                $updateData['language_id'] = $defaultLocale->language_id;
            }

            // تحديث بيانات المستخدم
            $user->update($updateData);

            // حفظ الـ Locale الجديد للمستخدم
            UserLocale::setUserLocale($user->id, $defaultLocale->id);

            $this->logActivity($user, 'country_changed', [
                'country_id' => $countryId,
                'default_language_id' => $defaultLocale->language_id,
                'locale_id' => $defaultLocale->id,
                'save_as_default' => $saveAsDefault,
            ]);

            // جلب اللغات المتاحة للبلد الجديد
            $availableLanguages = Locale::where('country_id', $countryId)
                                       ->active()
                                       ->with(['language'])
                                       ->ordered()
                                       ->get()
                                       ->map(function ($locale) {
                                           return [
                                               'locale_id' => $locale->id,
                                               'language_id' => $locale->language->id,
                                               'language_name' => $locale->language->name,
                                               'language_native_name' => $locale->language->native_name,
                                               'language_code' => $locale->language->code,
                                               'is_default' => $locale->is_default,
                                           ];
                                       });

            return [
                'country_changed' => true,
                'new_settings' => [
                    'country_id' => $countryId,
                    'language_id' => $defaultLocale->language_id,
                    'locale_id' => $defaultLocale->id,
                ],
                'country' => [
                    'id' => $defaultLocale->country->id,
                    'name' => $defaultLocale->country->name,
                    'code' => $defaultLocale->country->code,
                    'flag_emoji' => $defaultLocale->country->flag_emoji,
                ],
                'selected_language' => [
                    'id' => $defaultLocale->language->id,
                    'name' => $defaultLocale->language->name,
                    'native_name' => $defaultLocale->language->native_name,
                    'code' => $defaultLocale->language->code,
                    'is_default' => true,
                ],
                'available_languages' => $availableLanguages,
                'saved_as_default' => $saveAsDefault,
            ];
        }, 'Country changed successfully');
    }

    /**
     * Change Language for Current Country
     * 
     * Change language while keeping the same country.
     * This will create/update the user's locale preference.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('language_id', 'Language ID to switch to', required: true, type: 'integer', example: 2)]
    #[BodyParameter('country_id', 'Country ID (optional, uses current if not provided)', required: false, type: 'integer', example: 1)]
    #[BodyParameter('save_as_default', 'Save this language as user default', required: false, type: 'boolean', example: true)]
    public function changeLanguage(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'language_id' => 'required|exists:languages,id',
            'country_id' => 'nullable|exists:countries,id',
            'save_as_default' => 'nullable|boolean',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $languageId = $request->language_id;
        $countryId = $request->country_id ?? $this->getCurrentLocale($request)['country_id'];
        $saveAsDefault = $request->boolean('save_as_default', false);

        // التحقق من أن اللغة فعالة
        $language = Language::where('id', $languageId)->active()->first();
        if (!$language) {
            return $this->errorResponse('Language not found or inactive', 404);
        }

        // التحقق من وجود Locale لهذا المزيج
        $locale = Locale::where('country_id', $countryId)
                       ->where('language_id', $languageId)
                       ->active()
                       ->with(['language', 'country'])
                       ->first();

        if (!$locale) {
            return $this->errorResponse('This language is not available for the selected country', 400);
        }

        return $this->executeTransaction(function () use ($user, $languageId, $countryId, $locale, $saveAsDefault) {
            $updateData = [];

            // إذا المستخدم اختار حفظ كإعداد افتراضي
            if ($saveAsDefault) {
                $updateData['language_id'] = $languageId;
                $updateData['country_id'] = $countryId;
            }

            // تحديث البلد الحالي إذا لزم الأمر
            if ($user->current_country_id !== $countryId) {
                $updateData['current_country_id'] = $countryId;
            }

            if (!empty($updateData)) {
                $user->update($updateData);
            }

            // حفظ الـ Locale الجديد للمستخدم
            UserLocale::setUserLocale($user->id, $locale->id);

            $this->logActivity($user, 'language_changed', [
                'language_id' => $languageId,
                'country_id' => $countryId,
                'locale_id' => $locale->id,
                'save_as_default' => $saveAsDefault,
            ]);

            return [
                'language_changed' => true,
                'new_settings' => [
                    'country_id' => $countryId,
                    'language_id' => $languageId,
                    'locale_id' => $locale->id,
                ],
                'country' => [
                    'id' => $locale->country->id,
                    'name' => $locale->country->getName($languageId),
                    'code' => $locale->country->code,
                    'flag_emoji' => $locale->country->flag_emoji,
                ],
                'language' => [
                    'id' => $locale->language->id,
                    'name' => $locale->language->name,
                    'native_name' => $locale->language->native_name,
                    'code' => $locale->language->code,
                ],
                'locale' => [
                    'id' => $locale->id,
                    'code' => $locale->code,
                    'name' => $locale->name,
                ],
                'saved_as_default' => $saveAsDefault,
            ];
        }, 'Language changed successfully');
    }

    /**
     * Get Available Languages for Country
     * 
     * Get all available languages for a specific country.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function getCountryLanguages(Request $request, $countryId): JsonResponse
    {
        $country = Country::where('id', $countryId)->active()->first();
        
        if (!$country) {
            return $this->errorResponse('Country not found', 404);
        }

        $currentLocale = $this->getCurrentLocale($request);

        // جلب اللغات المتاحة للبلد
        $availableLanguages = Locale::where('country_id', $countryId)
                                   ->active()
                                   ->with(['language'])
                                   ->ordered()
                                   ->get()
                                   ->map(function ($locale) {
                                       return [
                                           'locale_id' => $locale->id,
                                           'language_id' => $locale->language->id,
                                           'language_name' => $locale->language->name,
                                           'language_native_name' => $locale->language->native_name,
                                           'language_code' => $locale->language->code,
                                           'is_default' => $locale->is_default,
                                       ];
                                   });

        return $this->successResponse([
            'country' => [
                'id' => $country->id,
                'name' => $country->getName($currentLocale['language_id']),
                'code' => $country->code,
                'flag_emoji' => $country->flag_emoji,
            ],
            'available_languages' => $availableLanguages,
            'total_languages' => $availableLanguages->count(),
        ], 'Available languages for country retrieved successfully');
    }

    /**
     * Reset to Default Settings
     * 
     * Reset user's locale to their default settings or system defaults.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('reset_type', 'Type of reset: user_default or system_default', required: false, type: 'string', example: 'user_default')]
    public function resetToDefaults(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'reset_type' => 'nullable|in:user_default,system_default',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $resetType = $request->input('reset_type', 'user_default');

        return $this->executeTransaction(function () use ($user, $resetType) {
            if ($resetType === 'system_default') {
                // استخدام إعدادات النظام الافتراضية
                $defaultCountry = Country::where('is_default', true)->active()->first();
                $defaultLanguage = Language::where('is_default', true)->active()->first();
                
                $countryId = $defaultCountry?->id;
                $languageId = $defaultLanguage?->id;
            } else {
                // استخدام إعدادات المستخدم الافتراضية
                $countryId = $user->country_id;
                $languageId = $user->language_id;
            }

            // البحث عن Locale مناسب
            $locale = Locale::where('country_id', $countryId)
                           ->where('language_id', $languageId)
                           ->active()
                           ->first();

            if (!$locale) {
                // البحث عن locale افتراضي للبلد
                $locale = Locale::where('country_id', $countryId)
                               ->where('is_default', true)
                               ->active()
                               ->first();
            }

            if (!$locale) {
                return $this->errorResponse('Unable to find suitable locale for reset', 400);
            }

            // تحديث بيانات المستخدم
            $user->update([
                'current_country_id' => $locale->country_id,
            ]);

            // حفظ الـ Locale
            UserLocale::setUserLocale($user->id, $locale->id);

            $this->logActivity($user, 'locale_reset', [
                'reset_type' => $resetType,
                'country_id' => $locale->country_id,
                'language_id' => $locale->language_id,
                'locale_id' => $locale->id,
            ]);

            return [
                'reset_completed' => true,
                'reset_type' => $resetType,
                'new_settings' => [
                    'country_id' => $locale->country_id,
                    'language_id' => $locale->language_id,
                    'locale_id' => $locale->id,
                ],
            ];
        }, 'Locale settings reset successfully');
    }
}