<?php

namespace App\Http\Controllers\Api;

use App\Models\Notification;
use App\Models\NotificationDelivery;
use App\Models\UserDevice;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Dedoc\Scramble\Attributes\HeaderParameter;
use Dedoc\Scramble\Attributes\BodyParameter;

/**
 * @tags 09. Notifications
 */
class NotificationApiController extends ApiController
{
    /**
     * Get User Notifications
     * 
     * Get paginated list of user notifications with filtering options.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function index(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:50',
            'status' => 'nullable|in:all,read,unread',
            'type' => 'nullable|in:system,promotional,offer,store,brand,user',
            'priority' => 'nullable|integer|min:1|max:10',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $perPage = $request->input('per_page', 20);
        $status = $request->input('status', 'all');
        $type = $request->input('type');
        $priority = $request->input('priority');

        // Get user's notification deliveries
        $query = NotificationDelivery::where('user_id', $user->id)
            ->with(['notification.targets'])
            ->latest();

        // Apply filters
        if ($status !== 'all') {
            if ($status === 'read') {
                $query->where('status', 'read');
            } else {
                $query->where('status', '!=', 'read');
            }
        }

        if ($type) {
            $query->whereHas('notification', function ($q) use ($type) {
                $q->where('type', $type);
            });
        }

        if ($priority) {
            $query->whereHas('notification', function ($q) use ($priority) {
                $q->where('priority', '>=', $priority);
            });
        }

        $deliveries = $query->paginate($perPage);

        $notifications = $deliveries->getCollection()->map(function ($delivery) {
            $notification = $delivery->notification;
            
            return [
                'id' => $notification->id,
                'type' => $notification->type,
                'title' => $notification->title,
                'message' => $notification->message,
                'data' => $notification->data,
                'priority' => $notification->priority,
                'is_read' => $delivery->status === 'read',
                'read_at' => $delivery->read_at,
                'delivered_at' => $delivery->created_at,
                'created_at' => $notification->created_at,
            ];
        });

        // Get counts
        $totalCount = NotificationDelivery::where('user_id', $user->id)->count();
        $unreadCount = NotificationDelivery::where('user_id', $user->id)
            ->where('status', '!=', 'read')
            ->count();

        return $this->successResponse([
            'notifications' => $notifications,
            'pagination' => [
                'current_page' => $deliveries->currentPage(),
                'per_page' => $deliveries->perPage(),
                'total' => $deliveries->total(),
                'last_page' => $deliveries->lastPage(),
                'has_more' => $deliveries->hasMorePages(),
            ],
            'counts' => [
                'total' => $totalCount,
                'unread' => $unreadCount,
                'read' => $totalCount - $unreadCount,
            ],
        ], 'Notifications retrieved successfully');
    }

    /**
     * Get Notification Details
     * 
     * Get detailed information about a specific notification.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function show(Request $request, $notificationId): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $delivery = NotificationDelivery::where('user_id', $user->id)
            ->whereHas('notification', function ($q) use ($notificationId) {
                $q->where('id', $notificationId);
            })
            ->with(['notification.targets'])
            ->first();

        if (!$delivery) {
            return $this->errorResponse('Notification not found', 404);
        }

        $notification = $delivery->notification;

        return $this->successResponse([
            'notification' => [
                'id' => $notification->id,
                'type' => $notification->type,
                'title' => $notification->title,
                'message' => $notification->message,
                'data' => $notification->data,
                'priority' => $notification->priority,
                'channels' => $notification->channels,
                'is_read' => $delivery->status === 'read',
                'read_at' => $delivery->read_at,
                'delivered_at' => $delivery->created_at,
                'created_at' => $notification->created_at,
                'targets_count' => $notification->targets->count(),
            ],
        ], 'Notification details retrieved successfully');
    }

    /**
     * Mark Notification as Read
     * 
     * Mark a specific notification as read for the current user.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function markAsRead(Request $request, $notificationId): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $delivery = NotificationDelivery::where('user_id', $user->id)
            ->whereHas('notification', function ($q) use ($notificationId) {
                $q->where('id', $notificationId);
            })
            ->first();

        if (!$delivery) {
            return $this->errorResponse('Notification not found', 404);
        }

        if ($delivery->status === 'read') {
            return $this->successResponse([
                'already_read' => true,
                'read_at' => $delivery->read_at,
            ], 'Notification was already marked as read');
        }

        $delivery->markAsRead();

        $this->logActivity($user, 'notification_read', [
            'notification_id' => $notificationId,
            'read_at' => now(),
        ]);

        return $this->successResponse([
            'marked_as_read' => true,
            'read_at' => $delivery->read_at,
        ], 'Notification marked as read successfully');
    }

    /**
     * Mark Notification as Unread
     * 
     * Mark a specific notification as unread for the current user.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function markAsUnread(Request $request, $notificationId): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $delivery = NotificationDelivery::where('user_id', $user->id)
            ->whereHas('notification', function ($q) use ($notificationId) {
                $q->where('id', $notificationId);
            })
            ->first();

        if (!$delivery) {
            return $this->errorResponse('Notification not found', 404);
        }

        if ($delivery->status !== 'read') {
            return $this->successResponse([
                'already_unread' => true,
            ], 'Notification was already unread');
        }

        $delivery->update([
            'status' => 'sent',
            'read_at' => null,
        ]);

        $this->logActivity($user, 'notification_unread', [
            'notification_id' => $notificationId,
        ]);

        return $this->successResponse([
            'marked_as_unread' => true,
        ], 'Notification marked as unread successfully');
    }

    /**
     * Mark All Notifications as Read
     * 
     * Mark all notifications as read for the current user.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function markAllAsRead(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $unreadDeliveries = NotificationDelivery::where('user_id', $user->id)
            ->where('status', '!=', 'read')
            ->get();

        $markedCount = 0;
        foreach ($unreadDeliveries as $delivery) {
            $delivery->markAsRead();
            $markedCount++;
        }

        $this->logActivity($user, 'all_notifications_read', [
            'marked_count' => $markedCount,
        ]);

        return $this->successResponse([
            'marked_count' => $markedCount,
            'all_read' => true,
        ], "Marked {$markedCount} notifications as read");
    }

    /**
     * Get Notification Counts
     * 
     * Get notification counts (total, read, unread) for the current user.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function getCounts(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $totalCount = NotificationDelivery::where('user_id', $user->id)->count();
        $unreadCount = NotificationDelivery::where('user_id', $user->id)
            ->where('status', '!=', 'read')
            ->count();
        $readCount = $totalCount - $unreadCount;

        // Get counts by type
        $typecounts = NotificationDelivery::where('user_id', $user->id)
            ->join('notifications', 'notification_deliveries.notification_id', '=', 'notifications.id')
            ->selectRaw('notifications.type, count(*) as count')
            ->groupBy('notifications.type')
            ->pluck('count', 'type')
            ->toArray();

        // Get priority counts
        $priorityCounts = NotificationDelivery::where('user_id', $user->id)
            ->join('notifications', 'notification_deliveries.notification_id', '=', 'notifications.id')
            ->selectRaw('
                CASE 
                    WHEN notifications.priority >= 8 THEN "high"
                    WHEN notifications.priority >= 5 THEN "medium"
                    ELSE "low"
                END as priority_level,
                count(*) as count
            ')
            ->groupBy('priority_level')
            ->pluck('count', 'priority_level')
            ->toArray();

        return $this->successResponse([
            'total' => $totalCount,
            'unread' => $unreadCount,
            'read' => $readCount,
            'by_type' => $typeounts,
            'by_priority' => $priorityCounts,
            'unread_percentage' => $totalCount > 0 ? round(($unreadCount / $totalCount) * 100, 1) : 0,
        ], 'Notification counts retrieved successfully');
    }

    /**
     * Update FCM Token
     * 
     * Update or register FCM token for push notifications.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('fcm_token', 'Firebase Cloud Messaging token', required: true, type: 'string', example: 'fGxJ8B9XY7M:APA91bF...')]
    #[BodyParameter('device_id', 'Unique device identifier', required: true, type: 'string', example: 'device123')]
    #[BodyParameter('platform', 'Device platform', required: false, type: 'string', example: 'android')]
    public function updateFcmToken(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'fcm_token' => 'required|string|max:500',
            'device_id' => 'required|string|max:255',
            'platform' => 'nullable|in:android,ios,web',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        return $this->executeTransaction(function () use ($request, $user) {
            $fcmToken = $request->fcm_token;
            $deviceId = $request->device_id;
            $platform = $request->input('platform', 'unknown');

            // Check if token already exists for another user
            $existingDevice = UserDevice::where('fcm_token', $fcmToken)
                ->where('user_id', '!=', $user->id)
                ->first();

            if ($existingDevice) {
                // Remove token from other user's device
                $existingDevice->update(['fcm_token' => null]);
            }

            // Update or create device for current user
            $device = UserDevice::updateOrCreate(
                [
                    'user_id' => $user->id,
                    'device_id' => $deviceId,
                ],
                [
                    'fcm_token' => $fcmToken,
                    'platform' => $platform,
                    'last_ip' => $request->ip(),
                    'ip_country' => $request->attributes->get('ip_country'),
                ]
            );

            $this->logActivity($user, 'fcm_token_updated', [
                'device_id' => $deviceId,
                'platform' => $platform,
                'token_length' => strlen($fcmToken),
            ]);

            return [
                'device' => [
                    'id' => $device->id,
                    'device_id' => $device->device_id,
                    'platform' => $device->platform,
                    'fcm_token_registered' => !empty($device->fcm_token),
                    'last_updated' => $device->updated_at,
                ],
            ];
        }, 'FCM token updated successfully');
    }

    /**
     * Get Notification Settings
     * 
     * Get user's notification preferences and settings.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function getSettings(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // في المستقبل يمكن إضافة جدول notification_settings
        // الآن نستخدم إعدادات افتراضية
        $defaultSettings = [
            'push_notifications' => true,
            'email_notifications' => true,
            'types' => [
                'system' => true,
                'promotional' => true,
                'offer' => true,
                'store' => true,
                'brand' => true,
            ],
            'priority_threshold' => 1, // أقل أولوية للإشعارات
            'quiet_hours' => [
                'enabled' => false,
                'start' => '22:00',
                'end' => '08:00',
            ],
        ];

        return $this->successResponse([
            'settings' => $defaultSettings,
            'devices_with_fcm' => $user->devices()->whereNotNull('fcm_token')->count(),
            'last_notification' => NotificationDelivery::where('user_id', $user->id)
                ->latest()
                ->with('notification')
                ->first()?->created_at,
        ], 'Notification settings retrieved successfully');
    }

    /**
     * Test Push Notification
     * 
     * Send a test push notification to user's devices (for debugging).
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function testPushNotification(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // Get user's devices with FCM tokens
        $devices = $user->devices()->whereNotNull('fcm_token')->get();

        if ($devices->isEmpty()) {
            return $this->errorResponse('No devices with FCM tokens found', 400);
        }

        // Create a test notification
        $testNotification = [
            'title' => 'Test Notification',
            'message' => 'This is a test notification sent at ' . now()->format('H:i:s'),
            'data' => [
                'test' => true,
                'sent_at' => now()->toISOString(),
            ],
        ];

        $sentToDevices = [];
        foreach ($devices as $device) {
            $sentToDevices[] = [
                'device_id' => $device->device_id,
                'platform' => $device->platform,
                'fcm_token_length' => strlen($device->fcm_token),
            ];
        }

        $this->logActivity($user, 'test_notification_sent', [
            'devices_count' => $devices->count(),
            'sent_at' => now(),
        ]);

        return $this->successResponse([
            'test_sent' => true,
            'notification' => $testNotification,
            'sent_to_devices' => $sentToDevices,
            'total_devices' => $devices->count(),
        ], 'Test notification sent successfully');
    }
}