<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\Country;
use App\Models\Language;
use App\Models\Locale;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

/**
 * @tags 01. Localization (Public)
 */
class LocalizationController extends Controller
{
    /**
     * Get all active countries with their available languages
     */
    public function countries(): JsonResponse
    {
        $countries = Country::active()
            ->ordered()
            ->with(['locales.language', 'translations'])
            ->get()
            ->map(function ($country) {
                return [
                    'id' => $country->id,
                    'name' => $country->name,
                    'code' => $country->code,
                    'iso_code' => $country->iso_code,
                    'iso2' => $country->iso2,
                    'dial_code' => $country->dial_code,
                    'currency' => $country->currency,
                    'currency_code' => $country->currency_code,
                    'currency_symbol' => $country->currency_symbol,
                    'flag_emoji' => $country->flag_emoji,
                    'flag_url' => $country->flag_url,
                    'is_default' => $country->is_default,
                    'available_languages' => $country->locales->map(function ($locale) use ($country) {
                        return [
                            'id' => $locale->language->id,
                            'name' => $locale->language->name,
                            'native_name' => $locale->language->native_name,
                            'code' => $locale->language->code,
                            'locale_id' => $locale->id,
                            'locale_code' => $locale->code,
                            'is_default' => $locale->is_default,
                            'country_name_localized' => $country->getName($locale->language->id),
                        ];
                    })->unique('id')->values(),
                    'translations' => $country->translations->map(function ($translation) {
                        return [
                            'language_id' => $translation->language_id,
                            'language_code' => $translation->language->code ?? null,
                            'name' => $translation->name,
                        ];
                    }),
                ];
            });

        return response()->json([
            'success' => true,
            'data' => $countries,
            'message' => 'Countries retrieved successfully'
        ]);
    }

    /**
     * Get available languages for a specific country
     */
    public function countryLanguages(Request $request, $countryId): JsonResponse
    {
        $country = Country::with(['locales.language', 'translations'])
            ->where('id', $countryId)
            ->active()
            ->first();

        if (!$country) {
            return response()->json([
                'success' => false,
                'message' => 'Country not found'
            ], 404);
        }

        $languages = $country->locales->map(function ($locale) use ($country) {
            return [
                'id' => $locale->language->id,
                'name' => $locale->language->name,
                'native_name' => $locale->language->native_name,
                'code' => $locale->language->code,
                'locale_id' => $locale->id,
                'locale_code' => $locale->code,
                'is_default' => $locale->is_default,
                'country_name_localized' => $country->getName($locale->language->id),
            ];
        })->unique('id')->values();

        return response()->json([
            'success' => true,
            'data' => [
                'country' => [
                    'id' => $country->id,
                    'name' => $country->name,
                    'code' => $country->code,
                    'flag_emoji' => $country->flag_emoji,
                    'translations' => $country->translations->map(function ($translation) {
                        return [
                            'language_id' => $translation->language_id,
                            'language_code' => $translation->language->code ?? null,
                            'name' => $translation->name,
                        ];
                    }),
                ],
                'languages' => $languages
            ],
            'message' => 'Country languages retrieved successfully'
        ]);
    }

    /**
     * Get all active countries and languages
     */
    public function languages(): JsonResponse
    {
        $languages = Language::active()
            ->ordered()
            ->get()
            ->map(function ($language) {
                return [
                    'id' => $language->id,
                    'name' => $language->name,
                    'native_name' => $language->native_name,
                    'code' => $language->code,
                    'is_default' => $language->is_default,
                ];
            });

        $countries = Country::active()
            ->ordered()
            ->with(['translations'])
            ->get()
            ->map(function ($country) {
                return [
                    'id' => $country->id,
                    'name' => $country->name,
                    'code' => $country->code,
                    'iso_code' => $country->iso_code,
                    'iso2' => $country->iso2,
                    'dial_code' => $country->dial_code,
                    'currency' => $country->currency,
                    'currency_code' => $country->currency_code,
                    'currency_symbol' => $country->currency_symbol,
                    'flag_emoji' => $country->flag_emoji,
                    'flag_url' => $country->flag_url,
                    'is_default' => $country->is_default,
                    'translations' => $country->translations->map(function ($translation) {
                        return [
                            'language_id' => $translation->language_id,
                            'language_code' => $translation->language->code ?? null,
                            'name' => $translation->name,
                        ];
                    }),
                ];
            });

        return response()->json([
            'success' => true,
            'data' => [
                'languages' => $languages,
                'countries' => $countries,
            ],
            'message' => 'Languages and countries retrieved successfully'
        ]);
    }

    /**
     * Get locale information by country and language
     */
    public function locale(Request $request): JsonResponse
    {
        $request->validate([
            'country_id' => 'required|exists:countries,id',
            'language_id' => 'required|exists:languages,id',
        ]);

        $locale = Locale::where('country_id', $request->country_id)
            ->where('language_id', $request->language_id)
            ->active()
            ->with(['country.translations', 'language'])
            ->first();

        if (!$locale) {
            return response()->json([
                'success' => false,
                'message' => 'Locale not found for this country and language combination'
            ], 404);
        }

        return response()->json([
            'success' => true,
            'data' => $locale->toLocalizedArray(),
            'message' => 'Locale retrieved successfully'
        ]);
    }

    /**
     * Get default country and language
     */
    public function defaults(): JsonResponse
    {
        $defaultCountry = Country::where('is_default', true)->active()->with('translations')->first();
        $defaultLanguage = Language::where('is_default', true)->active()->first();

        $defaultLocale = null;
        if ($defaultCountry && $defaultLanguage) {
            $defaultLocale = Locale::where('country_id', $defaultCountry->id)
                ->where('language_id', $defaultLanguage->id)
                ->active()
                ->first();
        }

        return response()->json([
            'success' => true,
            'data' => [
                'country' => $defaultCountry ? [
                    'id' => $defaultCountry->id,
                    'name' => $defaultCountry->name,
                    'code' => $defaultCountry->code,
                    'flag_emoji' => $defaultCountry->flag_emoji,
                    'translations' => $defaultCountry->translations->map(function ($translation) {
                        return [
                            'language_id' => $translation->language_id,
                            'language_code' => $translation->language->code ?? null,
                            'name' => $translation->name,
                        ];
                    }),
                ] : null,
                'language' => $defaultLanguage ? [
                    'id' => $defaultLanguage->id,
                    'name' => $defaultLanguage->name,
                    'native_name' => $defaultLanguage->native_name,
                    'code' => $defaultLanguage->code,
                ] : null,
                'locale' => $defaultLocale ? $defaultLocale->toLocalizedArray() : null,
            ],
            'message' => 'Default settings retrieved successfully'
        ]);
    }
}
