<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Subscription;
use App\Models\User;
use App\Models\Store;
use App\Models\Brand;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class SubscriptionController extends Controller
{
    /**
     * Display a listing of subscriptions
     */
    public function index(Request $request)
    {
        $query = Subscription::with(['user', 'subscribable']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->whereHas('user', function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%");
            });
        }

        // Filter by subscribable type (store, brand, category)
        if ($request->filled('subscribable_type')) {
            $type = $request->subscribable_type;
            $modelClass = match($type) {
                'store' => Store::class,
                'brand' => Brand::class,
                'category' => Category::class,
                default => null
            };

            if ($modelClass) {
                $query->where('subscribable_type', $modelClass);
            }
        }

        // Filter by subscription type (follow, notify, etc.)
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by specific user
        if ($request->filled('user_id')) {
            $query->where('user_id', $request->user_id);
        }

        // Filter by specific subscribable
        if ($request->filled('subscribable_id') && $request->filled('subscribable_type')) {
            $query->where('subscribable_id', $request->subscribable_id);
        }

        // Date range filter
        if ($request->filled('date_from')) {
            $query->whereDate('created_at', '>=', $request->date_from);
        }
        if ($request->filled('date_to')) {
            $query->whereDate('created_at', '<=', $request->date_to);
        }

        // Sort
        $sortBy = $request->get('sort', 'created_at');
        $sortOrder = $request->get('order', 'desc');
        $query->orderBy($sortBy, $sortOrder);

        // Get paginated results
        $subscriptions = $query->paginate(20)->withQueryString();

        // Get statistics
        $stats = $this->getStatistics();

        // Get filter options
        $users = User::select('id', 'name', 'email')
            ->whereHas('subscriptions')
            ->orderBy('name')
            ->limit(100)
            ->get();

        return view('admin.subscriptions.index', compact(
            'subscriptions',
            'stats',
            'users'
        ));
    }

    /**
     * Get subscription statistics
     */
    private function getStatistics()
    {
        return [
            'total' => Subscription::count(),
            'stores' => Subscription::where('subscribable_type', Store::class)->count(),
            'brands' => Subscription::where('subscribable_type', Brand::class)->count(),
            'categories' => Subscription::where('subscribable_type', Category::class)->count(),
            'followers' => Subscription::where('type', 'follow')->count(),
            'today' => Subscription::whereDate('created_at', today())->count(),
            'this_week' => Subscription::whereBetween('created_at', [now()->startOfWeek(), now()->endOfWeek()])->count(),
            'this_month' => Subscription::whereMonth('created_at', now()->month)->count(),
        ];
    }

    /**
     * Get store-specific statistics
     */
    public function storeStats(Request $request)
    {
        $stats = DB::table('subscriptions')
            ->select(
                'subscribable_id',
                DB::raw('COUNT(*) as followers_count')
            )
            ->where('subscribable_type', Store::class)
            ->where('type', 'follow')
            ->groupBy('subscribable_id')
            ->orderByDesc('followers_count')
            ->limit(10)
            ->get();

        $storeIds = $stats->pluck('subscribable_id');
        $stores = Store::whereIn('id', $storeIds)->get()->keyBy('id');

        $topStores = $stats->map(function ($stat) use ($stores) {
            $store = $stores->get($stat->subscribable_id);
            return [
                'id' => $stat->subscribable_id,
                'name' => $store ? $store->name : 'Unknown',
                'followers' => $stat->followers_count
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $topStores
        ]);
    }

    /**
     * Get brand-specific statistics
     */
    public function brandStats(Request $request)
    {
        $stats = DB::table('subscriptions')
            ->select(
                'subscribable_id',
                DB::raw('COUNT(*) as followers_count')
            )
            ->where('subscribable_type', Brand::class)
            ->where('type', 'follow')
            ->groupBy('subscribable_id')
            ->orderByDesc('followers_count')
            ->limit(10)
            ->get();

        $brandIds = $stats->pluck('subscribable_id');
        $brands = Brand::whereIn('id', $brandIds)->get()->keyBy('id');

        $topBrands = $stats->map(function ($stat) use ($brands) {
            $brand = $brands->get($stat->subscribable_id);
            return [
                'id' => $stat->subscribable_id,
                'name' => $brand ? $brand->name : 'Unknown',
                'followers' => $stat->followers_count
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $topBrands
        ]);
    }

    /**
     * Get category-specific statistics
     */
    public function categoryStats(Request $request)
    {
        $stats = DB::table('subscriptions')
            ->select(
                'subscribable_id',
                DB::raw('COUNT(*) as followers_count')
            )
            ->where('subscribable_type', Category::class)
            ->where('type', 'follow')
            ->groupBy('subscribable_id')
            ->orderByDesc('followers_count')
            ->limit(10)
            ->get();

        $categoryIds = $stats->pluck('subscribable_id');
        $categories = Category::whereIn('id', $categoryIds)->get()->keyBy('id');

        $topCategories = $stats->map(function ($stat) use ($categories) {
            $category = $categories->get($stat->subscribable_id);
            return [
                'id' => $stat->subscribable_id,
                'name' => $category ? $category->name : 'Unknown',
                'followers' => $stat->followers_count
            ];
        });

        return response()->json([
            'success' => true,
            'data' => $topCategories
        ]);
    }

    /**
     * Remove the specified subscription
     */
    public function destroy(Subscription $subscription)
    {
        try {
            $subscription->delete();

            flash_success(admin_trans('subscription_deleted_successfully'));
            return redirect()->route('admin.subscriptions.index');
        } catch (\Exception $e) {
            flash_error(admin_trans('error_deleting_subscription'));
            return back();
        }
    }

    /**
     * Handle bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete',
            'selected' => 'required|array',
            'selected.*' => 'exists:subscriptions,id'
        ]);

        try {
            $count = 0;

            switch ($request->action) {
                case 'delete':
                    $count = Subscription::whereIn('id', $request->selected)->delete();
                    break;
            }

            return response()->json([
                'success' => true,
                'message' => admin_trans('bulk_action_completed', ['count' => $count])
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_bulk_action')
            ], 500);
        }
    }

    /**
     * Export subscriptions data
     */
    public function export(Request $request)
    {
        // This can be implemented later with Excel/CSV export
        flash_info(admin_trans('export_feature_coming_soon'));
        return back();
    }
}
