<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Setting;
use App\Models\Country;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Artisan;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\File;

class SettingsController extends Controller
{
    /**
     * Display settings page
     */
    public function index()
    {
        $currentCountryId = current_filtered_country_id();
        
        $settings = [
            'general' => Setting::getGroup('general'),
            'email' => Setting::getGroup('email'),
            'social' => Setting::getGroup('social'),
            'seo' => Setting::getGroup('seo', $currentCountryId),
            'cache' => Setting::getGroup('cache'),
        ];

        $countries = Country::active()->ordered()->get();
        $currentCountry = $currentCountryId ? Country::find($currentCountryId) : null;

        return view('admin.settings.index', compact('settings', 'countries', 'currentCountry'));
    }

    /**
     * Update general settings
     */
    public function updateGeneral(Request $request)
    {
        $request->validate([
            'app_name' => 'required|string|max:255',
            'app_version' => 'required|string|max:50',
            'site_logo' => 'nullable|file|mimes:jpeg,png,jpg,gif,svg,webp|max:2048',
            'site_favicon' => 'nullable|file|mimes:ico,png,svg|max:1024',
        ]);

        try {
            // Handle logo upload
            if ($request->hasFile('site_logo')) {
                $logoFile = $request->file('site_logo');
                $logoName = 'logo_' . time() . '.' . $logoFile->getClientOriginalExtension();
                
                // Store in public/storage/settings
                $logoPath = $logoFile->storeAs('settings', $logoName, 'public');
                Setting::setValue('general', 'site_logo', '/storage/' . $logoPath);
            }

            // Handle favicon upload
            if ($request->hasFile('site_favicon')) {
                $faviconFile = $request->file('site_favicon');
                $faviconName = 'favicon_' . time() . '.' . $faviconFile->getClientOriginalExtension();
                
                // Store in public/storage/settings
                $faviconPath = $faviconFile->storeAs('settings', $faviconName, 'public');
                Setting::setValue('general', 'site_favicon', '/storage/' . $faviconPath);
            }

            // Update other settings
            Setting::setValue('general', 'app_name', $request->app_name);
            Setting::setValue('general', 'app_version', $request->app_version);

            // Update .env file for app name and version
            \Log::info('Updating .env APP_NAME to: ' . $request->app_name);
            $envUpdateResult = $this->updateEnvValue('APP_NAME', $request->app_name);
            \Log::info('ENV APP_NAME update result: ' . ($envUpdateResult ? 'success' : 'failed'));

            \Log::info('Updating .env API_VERSION to: ' . $request->app_version);
            $versionUpdateResult = $this->updateEnvValue('API_VERSION', $request->app_version);
            \Log::info('ENV API_VERSION update result: ' . ($versionUpdateResult ? 'success' : 'failed'));

            flash_success(admin_trans('general_settings_updated'));
        } catch (\Exception $e) {
            \Log::error('Settings update failed: ' . $e->getMessage());
            flash_error(admin_trans('error_updating_settings'));
        }

        return back();
    }

    /**
     * Update email settings
     */
    public function updateEmail(Request $request)
    {
        $request->validate([
            'smtp_host' => 'required|string',
            'smtp_port' => 'required|integer',
            'smtp_username' => 'required|string',
            'smtp_password' => 'nullable|string',
            'smtp_encryption' => 'required|in:tls,ssl',
            'from_name' => 'required|string',
            'from_address' => 'required|email',
        ]);

        try {
            // Update settings
            Setting::setGroup('email', [
                'smtp_host' => $request->smtp_host,
                'smtp_port' => $request->smtp_port,
                'smtp_username' => $request->smtp_username,
                'smtp_encryption' => $request->smtp_encryption,
                'from_name' => $request->from_name,
                'from_address' => $request->from_address,
            ]);

            // Update password only if provided
            if ($request->filled('smtp_password')) {
                Setting::setValue('email', 'smtp_password', $request->smtp_password);
            }

            // Update .env file
            $this->updateEnvValue('MAIL_MAILER', 'smtp');
            $this->updateEnvValue('MAIL_HOST', $request->smtp_host);
            $this->updateEnvValue('MAIL_PORT', $request->smtp_port);
            $this->updateEnvValue('MAIL_USERNAME', $request->smtp_username);
            $this->updateEnvValue('MAIL_ENCRYPTION', $request->smtp_encryption);
            $this->updateEnvValue('MAIL_FROM_ADDRESS', $request->from_address);
            $this->updateEnvValue('MAIL_FROM_NAME', $request->from_name);

            if ($request->filled('smtp_password')) {
                $this->updateEnvValue('MAIL_PASSWORD', $request->smtp_password);
            }

            flash_success(admin_trans('email_settings_updated'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_settings'));
        }

        return back();
    }

    /**
     * Update social media settings
     */
    public function updateSocial(Request $request)
    {
        $request->validate([
            'facebook_app_id' => 'nullable|string',
            'google_client_id' => 'nullable|string',
        ]);

        try {
            Setting::setGroup('social', $request->only([
                'facebook_app_id',
                'google_client_id',
            ]));

            flash_success(admin_trans('social_settings_updated'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_settings'));
        }

        return back();
    }

    /**
     * Update SEO settings (country-specific)
     */
    public function updateSeo(Request $request)
    {
        $request->validate([
            'meta_title_template' => 'required|string',
            'meta_description_template' => 'required|string',
            'meta_keywords' => 'nullable|string',
            'google_analytics_id' => 'nullable|string',
            'facebook_pixel_id' => 'nullable|string',
            'google_tag_manager' => 'nullable|string',
        ]);

        try {
            $countryId = current_filtered_country_id();
            
            Setting::setGroup('seo', $request->only([
                'meta_title_template',
                'meta_description_template',
                'meta_keywords',
                'google_analytics_id',
                'facebook_pixel_id',
                'google_tag_manager',
            ]), $countryId);

            $countryName = $countryId ? Country::find($countryId)->name : admin_trans('global');
            flash_success(admin_trans('seo_settings_updated') . ' (' . $countryName . ')');
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_settings'));
        }

        return back();
    }

    /**
     * Update cache settings
     */
    public function updateCache(Request $request)
    {
        $request->validate([
            'cache_enabled' => 'boolean',
            'cache_driver' => 'required|in:file,redis,memcached',
            'cache_lifetime' => 'required|integer|min:60',
        ]);

        try {
            Setting::setGroup('cache', [
                'cache_enabled' => $request->boolean('cache_enabled'),
                'cache_driver' => $request->cache_driver,
                'cache_lifetime' => $request->cache_lifetime,
            ]);

            // Update .env file
            $this->updateEnvValue('CACHE_DRIVER', $request->cache_driver);

            flash_success(admin_trans('cache_settings_updated'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_settings'));
        }

        return back();
    }

    /**
     * Clear all caches - FIXED: Returns JSON for AJAX
     */
    public function clearCache()
    {
        try {
            Artisan::call('cache:clear');
            Artisan::call('config:clear');
            Artisan::call('route:clear');
            Artisan::call('view:clear');

            // Return JSON instead of redirect
            return response()->json([
                'success' => true,
                'message' => admin_trans('cache_cleared_successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_clearing_cache') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Test email configuration - FIXED: Returns JSON for AJAX
     */
    public function testEmail(Request $request)
    {
        $request->validate([
            'test_email' => 'required|email',
        ]);

        try {
            // Send test email
            \Mail::raw('This is a test email from Qansah Admin Panel.', function ($message) use ($request) {
                $message->to($request->test_email)
                        ->subject('Test Email - Qansah');
            });
            
            // Return JSON instead of redirect
            return response()->json([
                'success' => true,
                'message' => admin_trans('test_email_sent_successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('test_email_failed') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Copy SEO settings between countries - FIXED: Returns JSON for AJAX
     */
    public function copySeoSettings(Request $request)
    {
        $request->validate([
            'from_country_id' => 'required|exists:countries,id',
            'to_country_id' => 'required|exists:countries,id|different:from_country_id',
        ]);

        try {
            Setting::copySeoSettings($request->from_country_id, $request->to_country_id);
            
            $fromCountry = Country::find($request->from_country_id);
            $toCountry = Country::find($request->to_country_id);
            
            // Return JSON instead of redirect
            return response()->json([
                'success' => true,
                'message' => "تم نسخ إعدادات الـ SEO من {$fromCountry->name} إلى {$toCountry->name}"
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_copying_settings') . ': ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Update .env file value
     */
    private function updateEnvValue($key, $value)
    {
        $envFile = base_path('.env');
        
        if (!File::exists($envFile)) {
            return false;
        }

        $envContent = File::get($envFile);
        
        // Clean the value - remove quotes and escape if needed
        $cleanValue = trim($value, '"\'');
        
        // Check if key exists
        if (preg_match("/^{$key}=/m", $envContent)) {
            // Update existing key - handle both quoted and unquoted values
            $envContent = preg_replace(
                "/^{$key}=.*/m",
                "{$key}=\"{$cleanValue}\"",
                $envContent
            );
        } else {
            // Add new key at the end
            $envContent .= "\n{$key}=\"{$cleanValue}\"";
        }

        File::put($envFile, $envContent);
        
        // Clear config cache to reload .env
        try {
            \Artisan::call('config:clear');
        } catch (\Exception $e) {
            // Silent fail if artisan command fails
        }
        
        return true;
    }
}
