<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Store;
use App\Models\Offer;
use App\Models\Country;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Subscription;
use App\Models\Notification;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * Display admin dashboard with real analytics
     */
    public function index()
    {
        // Get current country filter
        $currentCountry = current_filtered_country();
        
        // Main Statistics
        $stats = $this->getMainStatistics($currentCountry);
        
        // Charts Data
        $chartsData = $this->getChartsData($currentCountry);
        
        // Recent Activities
        $recentActivities = $this->getRecentActivities();
        
        // Top Performers
        $topPerformers = $this->getTopPerformers($currentCountry);
        
        // Quick Actions Data
        $quickActions = $this->getQuickActionsData($currentCountry);

        return view('admin.dashboard', compact(
            'stats',
            'chartsData', 
            'recentActivities',
            'topPerformers',
            'quickActions',
            'currentCountry'
        ));
    }

    /**
     * Get main dashboard statistics
     */
    private function getMainStatistics($currentCountry = null)
    {
        $now = now();
        $lastMonth = $now->copy()->subMonth();
        
        // Base queries with country filter if needed
        $userQuery = User::query();
        $storeQuery = Store::query();
        $offerQuery = Offer::query();
        
        if ($currentCountry) {
            $storeQuery->whereHas('countries', fn($q) => $q->where('countries.id', $currentCountry->id));
            $offerQuery->whereHas('countryStore.country', fn($q) => $q->where('countries.id', $currentCountry->id));
            $userQuery->where('current_country_id', $currentCountry->id);
        }

        // Current month stats
        $totalUsers = $userQuery->count();
        $totalStores = $storeQuery->count();
        $totalOffers = $offerQuery->count();
        $activeOffers = $offerQuery->where('status', 'active')->notExpired()->count();

        // Last month stats for comparison
        $lastMonthUsers = $userQuery->whereMonth('created_at', $lastMonth->month)->count();
        $lastMonthStores = $storeQuery->whereMonth('created_at', $lastMonth->month)->count();
        $lastMonthOffers = $offerQuery->whereMonth('created_at', $lastMonth->month)->count();
        $lastMonthActiveOffers = $offerQuery
            ->where('status', 'active')
            ->whereMonth('created_at', $lastMonth->month)
            ->count();

        // Calculate growth percentages
        $usersGrowth = $this->calculateGrowth($totalUsers, $lastMonthUsers);
        $storesGrowth = $this->calculateGrowth($totalStores, $lastMonthStores);
        $offersGrowth = $this->calculateGrowth($totalOffers, $lastMonthOffers);
        $activeOffersGrowth = $this->calculateGrowth($activeOffers, $lastMonthActiveOffers);

        // Additional metrics
        $totalSubscriptions = Subscription::count();
        $pendingNotifications = Notification::where('status', 'pending')->count();
        $expiringSoonOffers = $offerQuery
            ->where('status', 'active')
            ->whereBetween('expires_at', [now(), now()->addDays(7)])
            ->count();

        return [
            'total_users' => $totalUsers,
            'total_stores' => $totalStores,
            'total_offers' => $totalOffers,
            'active_offers' => $activeOffers,
            'total_subscriptions' => $totalSubscriptions,
            'pending_notifications' => $pendingNotifications,
            'expiring_soon_offers' => $expiringSoonOffers,
            
            // Growth data
            'users_growth' => $usersGrowth,
            'stores_growth' => $storesGrowth,
            'offers_growth' => $offersGrowth,
            'active_offers_growth' => $activeOffersGrowth,
        ];
    }

    /**
     * Get charts data for dashboard
     */
    private function getChartsData($currentCountry = null)
    {
        // Users registration chart (last 30 days)
        $usersChart = $this->getUsersChartData($currentCountry);
        
        // Offers creation chart (last 30 days)  
        $offersChart = $this->getOffersChartData($currentCountry);
        
        // Top categories by offers count
        $categoriesChart = $this->getCategoriesChartData($currentCountry);
        
        // Monthly revenue/activity chart
        $activityChart = $this->getActivityChartData($currentCountry);

        return [
            'users' => $usersChart,
            'offers' => $offersChart,
            'categories' => $categoriesChart,
            'activity' => $activityChart,
        ];
    }

    /**
     * Get recent activities
     */
    private function getRecentActivities()
    {
        $activities = [];

        // Recent users (last 5)
        $recentUsers = User::latest()->take(5)->get();
        foreach ($recentUsers as $user) {
            $activities[] = [
                'type' => 'user_registered',
                'title' => admin_trans('new_user'),
                'description' => str_replace(':name', $user->name, admin_trans('user_joined_platform')),
                'time' => $user->created_at->diffForHumans(),
                'icon' => 'user-plus',
                'color' => 'blue',
                'url' => route('admin.users.index'),
            ];
        }

        // Recent offers (last 5)
        $recentOffers = Offer::with(['countryStore.store.translations'])
            ->latest()
            ->take(5)
            ->get();
        
        foreach ($recentOffers as $offer) {
            $storeName = $offer->countryStore->store->translations->first()->name ?? admin_trans('store');
            $activities[] = [
                'type' => 'offer_created',
                'title' => admin_trans('new_offer'),
                'description' => str_replace(':store', $storeName, admin_trans('offer_added_in_store')),
                'time' => $offer->created_at->diffForHumans(),
                'icon' => 'tag',
                'color' => 'green',
                'url' => route('admin.offers.index'),
            ];
        }

        // Recent stores (last 3)
        $recentStores = Store::with(['translations'])
            ->latest()
            ->take(3)
            ->get();
            
        foreach ($recentStores as $store) {
            $storeName = $store->translations->first()->name ?? admin_trans('new_store');
            $activities[] = [
                'type' => 'store_created',
                'title' => admin_trans('new_store'),
                'description' => str_replace(':store', $storeName, admin_trans('store_added')),
                'time' => $store->created_at->diffForHumans(),
                'icon' => 'store',
                'color' => 'purple',
                'url' => route('admin.stores.index'),
            ];
        }

        // Sort by time and return top 10
        return collect($activities)
            ->sortByDesc(function ($activity) {
                // Convert time to timestamp for proper sorting
                return Carbon::parse($activity['time'])->timestamp;
            })
            ->take(10)
            ->values();
    }

    /**
     * Get top performers
     */
    private function getTopPerformers($currentCountry = null)
    {
        // Top stores by offers count
        $topStores = Store::select('stores.*')
            ->with(['translations', 'countryStores.stats'])
            ->withCount('offers')
            ->when($currentCountry, function ($query) use ($currentCountry) {
                $query->whereHas('countries', fn($q) => $q->where('countries.id', $currentCountry->id));
            })
            ->orderByDesc('offers_count')
            ->take(5)
            ->get();

        // Top brands by offers count
        $topBrands = Brand::select('brands.*')
            ->with(['translations'])
            ->withCount('offers')
            ->orderByDesc('offers_count')
            ->take(5)
            ->get();

        // Top categories by offers count
        $topCategories = Category::select('categories.*')
            ->with(['translations'])
            ->withCount('offers')
            ->orderByDesc('offers_count')
            ->take(5)
            ->get();

        // Most active users (by subscriptions)
        $topUsers = User::select('users.*')
            ->withCount('subscriptions')
            ->where('is_guest', false)
            ->orderByDesc('subscriptions_count')
            ->take(5)
            ->get();

        return [
            'stores' => $topStores,
            'brands' => $topBrands,
            'categories' => $topCategories,
            'users' => $topUsers,
        ];
    }

    /**
     * Get quick actions data
     */
    private function getQuickActionsData($currentCountry = null)
    {
        $actions = [];

        // Offers needing attention
        $draftOffers = Offer::where('status', 'draft')->count();
        if ($draftOffers > 0) {
            $actions[] = [
                'title' => admin_trans('draft_offers'),
                'description' => str_replace(':count', $draftOffers, admin_trans('offers_waiting_review')),
                'count' => $draftOffers,
                'url' => route('admin.offers.index', ['status' => 'draft']),
                'icon' => 'edit',
                'color' => 'warning',
            ];
        }

        // Expiring offers
        $expiringOffers = Offer::where('status', 'active')
            ->whereBetween('expires_at', [now(), now()->addDays(7)])
            ->count();
        if ($expiringOffers > 0) {
            $actions[] = [
                'title' => admin_trans('expiring_offers'),
                'description' => str_replace(':count', $expiringOffers, admin_trans('offers_expiring_soon')),
                'count' => $expiringOffers,
                'url' => route('admin.offers.index', ['expiry' => 'expiring_soon']),
                'icon' => 'clock',
                'color' => 'danger',
            ];
        }

        // Pending notifications
        $pendingNotifications = Notification::where('status', 'pending')->count();
        if ($pendingNotifications > 0) {
            $actions[] = [
                'title' => admin_trans('pending_notifications'),
                'description' => str_replace(':count', $pendingNotifications, admin_trans('notifications_waiting_send')),
                'count' => $pendingNotifications,
                'url' => route('admin.notifications.index', ['status' => 'pending']),
                'icon' => 'bell',
                'color' => 'info',
            ];
        }

        // Unverified users
        $unverifiedUsers = User::whereNull('email_verified_at')
            ->where('is_guest', false)
            ->count();
        if ($unverifiedUsers > 0) {
            $actions[] = [
                'title' => admin_trans('unverified_users'),
                'description' => str_replace(':count', $unverifiedUsers, admin_trans('users_need_verification')),
                'count' => $unverifiedUsers,
                'url' => route('admin.users.index', ['verified' => '0']),
                'icon' => 'user-x',
                'color' => 'warning',
            ];
        }

        return $actions;
    }

    /**
     * Helper methods for charts data
     */
    private function getUsersChartData($currentCountry = null)
    {
        $days = [];
        $data = [];
        
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $days[] = $date->format('M d');
            
            $query = User::whereDate('created_at', $date->format('Y-m-d'));
            if ($currentCountry) {
                $query->where('current_country_id', $currentCountry->id);
            }
            
            $data[] = $query->count();
        }

        return [
            'labels' => $days,
            'data' => $data,
            'total' => array_sum($data),
        ];
    }

    private function getOffersChartData($currentCountry = null)
    {
        $days = [];
        $data = [];
        
        for ($i = 29; $i >= 0; $i--) {
            $date = now()->subDays($i);
            $days[] = $date->format('M d');
            
            $query = Offer::whereDate('created_at', $date->format('Y-m-d'));
            if ($currentCountry) {
                $query->whereHas('countryStore.country', fn($q) => $q->where('countries.id', $currentCountry->id));
            }
            
            $data[] = $query->count();
        }

        return [
            'labels' => $days,
            'data' => $data,
            'total' => array_sum($data),
        ];
    }

    private function getCategoriesChartData($currentCountry = null)
    {
        $query = Category::select('categories.*')
            ->with('translations')
            ->withCount('offers')
            ->orderByDesc('offers_count')
            ->take(8);

        $categories = $query->get();
        
        $labels = [];
        $data = [];
        
        foreach ($categories as $category) {
            $labels[] = $category->translations->first()->name ?? admin_trans('category');
            $data[] = $category->offers_count;
        }

        return [
            'labels' => $labels,
            'data' => $data,
        ];
    }

    private function getActivityChartData($currentCountry = null)
    {
        $months = [];
        $usersData = [];
        $offersData = [];
        $storesData = [];
        
        for ($i = 11; $i >= 0; $i--) {
            $date = now()->subMonths($i);
            $months[] = $date->format('M Y');
            
            // Users count for this month
            $userQuery = User::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month);
            if ($currentCountry) {
                $userQuery->where('current_country_id', $currentCountry->id);
            }
            $usersData[] = $userQuery->count();
            
            // Offers count for this month
            $offerQuery = Offer::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month);
            if ($currentCountry) {
                $offerQuery->whereHas('countryStore.country', fn($q) => $q->where('countries.id', $currentCountry->id));
            }
            $offersData[] = $offerQuery->count();
            
            // Stores count for this month
            $storeQuery = Store::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month);
            if ($currentCountry) {
                $storeQuery->whereHas('countries', fn($q) => $q->where('countries.id', $currentCountry->id));
            }
            $storesData[] = $storeQuery->count();
        }

        return [
            'labels' => $months,
            'datasets' => [
                [
                    'label' => admin_trans('users'),
                    'data' => $usersData,
                    'color' => '#3B82F6',
                ],
                [
                    'label' => admin_trans('offers'),
                    'data' => $offersData,
                    'color' => '#16CC62',
                ],
                [
                    'label' => admin_trans('stores'),
                    'data' => $storesData,
                    'color' => '#7679FF',
                ],
            ],
        ];
    }

    /**
     * Calculate growth percentage
     */
    private function calculateGrowth($current, $previous)
    {
        if ($previous == 0) {
            return $current > 0 ? 100 : 0;
        }
        
        return round((($current - $previous) / $previous) * 100, 1);
    }

    /**
     * API endpoints for dashboard data
     */
    public function getStats(Request $request)
    {
        $currentCountry = current_filtered_country();
        $stats = $this->getMainStatistics($currentCountry);
        
        return response()->json([
            'success' => true,
            'data' => $stats
        ]);
    }

    public function getCharts(Request $request)
    {
        $currentCountry = current_filtered_country();
        $chartsData = $this->getChartsData($currentCountry);
        
        return response()->json([
            'success' => true,
            'data' => $chartsData
        ]);
    }

    public function getActivities(Request $request)
    {
        $activities = $this->getRecentActivities();
        
        return response()->json([
            'success' => true,
            'data' => $activities
        ]);
    }
}