<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Carousel;
use App\Models\CarouselSlide;
use App\Models\Country;
use App\Models\Locale;
use App\Models\Store;
use App\Models\Category;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Str;

class CarouselController extends Controller
{
    /**
     * Display a listing of carousels
     */
    public function index(Request $request)
    {
        $query = Carousel::with(['slides', 'country']);

        // Filter by country from session (country filter)
        $currentCountry = current_filtered_country();
        if ($currentCountry) {
            $query->where('country_id', $currentCountry->id);
        }

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;

            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('slug', 'like', "%{$search}%")
                    ->orWhere('type', 'like', "%{$search}%");
            });
        }

        // Filter by type
        if ($request->filled('type')) {
            $query->where('type', $request->type);
        }

        // Filter by status
        if ($request->filled('status')) {
            $isActive = $request->status === 'active';
            $query->where('is_active', $isActive);
        }

        // Get paginated results
        $carousels = $query->latest()->paginate(15)->withQueryString();

        // Get all countries for filter dropdown
        $countries = Country::orderBy('name')->get();

        return view('admin.carousels.index', compact('carousels', 'currentCountry', 'countries'));
    }

    /**
     * Store a newly created carousel
     */
    public function store(Request $request)
    {
        // Get current country from filter
        $currentCountry = current_filtered_country();

        if (!$currentCountry) {
            flash_error(admin_trans('please_select_country_first'));
            return back()->withInput();
        }

        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'slug' => 'required|string|max:255|unique:carousels,slug',
                'type' => 'required|string|max:100',
                'is_active' => 'nullable|boolean',
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            throw $e;
        }

        DB::beginTransaction();

        try {
            $data = [
                'name' => $request->name,
                'slug' => Str::slug($request->slug),
                'type' => $request->type,
                'country_id' => $currentCountry->id,
                'is_active' => $request->boolean('is_active', true),
            ];

            $carousel = Carousel::create($data);

            DB::commit();

            flash_success(admin_trans('carousel_created_successfully'));
            return redirect()->route('admin.carousels.show', $carousel);

        } catch (\Exception $e) {
            DB::rollBack();
            
            flash_error(admin_trans('error_creating_carousel') . ': ' . $e->getMessage());
            return back()->withInput();
        }
    }

    /**
     * Display the specified carousel with its slides
     */
    public function show(Carousel $carousel)
    {
        $carousel->load(['slides.translations.locale', 'slides.linkable', 'slides.media']);
        
        // Get current country for locale filtering
        $currentCountry = current_filtered_country();
        $availableLocales = collect();

        if ($currentCountry) {
            $availableLocales = Locale::where('country_id', $currentCountry->id)
                                    ->active()
                                    ->ordered()
                                    ->get();
        }

        // Get linkable options for slides
        $stores = Store::with('translations')->get();
        $categories = Category::with('translations')->get();

        // Get all countries for editing
        $countries = Country::orderBy('name')->get();

        return view('admin.carousels.show', compact(
            'carousel',
            'currentCountry',
            'availableLocales',
            'stores',
            'categories',
            'countries'
        ));
    }

    /**
     * Update the specified carousel
     */
    public function update(Request $request, Carousel $carousel)
    {
        try {
            $validated = $request->validate([
                'name' => 'required|string|max:255',
                'slug' => 'required|string|max:255|unique:carousels,slug,' . $carousel->id,
                'type' => 'required|string|max:100',
                'country_id' => 'required|exists:countries,id',
                'is_active' => 'nullable|boolean',
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            throw $e;
        }

        DB::beginTransaction();

        try {
            $carousel->update([
                'name' => $request->name,
                'slug' => Str::slug($request->slug),
                'type' => $request->type,
                'country_id' => $request->country_id,
                'is_active' => $request->boolean('is_active'),
            ]);

            DB::commit();

            flash_success(admin_trans('carousel_updated_successfully'));
            return redirect()->route('admin.carousels.show', $carousel);

        } catch (\Exception $e) {
            DB::rollBack();
            
            flash_error(admin_trans('error_updating_carousel'));
            return back()->withInput();
        }
    }

    /**
     * Remove the specified carousel
     */
    public function destroy(Carousel $carousel)
    {
        try {
            $carousel->delete();
            
            flash_success(admin_trans('carousel_deleted_successfully'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_deleting_carousel'));
        }

        return redirect()->route('admin.carousels.index');
    }

    /**
     * Toggle carousel status
     */
    public function toggleStatus(Carousel $carousel)
    {
        try {
            $carousel->update(['is_active' => !$carousel->is_active]);

            $message = $carousel->is_active
                ? admin_trans('carousel_activated_successfully')
                : admin_trans('carousel_deactivated_successfully');

            flash_success($message);
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_status'));
        }

        return back();
    }

    /**
     * Store a new slide for the carousel
     */
    public function storeSlide(Request $request, Carousel $carousel)
    {
        try {
            $validated = $request->validate([
                'translations' => 'required|array',
                'linkable_type' => 'nullable|string',
                'linkable_id' => 'nullable|integer',
                'image' => 'required|image|mimes:jpeg,png,jpg,webp|max:5120',
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            throw $e;
        }

        // Check if at least one translation exists
        $hasTranslation = false;
        
        foreach ($request->translations as $localeId => $translation) {
            if (!empty($translation['title'])) {
                $hasTranslation = true;
            }
        }

        if (!$hasTranslation) {
            return back()->withErrors([
                'translations' => admin_trans('at_least_one_slide_title_required')
            ])->withInput();
        }

        DB::beginTransaction();

        try {
            $slideData = [
                'linkable_type' => $request->linkable_type,
                'linkable_id' => $request->linkable_id,
            ];

            $slide = $carousel->slides()->create($slideData);

            // Store translations
            foreach ($request->translations as $localeId => $translation) {
                if (!empty($translation['title'])) {
                    $transData = [
                        'locale_id' => $localeId,
                        'title' => $translation['title'],
                        'description' => $translation['description'] ?? null,
                        'button_text' => $translation['button_text'] ?? null,
                        'button_link' => $translation['button_link'] ?? null,
                    ];
                    
                    $slide->translations()->create($transData);
                }
            }

            // Handle image upload
            if ($request->hasFile('image')) {
                $slide->addMediaFromRequest('image')
                    ->toMediaCollection('image');
            }

            DB::commit();

            flash_success(admin_trans('slide_created_successfully'));
            return redirect()->route('admin.carousels.show', $carousel);

        } catch (\Exception $e) {
            DB::rollBack();
            
            flash_error(admin_trans('error_creating_slide') . ': ' . $e->getMessage());
            return back()->withInput();
        }
    }

    /**
     * Get slide data for editing
     */
    public function editSlide(CarouselSlide $slide)
    {
        try {
            $slide->load(['translations.locale', 'linkable', 'media']);

            $data = [
                'id' => $slide->id,
                'carousel_id' => $slide->carousel_id,
                'linkable_type' => $slide->linkable_type,
                'linkable_id' => $slide->linkable_id,
                'translations' => $slide->translations->toArray(),
                'image_url' => $slide->getFirstMediaUrl('image'),
            ];

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_loading_slide')
            ], 500);
        }
    }

    /**
     * Update the specified slide
     */
    public function updateSlide(Request $request, CarouselSlide $slide)
    {
        try {
            $validated = $request->validate([
                'translations' => 'required|array',
                'linkable_type' => 'nullable|string',
                'linkable_id' => 'nullable|integer',
                'image' => 'nullable|image|mimes:jpeg,png,jpg,webp|max:5120',
            ]);

        } catch (\Illuminate\Validation\ValidationException $e) {
            throw $e;
        }

        // Check translations
        $hasTranslation = false;
        foreach ($request->translations as $translation) {
            if (!empty($translation['title'])) {
                $hasTranslation = true;
                break;
            }
        }

        if (!$hasTranslation) {
            return back()->withErrors([
                'translations' => admin_trans('at_least_one_slide_title_required')
            ])->withInput();
        }

        DB::beginTransaction();

        try {
            $slide->update([
                'linkable_type' => $request->linkable_type,
                'linkable_id' => $request->linkable_id,
            ]);

            // Clear and recreate translations
            $slide->translations()->delete();

            foreach ($request->translations as $localeId => $translation) {
                if (!empty($translation['title'])) {
                    $slide->translations()->create([
                        'locale_id' => $localeId,
                        'title' => $translation['title'],
                        'description' => $translation['description'] ?? null,
                        'button_text' => $translation['button_text'] ?? null,
                        'button_link' => $translation['button_link'] ?? null,
                    ]);
                }
            }

            // Handle image
            if ($request->hasFile('image')) {
                $slide->clearMediaCollection('image');
                $slide->addMediaFromRequest('image')
                    ->toMediaCollection('image');
            }

            DB::commit();

            flash_success(admin_trans('slide_updated_successfully'));
            return redirect()->route('admin.carousels.show', $slide->carousel);

        } catch (\Exception $e) {
            DB::rollBack();
            
            flash_error(admin_trans('error_updating_slide'));
            return back()->withInput();
        }
    }

    /**
     * Remove the specified slide
     */
    public function destroySlide(CarouselSlide $slide)
    {
        $carouselId = $slide->carousel_id;

        try {
            $slide->delete();
            
            flash_success(admin_trans('slide_deleted_successfully'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_deleting_slide'));
        }

        return redirect()->route('admin.carousels.show', $carouselId);
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $validated = $request->validate([
            'action' => 'required|in:delete,activate,deactivate',
            'selected' => 'required|array',
            'selected.*' => 'exists:carousels,id',
        ]);

        $carousels = Carousel::whereIn('id', $request->selected);
        $count = count($request->selected);

        try {
            switch ($request->action) {
                case 'delete':
                    $carousels->delete();
                    break;
                case 'activate':
                    $carousels->update(['is_active' => true]);
                    break;
                case 'deactivate':
                    $carousels->update(['is_active' => false]);
                    break;
            }

            return response()->json([
                'success' => true,
                'message' => "$count " . admin_trans('carousels') . " {$request->action}d " . admin_trans('successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_bulk_action')
            ]);
        }
    }
}
