<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\Brand;
use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Spatie\MediaLibrary\MediaCollections\Models\Media;

class BrandController extends Controller
{
    /**
     * Display a listing of brands
     */
    public function index(Request $request)
    {
        $query = Brand::with(['translations.language', 'media']);

        // Search functionality
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->whereHas('translations', function ($tq) use ($search) {
                    $tq->where('name', 'like', "%{$search}%")
                        ->orWhere('description', 'like', "%{$search}%");
                });
            });
        }

        // Filter by status
        if ($request->filled('status')) {
            $query->where('status', $request->status);
        }

        // Filter by featured
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->featured === '1');
        }

        // Get paginated results with ordering
        $brands = $query->ordered()->paginate(15)->withQueryString();

        // Get available languages for the multi-language form
        $languages = Language::active()->ordered()->get();

        return view('admin.brands.index', compact('brands', 'languages'));
    }

    /**
     * Store a newly created brand
     */
    public function store(Request $request)
    {
        // Custom validation
        $request->validate([
            'translations' => 'required|array',
            'order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,active',
            'is_featured' => 'nullable|boolean',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,webp,svg|max:2048',
        ]);

        // Check if at least one translation has a name
        $hasValidTranslation = false;
        foreach ($request->translations as $languageId => $translation) {
            if (!empty($translation['name'])) {
                $hasValidTranslation = true;
                break;
            }
        }

        if (!$hasValidTranslation) {
            return back()->withErrors([
                'translations' => 'At least one translation name is required.'
            ])->withInput();
        }

        // Validate individual translations that have names
        foreach ($request->translations as $languageId => $translation) {
            if (!empty($translation['name'])) {
                $validator = \Validator::make($translation, [
                    'name' => 'required|string|max:255',
                    'description' => 'nullable|string',
                ]);

                if ($validator->fails()) {
                    return back()->withErrors([
                        "translations.{$languageId}" => $validator->errors()->first()
                    ])->withInput();
                }
            }
        }

        DB::beginTransaction();
        try {
            $brand = Brand::create([
                'order' => $request->order ?? 0,
                'status' => $request->status,
                'is_featured' => $request->boolean('is_featured'),
            ]);

            // Store only translations with names
            foreach ($request->translations as $languageId => $translation) {
                if (!empty($translation['name'])) {
                    $brand->translations()->create([
                        'language_id' => $languageId,
                        'name' => $translation['name'],
                        'description' => $translation['description'] ?? null,
                    ]);
                }
            }

            // Handle logo upload
            if ($request->hasFile('logo')) {
                $brand->addMediaFromRequest('logo')
                    ->toMediaCollection('logo');
            }

            DB::commit();
            flash_success('Brand created successfully');
            return redirect()->route('admin.brands.index');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Brand creation failed: ' . $e->getMessage());
            flash_error('Error creating brand');
            return back()->withInput();
        }
    }


    public function update(Request $request, Brand $brand)
    {
        // Custom validation
        $request->validate([
            'translations' => 'required|array',
            'order' => 'nullable|integer|min:0',
            'status' => 'required|in:draft,active',
            'is_featured' => 'nullable|boolean',
            'logo' => 'nullable|image|mimes:jpeg,png,jpg,webp,svg|max:2048',
        ]);

        // Check if at least one translation has a name
        $hasValidTranslation = false;
        foreach ($request->translations as $languageId => $translation) {
            if (!empty($translation['name'])) {
                $hasValidTranslation = true;
                break;
            }
        }

        if (!$hasValidTranslation) {
            return back()->withErrors([
                'translations' => 'At least one translation name is required.'
            ])->withInput();
        }

        DB::beginTransaction();
        try {
            $brand->update([
                'order' => $request->order ?? 0,
                'status' => $request->status,
                'is_featured' => $request->boolean('is_featured'),
            ]);

            // Clear existing translations
            $brand->translations()->delete();

            // Store only translations with names
            foreach ($request->translations as $languageId => $translation) {
                if (!empty($translation['name'])) {
                    $brand->translations()->create([
                        'language_id' => $languageId,
                        'name' => $translation['name'],
                        'description' => $translation['description'] ?? null,
                    ]);
                }
            }

            // Handle logo upload
            if ($request->hasFile('logo')) {
                $brand->clearMediaCollection('logo');
                $brand->addMediaFromRequest('logo')
                    ->toMediaCollection('logo');
            }

            DB::commit();
            flash_success('Brand updated successfully');
            return redirect()->route('admin.brands.index');
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Brand update failed: ' . $e->getMessage());
            flash_error('Error updating brand');
            return back()->withInput();
        }
    }
    /**
     * Show the form for editing the specified brand (API endpoint)
     */
    public function edit(Brand $brand)
    {
        try {
            $brand->load(['translations.language', 'media']);

            $data = [
                'id' => $brand->id,
                'order' => $brand->order,
                'status' => $brand->status,
                'is_featured' => $brand->is_featured,
                'translations' => $brand->translations->toArray(),
                'logo_url' => $brand->getFirstMediaUrl('logo'),
            ];

            return response()->json([
                'success' => true,
                'data' => $data
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_loading_brand')
            ], 500);
        }
    }

    /**
     * Remove the specified brand
     */
    public function destroy(Brand $brand)
    {
        // Check if brand has related records
        if ($brand->offers()->count() > 0) {
            flash_error(admin_trans('cannot_delete_brand_with_relations'));
            return back();
        }

        try {
            $brand->delete();
            flash_success(admin_trans('brand_deleted_successfully'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_deleting_brand'));
        }

        return redirect()->route('admin.brands.index');
    }

    /**
     * Update brands order
     */
    public function updateOrder(Request $request)
    {
        $request->validate([
            'orders' => 'required|array',
            'orders.*' => 'required|integer|exists:brands,id',
        ]);

        DB::beginTransaction();
        try {
            foreach ($request->orders as $order => $id) {
                Brand::where('id', $id)->update(['order' => $order]);
            }

            DB::commit();
            return response()->json(['success' => true]);
        } catch (\Exception $e) {
            DB::rollBack();
            return response()->json(['success' => false], 500);
        }
    }

    /**
     * Toggle brand status
     */
    public function toggleStatus(Brand $brand)
    {
        try {
            $newStatus = $brand->status === 'active' ? 'draft' : 'active';
            $brand->update(['status' => $newStatus]);

            flash_success(admin_trans('brand_status_updated'));
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_status'));
        }

        return back();
    }

    /**
     * Toggle brand featured status
     */
    public function toggleFeatured(Brand $brand)
    {
        try {
            $brand->update(['is_featured' => !$brand->is_featured]);

            $message = $brand->is_featured
                ? admin_trans('brand_featured_successfully')
                : admin_trans('brand_unfeatured_successfully');

            flash_success($message);
        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_featured'));
        }

        return back();
    }

    /**
     * Bulk actions
     */
    public function bulkAction(Request $request)
    {
        $request->validate([
            'action' => 'required|in:delete,activate,deactivate,feature,unfeature',
            'selected' => 'required|array',
            'selected.*' => 'exists:brands,id',
        ]);

        $brands = Brand::whereIn('id', $request->selected);
        $count = count($request->selected);

        try {
            switch ($request->action) {
                case 'delete':
                    $brandsWithRelations = Brand::whereIn('id', $request->selected)
                        ->whereHas('offers')
                        ->count();

                    if ($brandsWithRelations > 0) {
                        return response()->json([
                            'success' => false,
                            'message' => admin_trans('some_brands_have_relations')
                        ]);
                    }

                    $brands->delete();
                    break;

                case 'activate':
                    $brands->update(['status' => 'active']);
                    break;

                case 'deactivate':
                    $brands->update(['status' => 'draft']);
                    break;

                case 'feature':
                    $brands->update(['is_featured' => true]);
                    break;

                case 'unfeature':
                    $brands->update(['is_featured' => false]);
                    break;
            }

            return response()->json([
                'success' => true,
                'message' => "$count " . admin_trans('brands') . " {$request->action}d " . admin_trans('successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_bulk_action')
            ]);
        }
    }

    /**
     * Remove media from brand
     */
    public function removeMedia(Brand $brand, $collection)
    {
        try {
            $brand->clearMediaCollection($collection);

            return response()->json([
                'success' => true,
                'message' => admin_trans('media_removed_successfully')
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => admin_trans('error_removing_media')
            ], 500);
        }
    }
}
