<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;

class AuthController extends Controller
{
    /**
     * Show login form
     */
    public function showLogin()
    {
        // Redirect if already logged in
        if (Auth::check() && Auth::user()->role === 'admin') {
            return redirect()->route('admin.dashboard');
        }

        return view('admin.auth.login');
    }

    /**
     * Handle login request
     */
    public function login(Request $request)
    {
        // Validate input
        $validator = Validator::make($request->all(), [
            'email' => 'required|email',
            'password' => 'required|string|min:6',
        ]);

        if ($validator->fails()) {
            return back()
                ->withErrors($validator)
                ->withInput($request->only('email', 'remember'));
        }

        // Attempt login
        $credentials = $request->only('email', 'password');
        $remember = $request->boolean('remember');

        if (Auth::attempt($credentials, $remember)) {
            // Check if user is admin
            if (Auth::user()->role === 'admin') {
                $request->session()->regenerate();
                
                return redirect()->intended(route('admin.dashboard'))
                    ->with('success', admin_trans('login_success'));
            }

            // Not admin, logout
            Auth::logout();
            return back()
                ->withInput($request->only('email', 'remember'))
                ->with('error', admin_trans('unauthorized_access'));
        }

        // Login failed
        return back()
            ->withInput($request->only('email', 'remember'))
            ->with('error', admin_trans('invalid_credentials'));
    }

    /**
     * Handle logout request
     */
    public function logout(Request $request)
    {
        Auth::logout();

        $request->session()->invalidate();
        $request->session()->regenerateToken();

        return redirect()->route('admin.login')
            ->with('success', admin_trans('logout_success'));
    }

    /**
     * Switch locale
     */
    public function switchLocale($locale)
    {
        if (in_array($locale, ['ar', 'en'])) {
            session(['admin_locale' => $locale]);
        }

        return back();
    }
}