<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Services\ClarityService;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Carbon\Carbon;

class AnalyticsController extends Controller
{
    protected $clarityService;

    public function __construct(ClarityService $clarityService)
    {
        $this->clarityService = $clarityService;
    }

    /**
     * Analytics dashboard page
     */
    public function index()
    {
        return view('admin.analytics.index');
    }

    /**
     * Get analytics overview stats
     */
    public function getStats(Request $request): JsonResponse
    {
        $request->validate([
            'period' => 'nullable|in:today,yesterday,last_7_days,last_30_days',
            'start_date' => 'nullable|date',
            'end_date' => 'nullable|date|after_or_equal:start_date',
        ]);

        $period = $request->input('period', 'today');
        
        // Calculate date range
        if ($request->filled('start_date') && $request->filled('end_date')) {
            $startDate = $request->input('start_date');
            $endDate = $request->input('end_date');
        } else {
            [$startDate, $endDate] = $this->calculateDateRange($period);
        }

        try {
            $analytics = $this->clarityService->getAnalytics($startDate, $endDate);

            if (!$analytics['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to retrieve analytics data'
                ], 500);
            }

            $data = $analytics['data'];

            // Calculate stats
            $stats = [
                'total_page_views' => count($data['page_views']),
                'total_events' => count($data['events']),
                'total_sessions' => $data['sessions']['total_sessions'],
                'unique_sessions' => count($data['sessions']['unique_sessions']),
                
                // Additional metrics
                'pages_per_session' => $data['sessions']['total_sessions'] > 0 
                    ? round(count($data['page_views']) / $data['sessions']['total_sessions'], 2) 
                    : 0,
                'events_per_session' => $data['sessions']['total_sessions'] > 0 
                    ? round(count($data['events']) / $data['sessions']['total_sessions'], 2) 
                    : 0,
            ];

            // Get platform breakdown
            $platformStats = $this->getPlatformStats($data['page_views']);
            
            // Get top pages
            $topPages = $this->getTopPages($data['page_views'], 10);
            
            // Get top events
            $topEvents = $this->getTopEvents($data['events'], 10);
            
            // Get hourly activity (for today/yesterday)
            $hourlyActivity = [];
            if (in_array($period, ['today', 'yesterday'])) {
                $hourlyActivity = $this->getHourlyActivity($data['page_views'], $startDate);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'period' => $period,
                    'date_range' => [
                        'start' => $startDate,
                        'end' => $endDate,
                    ],
                    'stats' => $stats,
                    'platform_stats' => $platformStats,
                    'top_pages' => $topPages,
                    'top_events' => $topEvents,
                    'hourly_activity' => $hourlyActivity,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving analytics: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get real-time stats
     */
    public function getRealTimeStats(): JsonResponse
    {
        try {
            $today = now()->format('Y-m-d');
            $analytics = $this->clarityService->getAnalytics($today, $today);

            if (!$analytics['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to retrieve real-time data'
                ], 500);
            }

            $data = $analytics['data'];

            // Get last hour activity
            $lastHour = now()->subHour();
            $recentPageViews = collect($data['page_views'])
                ->filter(function ($view) use ($lastHour) {
                    return Carbon::parse($view['timestamp'])->gt($lastHour);
                });

            $recentEvents = collect($data['events'])
                ->filter(function ($event) use ($lastHour) {
                    return Carbon::parse($event['timestamp'])->gt($lastHour);
                });

            return response()->json([
                'success' => true,
                'data' => [
                    'active_sessions_estimate' => $recentPageViews->pluck('session_id')->unique()->count(),
                    'page_views_last_hour' => $recentPageViews->count(),
                    'events_last_hour' => $recentEvents->count(),
                    'top_active_pages' => $recentPageViews
                        ->groupBy('page_url')
                        ->map(function ($group) {
                            return [
                                'url' => $group->first()['page_url'],
                                'title' => $group->first()['page_title'] ?? '',
                                'views' => $group->count(),
                            ];
                        })
                        ->sortByDesc('views')
                        ->take(5)
                        ->values(),
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving real-time stats: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Get events breakdown
     */
    public function getEventsBreakdown(Request $request): JsonResponse
    {
        $request->validate([
            'period' => 'nullable|in:today,yesterday,last_7_days,last_30_days',
            'event_type' => 'nullable|string',
        ]);

        $period = $request->input('period', 'last_7_days');
        $eventType = $request->input('event_type');
        
        [$startDate, $endDate] = $this->calculateDateRange($period);

        try {
            $analytics = $this->clarityService->getAnalytics($startDate, $endDate);

            if (!$analytics['success']) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to retrieve events data'
                ], 500);
            }

            $events = collect($analytics['data']['events']);

            // Filter by event type if specified
            if ($eventType) {
                $events = $events->filter(function ($event) use ($eventType) {
                    return str_contains($event['event_name'], $eventType);
                });
            }

            // Group events by type and count
            $eventBreakdown = $events
                ->groupBy('event_name')
                ->map(function ($group, $eventName) {
                    return [
                        'event_name' => $eventName,
                        'count' => $group->count(),
                        'unique_users' => $group->pluck('user_id')->filter()->unique()->count(),
                    ];
                })
                ->sortByDesc('count')
                ->values();

            // Daily breakdown for charts
            $dailyBreakdown = $events
                ->groupBy(function ($event) {
                    return Carbon::parse($event['timestamp'])->format('Y-m-d');
                })
                ->map(function ($group, $date) {
                    return [
                        'date' => $date,
                        'count' => $group->count(),
                    ];
                })
                ->sortBy('date')
                ->values();

            return response()->json([
                'success' => true,
                'data' => [
                    'period' => $period,
                    'total_events' => $events->count(),
                    'event_breakdown' => $eventBreakdown,
                    'daily_breakdown' => $dailyBreakdown,
                ]
            ]);

        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error retrieving events breakdown: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Calculate date range based on period
     */
    private function calculateDateRange($period): array
    {
        switch ($period) {
            case 'today':
                return [now()->format('Y-m-d'), now()->format('Y-m-d')];
            case 'yesterday':
                return [now()->subDay()->format('Y-m-d'), now()->subDay()->format('Y-m-d')];
            case 'last_7_days':
                return [now()->subDays(7)->format('Y-m-d'), now()->format('Y-m-d')];
            case 'last_30_days':
                return [now()->subDays(30)->format('Y-m-d'), now()->format('Y-m-d')];
            default:
                return [now()->format('Y-m-d'), now()->format('Y-m-d')];
        }
    }

    /**
     * Get platform statistics
     */
    private function getPlatformStats($pageViews): array
    {
        $platformCounts = collect($pageViews)
            ->groupBy('platform')
            ->map(function ($group) {
                return $group->count();
            });

        $total = $platformCounts->sum();

        return $platformCounts->map(function ($count) use ($total) {
            return [
                'count' => $count,
                'percentage' => $total > 0 ? round(($count / $total) * 100, 1) : 0,
            ];
        })->toArray();
    }

    /**
     * Get top pages
     */
    private function getTopPages($pageViews, $limit = 10): array
    {
        return collect($pageViews)
            ->groupBy('page_url')
            ->map(function ($group) {
                return [
                    'url' => $group->first()['page_url'],
                    'title' => $group->first()['page_title'] ?? '',
                    'views' => $group->count(),
                    'unique_sessions' => $group->pluck('session_id')->unique()->count(),
                ];
            })
            ->sortByDesc('views')
            ->take($limit)
            ->values()
            ->toArray();
    }

    /**
     * Get top events
     */
    private function getTopEvents($events, $limit = 10): array
    {
        return collect($events)
            ->groupBy('event_name')
            ->map(function ($group, $eventName) {
                return [
                    'event_name' => $eventName,
                    'count' => $group->count(),
                    'unique_users' => $group->pluck('user_id')->filter()->unique()->count(),
                ];
            })
            ->sortByDesc('count')
            ->take($limit)
            ->values()
            ->toArray();
    }

    /**
     * Get hourly activity for a specific date
     */
    private function getHourlyActivity($pageViews, $date): array
    {
        $hourlyData = [];
        
        // Initialize all hours with 0
        for ($hour = 0; $hour < 24; $hour++) {
            $hourlyData[$hour] = [
                'hour' => $hour,
                'views' => 0,
            ];
        }

        // Count views by hour
        collect($pageViews)
            ->filter(function ($view) use ($date) {
                return Carbon::parse($view['timestamp'])->format('Y-m-d') === $date;
            })
            ->each(function ($view) use (&$hourlyData) {
                $hour = Carbon::parse($view['timestamp'])->hour;
                $hourlyData[$hour]['views']++;
            });

        return array_values($hourlyData);
    }
}