<?php

use App\Http\Controllers\Api\TestController;
use App\Http\Controllers\Api\LocalizationController;
use App\Http\Controllers\Api\AuthController;
use App\Http\Controllers\Api\UserController;
use App\Http\Controllers\Api\SubscriptionController;
use App\Http\Controllers\Api\CategoryController;
use App\Http\Controllers\Api\UserLocaleController;
use App\Http\Controllers\Api\BrandController;
use App\Http\Controllers\Api\StoreController;
use App\Http\Controllers\Api\OfferController;
use App\Http\Controllers\Api\OccasionController;
use App\Http\Controllers\Api\NotificationApiController;
use App\Http\Controllers\Api\SettingsApiController;
use App\Http\Controllers\Api\PageController;
use App\Http\Controllers\Api\ClarityController;
use Illuminate\Support\Facades\Route;

/*
|--------------------------------------------------------------------------
| API Routes
|--------------------------------------------------------------------------
*/

// Test Routes (no authentication required)
Route::prefix('test')->name('test.')->group(function () {
    Route::get('/health', [TestController::class, 'health'])->name('health');
    Route::get('/info', [TestController::class, 'info'])->name('info');
    Route::post('/echo', [TestController::class, 'echo'])->name('echo');
});

/*
|--------------------------------------------------------------------------
| Localization Routes (Public)
|--------------------------------------------------------------------------
*/
Route::prefix('localization')->name('localization.')->group(function () {
    Route::get('/countries', [LocalizationController::class, 'countries'])->name('countries');
    Route::get('/countries/{country}/languages', [LocalizationController::class, 'countryLanguages'])->name('country-languages');
    Route::get('/languages', [LocalizationController::class, 'languages'])->name('languages');
    Route::get('/locale', [LocalizationController::class, 'locale'])->name('locale');
    Route::get('/defaults', [LocalizationController::class, 'defaults'])->name('defaults');
});

/*
|--------------------------------------------------------------------------
| Main API Routes with Locale Middleware
|--------------------------------------------------------------------------
*/
Route::middleware(['locale.api'])->group(function () {

    /*
    |--------------------------------------------------------------------------
    | Authentication Routes (Public)
    |--------------------------------------------------------------------------
    */
    Route::prefix('auth')->name('auth.')->group(function () {
        // User Registration
        Route::post('/register', [AuthController::class, 'register'])->name('register');

        // Guest Registration
        Route::post('/register-guest', [AuthController::class, 'registerGuest'])->name('register-guest');

        // Login Routes
        Route::post('/login', [AuthController::class, 'login'])->name('login');
        Route::post('/social-login', [AuthController::class, 'socialLogin'])->name('social-login');

        // Password Reset Routes
        Route::post('/forgot-password', [AuthController::class, 'forgotPassword'])->name('forgot-password');
        Route::post('/verify-otp', [AuthController::class, 'verifyOTP'])->name('verify-otp');
        Route::post('/reset-password', [AuthController::class, 'resetPassword'])->name('reset-password');
    });

    /*
    |--------------------------------------------------------------------------
    | Authenticated Routes (Require Token)
    |--------------------------------------------------------------------------
    */
    Route::middleware(['auth:sanctum'])->group(function () {

        // Auth Management Routes
        Route::prefix('auth')->name('auth.')->group(function () {
            Route::post('/logout', [AuthController::class, 'logout'])->name('logout');
            Route::post('/logout-all', [AuthController::class, 'logoutAll'])->name('logout-all');
        });

        // User Profile Routes
        Route::prefix('user')->name('user.')->group(function () {
            // Profile Management
            Route::get('/profile', [UserController::class, 'profile'])->name('profile');
            Route::put('/profile', [UserController::class, 'updateProfile'])->name('update-profile');
            Route::post('/change-password', [UserController::class, 'changePassword'])->name('change-password');

            // Account Management
            Route::post('/upgrade-account', [UserController::class, 'upgradeAccount'])->name('upgrade-account');
            Route::delete('/delete-account', [UserController::class, 'deleteAccount'])->name('delete-account');

            // Device Management
            Route::get('/devices', [UserController::class, 'devices'])->name('devices');
            Route::delete('/devices/{deviceId}', [UserController::class, 'removeDevice'])->name('remove-device');
            Route::post('/logout-device/{deviceId}', [UserController::class, 'logoutDevice'])->name('logout-device');
        });

        /*
        |--------------------------------------------------------------------------
        | Subscription/Interest Routes with Rate Limiting
        |--------------------------------------------------------------------------
        */
        Route::prefix('subscriptions')->name('subscriptions.')->group(function () {
            // Get all user subscriptions (no rate limit needed for viewing)
            Route::get('/', [SubscriptionController::class, 'index'])
                ->name('index');

            Route::get('/others', [SubscriptionController::class, 'others'])
                ->name('others');

            // Subscribe to different types (20 requests per minute)
            Route::post('/stores', [SubscriptionController::class, 'subscribeToStore'])
                ->middleware('subscription.rate:subscribe')
                ->name('subscribe-store');


            Route::post('/categories', [SubscriptionController::class, 'subscribeToCategory'])
                ->middleware('subscription.rate:subscribe')
                ->name('subscribe-category');

            Route::post('/brands', [SubscriptionController::class, 'subscribeToBrand'])
                ->middleware('subscription.rate:subscribe')
                ->name('subscribe-brand');

            Route::post('/offers', [SubscriptionController::class, 'subscribeToOffer'])
                ->middleware('subscription.rate:subscribe')
                ->name('subscribe-offer');

            // Unsubscribe (30 requests per minute)
            Route::delete('/{subscription}', [SubscriptionController::class, 'unsubscribe'])
                ->middleware('subscription.rate:unsubscribe')
                ->name('unsubscribe');
                
            // Bulk subscribe to categories (5 requests per minute - stricter limit)
            Route::post('/bulk-subscribe-categories', [SubscriptionController::class, 'bulkSubscribeCategories'])
                ->middleware('subscription.rate:bulk')
                ->name('bulk-subscribe-categories');

            // Bulk operations (5 requests per minute - stricter limit)
            Route::post('/bulk-unsubscribe', [SubscriptionController::class, 'bulkUnsubscribe'])
                ->middleware('subscription.rate:bulk')
                ->name('bulk-unsubscribe');

            // Utility routes
            Route::get('/check-status', [SubscriptionController::class, 'checkStatus'])
                ->middleware('subscription.rate:check')
                ->name('check-status');
        });









        /*
|--------------------------------------------------------------------------
| Categories Routes (Public with optional authentication)
|--------------------------------------------------------------------------
*/
        Route::prefix('categories')->name('categories.')->group(function () {
            // Categories List
            Route::get('/', [CategoryController::class, 'index'])->name('index');

            // Category Details
            Route::get('/{category}', [CategoryController::class, 'show'])->name('show');

            // Stores in Category
            Route::get('/{category}/stores', [CategoryController::class, 'stores'])->name('stores');

            // Offers in Category
            Route::get('/{category}/offers', [CategoryController::class, 'offers'])->name('offers');
        });


        /*
|--------------------------------------------------------------------------
| User Locale Management Routes
|--------------------------------------------------------------------------
*/
        Route::prefix('user/locale')->name('user.locale.')->group(function () {
            // Get current locale settings
            Route::get('/settings', [UserLocaleController::class, 'getCurrentSettings'])
                ->name('settings');

            // Change country (with automatic default language)
            Route::post('/change-country', [UserLocaleController::class, 'changeCountry'])
                ->name('change-country');

            // Change language for current country
            Route::post('/change-language', [UserLocaleController::class, 'changeLanguage'])
                ->name('change-language');

            // Get available languages for specific country
            Route::get('/countries/{country}/languages', [UserLocaleController::class, 'getCountryLanguages'])
                ->name('country-languages');

            // Reset to default settings
            Route::post('/reset', [UserLocaleController::class, 'resetToDefaults'])
                ->name('reset');
        });

        /*
|--------------------------------------------------------------------------
| Stores Routes (Public with optional authentication) 
|--------------------------------------------------------------------------
*/
        Route::prefix('stores')->name('stores.')->group(function () {
            // Stores List
            Route::get('/', [StoreController::class, 'index'])->name('index');

            // Store Details
            Route::get('/{store}', [StoreController::class, 'show'])->name('show');

            // Store Offers
            Route::get('/{store}/offers', [StoreController::class, 'offers'])->name('offers');
        });


        /*
|--------------------------------------------------------------------------
| Brands Routes (Public with optional authentication)
|--------------------------------------------------------------------------
*/
        Route::prefix('brands')->name('brands.')->group(function () {
            // Brands List
            Route::get('/', [BrandController::class, 'index'])->name('index');

            // Brand Details
            Route::get('/{brand}', [BrandController::class, 'show'])->name('show');

            // Brand Stores
            Route::get('/{brand}/stores', [BrandController::class, 'stores'])->name('stores');

            // Brand Offers
            Route::get('/{brand}/offers', [BrandController::class, 'offers'])->name('offers');
        });

        /*
|--------------------------------------------------------------------------
| Occasions Routes (Public with optional authentication)
|--------------------------------------------------------------------------
*/
        Route::prefix('occasions')->name('occasions.')->group(function () {
            // Featured Occasions (for homepage)
            Route::get('/featured', [OccasionController::class, 'featured'])->name('featured');

            // Occasions List
            Route::get('/', [OccasionController::class, 'index'])->name('index');

            // Occasion Details
            Route::get('/{occasion}', [OccasionController::class, 'show'])->name('show');

            // Stores in Occasion
            Route::get('/{occasion}/stores', [OccasionController::class, 'stores'])->name('stores');

            // Offers in Occasion
            Route::get('/{occasion}/offers', [OccasionController::class, 'offers'])->name('offers');
        });


        /*
|--------------------------------------------------------------------------
| Offers Routes (Public with optional authentication)
|--------------------------------------------------------------------------
*/
        Route::prefix('offers')->name('offers.')->group(function () {
            // Main Offers List with Advanced Filtering
            Route::get('/', [OfferController::class, 'index'])->name('index');

            // Featured Offers (for homepage)
            Route::get('/featured', [OfferController::class, 'featured'])->name('featured');

            // Expiring Soon Offers
            Route::get('/expiring-soon', [OfferController::class, 'expiringSoon'])->name('expiring-soon');

            // Offer Details
            Route::get('/{offer}', [OfferController::class, 'show'])->name('show');

            // Track Offer Usage
            Route::post('/{offer}/track', [OfferController::class, 'trackUsage'])->name('track');
        });

        /*
|--------------------------------------------------------------------------
| Notifications API Routes
|--------------------------------------------------------------------------
*/
        Route::prefix('notifications')->name('notifications.')->group(function () {

            // 📥 Get Notifications
            Route::get('/', [NotificationApiController::class, 'index'])
                ->name('index');

            Route::get('/counts', [NotificationApiController::class, 'getCounts'])
                ->name('counts');

            Route::get('/settings', [NotificationApiController::class, 'getSettings'])
                ->name('settings');

            Route::get('/{notification}', [NotificationApiController::class, 'show'])
                ->name('show');

            // ✅ Mark as Read/Unread
            Route::patch('/{notification}/read', [NotificationApiController::class, 'markAsRead'])
                ->name('mark-read');

            Route::patch('/{notification}/unread', [NotificationApiController::class, 'markAsUnread'])
                ->name('mark-unread');

            Route::patch('/mark-all-read', [NotificationApiController::class, 'markAllAsRead'])
                ->name('mark-all-read');

            // 📱 FCM Token Management
            Route::post('/fcm-token', [NotificationApiController::class, 'updateFcmToken'])
                ->name('fcm-token');

            // 🧪 Test Notification
            Route::post('/test-push', [NotificationApiController::class, 'testPushNotification'])
                ->name('test-push');
        });


        /*
|--------------------------------------------------------------------------
| App Settings API (Public)
|--------------------------------------------------------------------------
*/
        Route::prefix('settings')->name('settings.')->group(function () {
            // Get all app settings
            Route::get('/', [SettingsApiController::class, 'index'])->name('index');

            // Check app version
            Route::post('/check-version', [SettingsApiController::class, 'checkVersion'])->name('check-version');
        });

        /*
|--------------------------------------------------------------------------
| Pages Routes (Public with optional authentication)
|--------------------------------------------------------------------------
*/
        Route::prefix('pages')->name('pages.')->group(function () {
            // Get all pages
            Route::get('/', [PageController::class, 'index'])->name('index');

            // Search pages
            Route::get('/search', [PageController::class, 'search'])->name('search');

            // Get specific page details
            Route::get('/{page}', [PageController::class, 'show'])->name('show');
        });
    });





    Route::prefix('analytics')->name('analytics.')->group(function () {
        // Public tracking routes (for mobile app)
        Route::post('/track/page-view', [ClarityController::class, 'trackPageView'])->name('track-page-view');
        Route::post('/track/event', [ClarityController::class, 'trackEvent'])->name('track-event');
        Route::post('/track/interaction', [ClarityController::class, 'trackInteraction'])->name('track-interaction');
        Route::post('/track/offer-usage', [ClarityController::class, 'trackOfferUsage'])->name('track-offer-usage');
        Route::post('/track/store-follow', [ClarityController::class, 'trackStoreFollow'])->name('track-store-follow');

        // Admin only routes
        Route::middleware(['auth:sanctum'])->group(function () {
            Route::get('/data', [ClarityController::class, 'getAnalytics'])->name('data');
            Route::get('/session-stats', [ClarityController::class, 'getSessionStats'])->name('session-stats');
        });
    });
});
