@props([
    'label' => 'رفع ملف',
    'name' => 'file',
    'accept' => '*',
    'multiple' => false,
    'maxSize' => '10MB',
    'preview' => true,
])

<div x-data="fileUpload()" class="w-full">
    @if($label)
        <label class="block text-sm font-semibold mb-2 text-primary">{{ $label }}</label>
    @endif
    
    <div class="relative">
        <input type="file" 
               name="{{ $name }}"
               accept="{{ $accept }}"
               @if($multiple) multiple @endif
               @change="handleFiles($event.target.files)"
               class="hidden"
               x-ref="fileInput">
        
        <div @click="$refs.fileInput.click()"
             @dragover.prevent="dragover = true"
             @dragleave.prevent="dragover = false"
             @drop.prevent="handleDrop($event)"
             :class="dragover ? 'border-primary bg-primary-light' : 'border-gray-300'"
             class="border-2 border-dashed rounded-xl p-8 text-center cursor-pointer transition-all hover:border-primary hover:bg-primary-light">
            
            <i data-lucide="upload-cloud" class="w-12 h-12 mx-auto mb-4 text-tertiary"></i>
            <h3 class="text-lg font-semibold mb-2 text-primary">اسحب الملفات هنا أو اضغط للاختيار</h3>
            <p class="text-sm text-tertiary">الحد الأقصى: {{ $maxSize }}</p>
        </div>
        
        @if($preview)
            <div x-show="files.length > 0" class="mt-4 space-y-2">
                <template x-for="file in files" :key="file.name">
                    <div class="flex items-center gap-3 p-3 rounded-lg border">
                        <i data-lucide="file" class="w-5 h-5 text-primary"></i>
                        <div class="flex-1">
                            <p class="text-sm font-medium text-primary" x-text="file.name"></p>
                            <p class="text-xs text-tertiary" x-text="formatFileSize(file.size)"></p>
                        </div>
                        <button @click="removeFile(file)" 
                                class="p-1 rounded-full hover:bg-red-100 text-red-500">
                            <i data-lucide="x" class="w-4 h-4"></i>
                        </button>
                    </div>
                </template>
            </div>
        @endif
    </div>
</div>

<script>
function fileUpload() {
    return {
        files: [],
        dragover: false,
        
        handleFiles(fileList) {
            this.files = Array.from(fileList);
        },
        
        handleDrop(e) {
            this.dragover = false;
            this.handleFiles(e.dataTransfer.files);
        },
        
        removeFile(fileToRemove) {
            this.files = this.files.filter(file => file !== fileToRemove);
        },
        
        formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }
    }
}
</script>