@extends('admin.layouts.app')

@section('title', admin_trans('media_library'))
@section('page-title', admin_trans('media_library'))

@section('content')

    <div class="space-y-6" x-data="mediaLibrary()">

        <!-- Header with Stats -->
        <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
            <x-stat-card title="{{ admin_trans('total_files') }}" value="{{ $stats['total_files'] }}" icon="file"
                gradient="primary" />
            <x-stat-card title="{{ admin_trans('total_size') }}" value="{{ $stats['total_size'] }}" icon="hard-drive"
                gradient="secondary" />
            <x-stat-card title="{{ admin_trans('images') }}" value="{{ $stats['images'] }}" icon="image"
                gradient="success" />
            <x-stat-card title="{{ admin_trans('videos') }}" value="{{ $stats['videos'] }}" icon="video"
                gradient="warning" />
        </div>

        <!-- Upload Section -->
        <x-card :title="admin_trans('upload_files')" icon="upload">
            <div x-data="fileUploader()" class="space-y-4">

                <!-- Model Selection -->
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4 p-4 bg-gray-50 rounded-xl">
                    <div>
                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('element_type') }}</label>
                        <select x-model="selectedModelType" @change="loadModelItems()" class="input">
                            <option value="">{{ admin_trans('select_type') }}</option>
                            @foreach ($availableModels as $class => $label)
                                <option value="{{ $class }}">{{ $label }}</option>
                            @endforeach
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('element') }}</label>
                        <select x-model="selectedModelId" class="input" :disabled="!selectedModelType">
                            <option value="">{{ admin_trans('select_element') }}</option>
                            <template x-for="(name, id) in modelItems" :key="id">
                                <option :value="id" x-text="name"></option>
                            </template>
                        </select>
                    </div>

                    <div>
                        <label class="block text-sm font-semibold mb-2 text-primary">{{ admin_trans('collection') }}</label>
                        <select x-model="selectedCollection" class="input">
                            <option value="default">{{ admin_trans('collection_default') }}</option>
                            <option value="logo">{{ admin_trans('collection_logo') }}</option>
                            <option value="gallery">{{ admin_trans('collection_gallery') }}</option>
                            <option value="icon">{{ admin_trans('collection_icon') }}</option>
                            <option value="banner">{{ admin_trans('collection_banner') }}</option>
                        </select>
                    </div>
                </div>

                <!-- Upload Area -->
                <div @dragover.prevent="dragover = true" @dragleave.prevent="dragover = false"
                    @drop.prevent="handleDrop($event)"
                    :class="dragover ? 'border-primary bg-primary-light' : 'border-gray-300'" @click="openFileDialog()"
                    class="border-2 border-dashed rounded-xl p-8 text-center cursor-pointer transition-all hover:border-primary hover:bg-primary-light">
                    <input type="file" multiple accept="*" @change="handleFiles($event.target.files)"
                        x-ref="fileInput" class="hidden">

                    <i data-lucide="upload-cloud" class="w-16 h-16 mx-auto mb-4 text-tertiary"></i>
                    <h3 class="text-lg font-semibold mb-2 text-primary">{{ admin_trans('drag_files_or_click') }}</h3>
                    <p class="text-sm text-tertiary">{{ admin_trans('upload_limit_text') }}</p>
                </div>

                <!-- Upload Progress -->
                <div x-show="uploading" x-cloak class="space-y-2">
                    <div class="flex items-center justify-between">
                        <span class="text-sm font-medium text-primary">{{ admin_trans('uploading') }}...</span>
                        <span class="text-sm text-tertiary" x-text="`${uploadProgress}%`"></span>
                    </div>
                    <div class="w-full bg-gray-200 rounded-full h-2">
                        <div class="bg-primary h-2 rounded-full transition-all" :style="`width: ${uploadProgress}%`"></div>
                    </div>
                </div>

                <!-- Files Preview -->
                <div x-show="selectedFiles.length > 0" x-cloak class="grid grid-cols-2 md:grid-cols-4 gap-4">
                    <template x-for="(file, index) in selectedFiles" :key="index">
                        <div class="relative p-3 border rounded-lg bg-gray-50">
                            <div class="flex items-center gap-2 mb-2">
                                <i data-lucide="file" class="w-4 h-4 text-primary"></i>
                                <span class="text-xs font-medium text-primary truncate" x-text="file.name"></span>
                            </div>
                            <p class="text-xs text-tertiary" x-text="formatFileSize(file.size)"></p>
                            <button @click="removeFile(index)"
                                class="absolute top-1 right-1 p-1 rounded-full hover:bg-red-100 text-red-500">
                                <i data-lucide="x" class="w-3 h-3"></i>
                            </button>
                        </div>
                    </template>
                </div>

                <!-- Upload Button -->
                <div x-show="selectedFiles.length > 0 && !uploading" x-cloak class="flex justify-end">
                    <x-button @click="uploadFiles()" variant="primary" icon="upload">
                        {{ admin_trans('upload_files') }}
                    </x-button>
                </div>
            </div>
        </x-card>

<!-- Filters -->
<x-card>
    <div class="space-y-4">
        
        <!-- Search Bar -->
        <div class="w-full">
            <form method="GET" action="{{ route('admin.media.index') }}" class="relative">
                @foreach (request()->except(['search', 'page']) as $key => $value)
                    <input type="hidden" name="{{ $key }}" value="{{ $value }}">
                @endforeach

                <div class="relative">
                    <input type="text" name="search" value="{{ request('search') }}"
                        placeholder="{{ admin_trans('search_files') }}"
                        class="input w-full {{ is_rtl() ? 'pr-12 pl-12' : 'pl-12 pr-12' }}" 
                        autocomplete="off">
                    
                    
                    <!-- Clear Search Button -->
                    @if (request('search'))
                        <a href="{{ route('admin.media.index', request()->except(['search', 'page'])) }}"
                           class="absolute inset-y-0 {{ is_rtl() ? 'left-0' : 'right-0' }} flex items-center {{ is_rtl() ? 'pl-4' : 'pr-4' }} hover:text-primary transition-colors">
                            <i data-lucide="x-circle" class="w-5 h-5 text-tertiary hover:text-red-500"></i>
                        </a>
                    @else
                        <!-- Search Submit Button -->
                        <button type="submit" 
                                class="absolute inset-y-0 {{ is_rtl() ? 'left-0' : 'right-0' }} flex items-center {{ is_rtl() ? 'pl-4' : 'pr-4' }} hover:text-primary transition-colors">
                            <i data-lucide="arrow-right" class="w-5 h-5 text-tertiary {{ is_rtl() ? 'rotate-180' : '' }}"></i>
                        </button>
                    @endif
                </div>
            </form>
        </div>

        <!-- Filter Controls -->
        <div class="flex flex-col sm:flex-row gap-3 items-start sm:items-center justify-between">
            
            <!-- Filter Dropdowns -->
            <div class="flex flex-wrap gap-2 w-full sm:w-auto">
                <form method="GET" action="{{ route('admin.media.index') }}" 
                      class="flex flex-wrap gap-2 w-full sm:w-auto" 
                      id="filtersForm">
                    
                    @if (request('search'))
                        <input type="hidden" name="search" value="{{ request('search') }}">
                    @endif

                    <!-- File Type Filter -->
                    <div class="min-w-[120px]">
                        <select name="file_type" 
                                class="input text-sm w-full" 
                                onchange="document.getElementById('filtersForm').submit()">
                            <option value="">🗂️ جميع الأنواع</option>
                            <option value="image" {{ request('file_type') === 'image' ? 'selected' : '' }}>
                                🖼️ {{ admin_trans('images') }}
                            </option>
                            <option value="video" {{ request('file_type') === 'video' ? 'selected' : '' }}>
                                🎥 {{ admin_trans('videos') }}
                            </option>
                            <option value="application" {{ request('file_type') === 'application' ? 'selected' : '' }}>
                                📄 {{ admin_trans('documents') }}
                            </option>
                        </select>
                    </div>

                    <!-- Collection Filter -->
                    @if ($collections->count() > 0)
                        <div class="min-w-[120px]">
                            <select name="collection" 
                                    class="input text-sm w-full" 
                                    onchange="document.getElementById('filtersForm').submit()">
                                <option value="">📁 جميع المجموعات</option>
                                @foreach ($collections as $collection)
                                    <option value="{{ $collection }}"
                                        {{ request('collection') === $collection ? 'selected' : '' }}>
                                        @switch($collection)
                                            @case('logo')
                                                🏷️ {{ admin_trans('collection_logo') }}
                                                @break
                                            @case('gallery')
                                                🖼️ {{ admin_trans('collection_gallery') }}
                                                @break
                                            @case('icon')
                                                ⭐ {{ admin_trans('collection_icon') }}
                                                @break
                                            @case('banner')
                                                🎯 {{ admin_trans('collection_banner') }}
                                                @break
                                            @default
                                                📂 {{ ucfirst($collection) }}
                                        @endswitch
                                    </option>
                                @endforeach
                            </select>
                        </div>
                    @endif

                    <!-- Model Type Filter -->
                    <div class="min-w-[120px]">
                        <select name="model_type" 
                                class="input text-sm w-full" 
                                onchange="document.getElementById('filtersForm').submit()">
                            <option value="">🏢 جميع الأنواع</option>
                            @foreach ($availableModels as $class => $label)
                                <option value="{{ $class }}"
                                    {{ request('model_type') === $class ? 'selected' : '' }}>
                                    @switch($class)
                                        @case('App\Models\Store')
                                            🏪
                                            @break
                                        @case('App\Models\Brand')
                                            🏷️
                                            @break
                                        @case('App\Models\Category')
                                            📂
                                            @break
                                        @case('App\Models\Occasion')
                                            🎉
                                            @break
                                        @case('App\Models\CarouselSlide')
                                            🎠
                                            @break
                                        @default
                                            📋
                                    @endswitch
                                    {{ $label }}
                                </option>
                            @endforeach
                        </select>
                    </div>

                    <!-- Sort -->
                    <div class="min-w-[120px]">
                        <select name="sort" 
                                class="input text-sm w-full" 
                                onchange="document.getElementById('filtersForm').submit()">
                            <option value="created_at" {{ request('sort') === 'created_at' ? 'selected' : '' }}>
                                🕒 {{ admin_trans('newest_first') }}
                            </option>
                            <option value="name" {{ request('sort') === 'name' ? 'selected' : '' }}>
                                🔤 {{ admin_trans('name') }}
                            </option>
                            <option value="size" {{ request('sort') === 'size' ? 'selected' : '' }}>
                                📏 {{ admin_trans('file_size') }}
                            </option>
                        </select>
                    </div>
                </form>
            </div>

            <!-- Action Buttons -->
            <div class="flex gap-2 w-full sm:w-auto justify-end">
                <!-- Clear Filters -->
                @if (request()->hasAny(['search', 'file_type', 'collection', 'sort', 'model_type']))
                    <x-button variant="outline" size="sm" icon="refresh-cw"
                        href="{{ route('admin.media.index') }}"
                        class="bg-red-50 border-red-200 text-red-600 hover:bg-red-100 transition-colors">
                        {{ admin_trans('clear_filters') }}
                    </x-button>
                @endif

                <!-- Refresh Button -->
                <x-button variant="outline" size="sm" icon="refresh-ccw"
                    onclick="window.location.reload()"
                    class="bg-blue-50 border-blue-200 text-blue-600 hover:bg-blue-100 transition-colors">
                    {{ admin_trans('update') }}
                </x-button>
            </div>
        </div>

        <!-- Active Filters Tags -->
        @if (request()->hasAny(['search', 'file_type', 'collection', 'sort', 'model_type']))
            <div class="flex flex-wrap gap-2 pt-2 border-t border-gray-100">
                <span class="text-sm text-tertiary {{ is_rtl() ? 'ml-2' : 'mr-2' }}">{{ admin_trans('active_filters') }}:</span>
                
                @if (request('search'))
                    <span class="inline-flex items-center gap-1 px-2 py-1 bg-primary-50 text-primary-700 rounded-md text-xs">
                        <i data-lucide="search" class="w-3 h-3"></i>
                        "{{ request('search') }}"
                        <a href="{{ route('admin.media.index', request()->except(['search', 'page'])) }}" 
                           class="hover:text-red-500 transition-colors">
                            <i data-lucide="x" class="w-3 h-3"></i>
                        </a>
                    </span>
                @endif
                
                @if (request('file_type'))
                    <span class="inline-flex items-center gap-1 px-2 py-1 bg-green-50 text-green-700 rounded-md text-xs">
                        <i data-lucide="file" class="w-3 h-3"></i>
                        {{ request('file_type') === 'image' ? admin_trans('images') : (request('file_type') === 'video' ? admin_trans('videos') : admin_trans('documents')) }}
                        <a href="{{ route('admin.media.index', request()->except(['file_type', 'page'])) }}"
                           class="hover:text-red-500 transition-colors">
                            <i data-lucide="x" class="w-3 h-3"></i>
                        </a>
                    </span>
                @endif
                
                @if (request('collection'))
                    <span class="inline-flex items-center gap-1 px-2 py-1 bg-purple-50 text-purple-700 rounded-md text-xs">
                        <i data-lucide="folder" class="w-3 h-3"></i>
                        {{ ucfirst(request('collection')) }}
                        <a href="{{ route('admin.media.index', request()->except(['collection', 'page'])) }}" 
                           class="hover:text-red-500 transition-colors">
                            <i data-lucide="x" class="w-3 h-3"></i>
                        </a>
                    </span>
                @endif
                
                @if (request('model_type'))
                    <span class="inline-flex items-center gap-1 px-2 py-1 bg-orange-50 text-orange-700 rounded-md text-xs">
                        <i data-lucide="tag" class="w-3 h-3"></i>
                        {{ $availableModels[request('model_type')] ?? request('model_type') }}
                        <a href="{{ route('admin.media.index', request()->except(['model_type', 'page'])) }}" 
                           class="hover:text-red-500 transition-colors">
                            <i data-lucide="x" class="w-3 h-3"></i>
                        </a>
                    </span>
                @endif
            </div>
        @endif
    </div>
</x-card>

        <!-- Bulk Actions -->
        <div x-show="selectedItems.length > 0" x-cloak>
            <x-alert variant="info">
                <div class="flex items-center justify-between w-full">
                    <span x-text="`${selectedItems.length} {{ admin_trans('files_selected') }}`"></span>
                    <div class="flex gap-2">
                        <x-button size="sm" variant="outline"
                            @click="downloadSelected()">{{ admin_trans('download') }}</x-button>
                        <x-button size="sm" variant="danger"
                            @click="deleteSelected()">{{ admin_trans('delete') }}</x-button>
                        <x-button size="sm" variant="outline"
                            @click="clearSelection()">{{ admin_trans('clear') }}</x-button>
                    </div>
                </div>
            </x-alert>
        </div>

        <!-- Media Grid -->
        <x-card>
            @if ($media->isEmpty())
                <x-empty-state icon="image" :title="admin_trans('no_media_files')" :description="admin_trans('start_by_uploading_files')">
                    <x-button variant="primary" icon="upload" @click="$refs.fileInput.click()">
                        {{ admin_trans('upload_first_file') }}
                    </x-button>
                </x-empty-state>
            @else
                <!-- Media Grid -->
                <div class="grid grid-cols-3 sm:grid-cols-4 md:grid-cols-6 lg:grid-cols-8 xl:grid-cols-10 gap-3">
                    @foreach ($media as $item)
                        <div class="media-item relative group cursor-pointer" @click="viewMedia({{ $item->id }})"
                            data-id="{{ $item->id }}">

                            <!-- Selection Checkbox -->
                            <div class="absolute top-1 left-1 z-10">
                                <input type="checkbox" value="{{ $item->id }}"
                                    @change="toggleItem({{ $item->id }})"
                                    :checked="selectedItems.includes({{ $item->id }})" @click.stop
                                    class="w-4 h-4 text-primary bg-white border-2 border-gray-300 rounded focus:ring-primary focus:ring-2">
                            </div>

                            <!-- File Preview -->
                            <div
                                class="aspect-square rounded-lg overflow-hidden border bg-gray-100 hover:border-primary transition-colors">
                                @if (str_starts_with($item->mime_type, 'image/'))
                                    @php
                                        // محاولة الحصول على رابط الصورة بطرق مختلفة
                                        $url = null;
                                        $customProperties = $item->custom_properties ?? [];

                                        // الطريقة الأولى: من custom_properties
                                        if (
                                            isset($customProperties['path']) &&
                                            Storage::disk('public')->exists($customProperties['path'])
                                        ) {
                                            $url = asset('storage/' . $customProperties['path']);
                                        }

                                        // الطريقة الثانية: جرب getUrl إذا كان متاح
                                        if (!$url) {
                                            try {
                                                if (method_exists($item, 'getUrl')) {
                                                    $tempUrl = $item->getUrl();
                                                    if ($tempUrl && $tempUrl !== '') {
                                                        $url = $tempUrl;
                                                    }
                                                }
                                            } catch (\Exception $e) {
                                                // استمر للطريقة التالية
                                            }
                                        }

                                        // الطريقة الثالثة: جرب المسارات المحتملة
                                        if (!$url) {
                                            $possiblePaths = [
                                                "media/{$item->getTable()}/{$item->model_id}/{$item->file_name}",
                                                "media/{$item->id}/{$item->file_name}",
                                                "media/{$item->file_name}",
                                                $item->file_name,
                                            ];

                                            foreach ($possiblePaths as $path) {
                                                if (Storage::disk('public')->exists($path)) {
                                                    $url = asset('storage/' . $path);
                                                    break;
                                                }
                                            }
                                        }
                                    @endphp

                                    @if ($url)
                                        <img src="{{ $url }}" alt="{{ $item->name }}"
                                            class="w-full h-full object-cover" loading="lazy">
                                    @else
                                        <div
                                            class="w-full h-full flex items-center justify-center bg-gradient-to-br from-gray-100 to-gray-200">
                                            <div class="text-center">
                                                <i data-lucide="image-off" class="w-6 h-6 text-gray-400 mb-1"></i>
                                                <p class="text-xs text-gray-500">{{ admin_trans('not_available') }}</p>
                                            </div>
                                        </div>
                                    @endif
                                @elseif(str_starts_with($item->mime_type, 'video/'))
                                    <div
                                        class="w-full h-full flex items-center justify-center bg-gradient-to-br from-purple-100 to-purple-200">
                                        <i data-lucide="video" class="w-6 h-6 text-purple-600"></i>
                                    </div>
                                @else
                                    <div
                                        class="w-full h-full flex items-center justify-center bg-gradient-to-br from-blue-100 to-blue-200">
                                        <i data-lucide="file-text" class="w-6 h-6 text-blue-600"></i>
                                    </div>
                                @endif
                            </div>

                            <!-- File Info -->
                            <div class="mt-1 px-1">
                                <h4 class="text-xs font-medium text-primary truncate" title="{{ $item->name }}">
                                    {{ Str::limit($item->name, 15) }}
                                </h4>
                                <div class="flex items-center justify-between mt-0.5">
                                    <span class="text-xs text-tertiary">
                                        {{ number_format($item->size / 1024, 0) }}KB
                                    </span>
                                    @if ($item->collection_name !== 'default')
                                        <span class="text-xs px-1 py-0.5 bg-secondary text-secondary rounded">
                                            {{ Str::limit($item->collection_name, 4) }}
                                        </span>
                                    @endif
                                </div>
                                @if ($item->model_type)
                                    <div class="text-xs text-gray-500 mt-1">
                                        {{ class_basename($item->model_type) }} #{{ $item->model_id }}
                                    </div>
                                @endif
                            </div>

                            <!-- Actions on Hover -->
                            <div
                                class="absolute inset-0 bg-black/60 opacity-0 group-hover:opacity-100 transition-opacity rounded-lg flex items-center justify-center gap-1">
                                <button @click.stop="viewMedia({{ $item->id }})"
                                    class="p-2 bg-white/90 rounded-lg hover:bg-white transition-colors">
                                    <i data-lucide="eye" class="w-4 h-4 text-gray-700"></i>
                                </button>
                                <a href="{{ route('admin.media.download', $item) }}" @click.stop
                                    class="p-2 bg-white/90 rounded-lg hover:bg-white transition-colors">
                                    <i data-lucide="download" class="w-4 h-4 text-gray-700"></i>
                                </a>
                                <a href="{{ route('admin.media.edit', $item) }}" @click.stop
                                    class="p-2 bg-white/90 rounded-lg hover:bg-white transition-colors">
                                    <i data-lucide="edit" class="w-4 h-4 text-blue-600"></i>
                                </a>
                                <button @click.stop="deleteMedia({{ $item->id }})"
                                    class="p-2 bg-white/90 rounded-lg hover:bg-white transition-colors">
                                    <i data-lucide="trash-2" class="w-4 h-4 text-red-600"></i>
                                </button>
                            </div>
                        </div>
                    @endforeach
                </div>

                <!-- Pagination -->
                @if ($media->hasPages())
                    <div class="mt-6">
                        <x-pagination :paginator="$media" />
                    </div>
                @endif
            @endif
        </x-card>

        <!-- Media Details Modal -->
        <x-modal name="media-details-modal" maxWidth="2xl">
            <div class="p-6" x-data="mediaDetails()">
                <div class="flex items-center justify-between mb-6">
                    <h3 class="text-xl font-bold text-primary">{{ admin_trans('media_details') }}</h3>
                    <button @click="$dispatch('close-modal', 'media-details-modal')"
                        class="p-2 rounded-lg hover:bg-primary-light">
                        <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
                    </button>
                </div>

                <div x-show="loading" class="flex items-center justify-center py-12">
                    <div class="text-center">
                        <div
                            class="w-12 h-12 mx-auto mb-4 border-4 border-primary/20 border-t-primary rounded-full animate-spin">
                        </div>
                        <p class="text-sm text-tertiary">{{ admin_trans('loading') }}...</p>
                    </div>
                </div>

                <div x-show="!loading && mediaData" x-cloak class="space-y-6">
                    <!-- Media Preview -->
                    <div class="text-center">
                        <div x-show="mediaData && mediaData.mime_type && mediaData.mime_type.startsWith('image/')"
                            class="inline-block max-w-full">
                            <img :src="mediaData?.url" :alt="mediaData?.name" class="max-h-64 rounded-lg border">
                        </div>
                        <div x-show="mediaData && mediaData.mime_type && !mediaData.mime_type.startsWith('image/')"
                            class="w-32 h-32 mx-auto bg-gray-100 rounded-lg flex items-center justify-center">
                            <i data-lucide="file" class="w-16 h-16 text-gray-400"></i>
                        </div>
                    </div>

                    <!-- Media Info -->
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4 text-sm">
                        <div>
                            <span class="font-medium text-tertiary">{{ admin_trans('filename') }}:</span>
                            <p class="text-primary mt-1" x-text="mediaData?.file_name"></p>
                        </div>
                        <div>
                            <span class="font-medium text-tertiary">{{ admin_trans('file_size') }}:</span>
                            <p class="text-primary mt-1" x-text="mediaData?.size"></p>
                        </div>
                        <div>
                            <span class="font-medium text-tertiary">{{ admin_trans('file_type') }}:</span>
                            <p class="text-primary mt-1" x-text="mediaData?.mime_type"></p>
                        </div>
                        <div>
                            <span class="font-medium text-tertiary">{{ admin_trans('collection') }}:</span>
                            <p class="text-primary mt-1" x-text="mediaData?.collection"></p>
                        </div>
                        <div>
                            <span class="font-medium text-tertiary">{{ admin_trans('uploaded_at') }}:</span>
                            <p class="text-primary mt-1" x-text="mediaData?.created_at"></p>
                        </div>
                        <div x-show="mediaData?.model_type">
                            <span class="font-medium text-tertiary">{{ admin_trans('attached_to') }}:</span>
                            <p class="text-primary mt-1" x-text="`${mediaData?.model_type} #${mediaData?.model_id}`"></p>
                        </div>
                    </div>

                    <!-- Actions -->
                    <div class="flex gap-3 pt-4 border-t" style="border-color: var(--color-border);">
                        <x-button variant="outline" icon="download" @click="downloadMedia()">
                            {{ admin_trans('download') }}
                        </x-button>
                        <x-button variant="outline" icon="edit" @click="editMedia()">
                            {{ admin_trans('edit') }}
                        </x-button>
                        <x-button variant="danger" icon="trash-2" @click="deleteCurrentMedia()">
                            {{ admin_trans('delete') }}
                        </x-button>
                    </div>
                </div>
            </div>
        </x-modal>

        <!-- Cleanup Modal -->
        <x-modal name="cleanup-modal" maxWidth="md">
            <div class="p-6" x-data="cleanupModal()">
                <div class="text-center mb-6">
                    <div class="w-16 h-16 rounded-2xl bg-warning-50 flex items-center justify-center mx-auto mb-4">
                        <i data-lucide="trash-2" class="w-8 h-8 text-warning-500"></i>
                    </div>
                    <h3 class="text-xl font-bold text-primary">{{ admin_trans('cleanup_unused_media') }}</h3>
                    <p class="text-sm text-secondary mt-2">{{ admin_trans('unused_files_warning') }}</p>
                </div>

                <div x-show="!checking && !cleaning" class="text-center">
                    <x-button @click="checkUnused()" variant="warning" icon="search">
                        {{ admin_trans('check_unused_files') }}
                    </x-button>
                </div>

                <div x-show="checking" x-cloak class="flex items-center justify-center py-8">
                    <div class="text-center">
                        <div
                            class="w-10 h-10 mx-auto mb-3 border-4 border-warning-200 border-t-warning-500 rounded-full animate-spin">
                        </div>
                        <p class="text-sm text-tertiary">{{ admin_trans('checking') }}...</p>
                    </div>
                </div>

                <div x-show="unusedCount > 0 && !cleaning" x-cloak class="text-center space-y-4">
                    <div class="p-4 rounded-lg bg-warning-50">
                        <p class="text-warning-700">
                            <span x-text="unusedCount"></span> {{ admin_trans('unused_files_found') }}
                        </p>
                    </div>
                    <div class="flex gap-3 justify-center">
                        <x-button @click="$dispatch('close-modal', 'cleanup-modal')" variant="outline">
                            {{ admin_trans('cancel') }}
                        </x-button>
                        <x-button @click="executeCleanup()" variant="danger" icon="trash-2">
                            {{ admin_trans('delete_unused_files') }}
                        </x-button>
                    </div>
                </div>

                <div x-show="cleaning" x-cloak class="flex items-center justify-center py-8">
                    <div class="text-center">
                        <div
                            class="w-10 h-10 mx-auto mb-3 border-4 border-danger-200 border-t-danger-500 rounded-full animate-spin">
                        </div>
                        <p class="text-sm text-tertiary">{{ admin_trans('cleaning_up') }}...</p>
                    </div>
                </div>

                <div x-show="unusedCount === 0 && !checking && !cleaning" x-cloak class="text-center">
                    <div class="p-4 rounded-lg bg-success-50">
                        <p class="text-success-700">{{ admin_trans('no_unused_files') }}</p>
                    </div>
                    <div class="mt-4">
                        <x-button @click="$dispatch('close-modal', 'cleanup-modal')" variant="outline">
                            {{ admin_trans('close') }}
                        </x-button>
                    </div>
                </div>
            </div>
        </x-modal>
    </div>

    <!-- Quick Actions Toolbar -->
    <div class="fixed bottom-6 {{ is_rtl() ? 'left-6' : 'right-6' }} z-40">
        <div class="flex flex-col gap-3">
            <!-- Upload Button -->
            <button @click="document.querySelector('[x-data*=fileUploader]').__x.$data.openFileDialog()"
                class="w-14 h-14 rounded-full gradient-primary shadow-lg flex items-center justify-center hover:shadow-xl transition-all hover:scale-105">
                <i data-lucide="upload" class="w-6 h-6 text-white"></i>
            </button>

            <!-- Cleanup Button -->
            <button @click="$dispatch('open-modal', 'cleanup-modal')"
                class="w-12 h-12 rounded-full bg-warning-500 shadow-lg flex items-center justify-center hover:shadow-xl transition-all hover:scale-105">
                <i data-lucide="trash-2" class="w-5 h-5 text-white"></i>
            </button>
        </div>
    </div>

@endsection

@push('scripts')
    <script>
        // Media Library Main Component
        function mediaLibrary() {
            return {
                selectedItems: [],

                toggleItem(id) {
                    const index = this.selectedItems.indexOf(id);
                    if (index > -1) {
                        this.selectedItems.splice(index, 1);
                    } else {
                        this.selectedItems.push(id);
                    }
                },

                clearSelection() {
                    this.selectedItems = [];
                    document.querySelectorAll('input[type="checkbox"]').forEach(cb => cb.checked = false);
                },

                async downloadSelected() {
                    if (this.selectedItems.length === 0) {
                        toast.warning('{{ admin_trans('please_select_files') }}');
                        return;
                    }

                    try {
                        const response = await fetch('/admin/media/bulk-action', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify({
                                action: 'download',
                                selected: this.selectedItems
                            })
                        });

                        const result = await response.json();
                        if (result.success) {
                            toast.info(result.message);
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_downloading_files') }}');
                    }
                },

                async deleteSelected() {
                    if (this.selectedItems.length === 0) {
                        toast.warning('{{ admin_trans('please_select_files') }}');
                        return;
                    }

                    if (!confirm('{{ admin_trans('confirm_delete_selected_files') }}')) return;

                    try {
                        const response = await fetch('/admin/media/bulk-action', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify({
                                action: 'delete',
                                selected: this.selectedItems
                            })
                        });

                        const result = await response.json();
                        if (result.success) {
                            toast.success(result.message);
                            setTimeout(() => window.location.reload(), 1000);
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_deleting_files') }}');
                    }
                },

                viewMedia(id) {
                    window.dispatchEvent(new CustomEvent('open-modal', {
                        detail: 'media-details-modal'
                    }));
                    setTimeout(() => {
                        const modal = document.querySelector('[x-data*="mediaDetails"]');
                        if (modal && modal._x_dataStack) {
                            modal._x_dataStack[0].loadMedia(id);
                        }
                    }, 100);
                }
            }
        }

        // File Uploader Component
        function fileUploader() {
            return {
                selectedFiles: [],
                uploading: false,
                uploadProgress: 0,
                dragover: false,
                selectedModelType: '',
                selectedModelId: '',
                selectedCollection: 'default',
                modelItems: @json($modelItems),

                openFileDialog() {
                    this.$refs.fileInput.click();
                },

                handleFiles(files) {
                    this.selectedFiles = Array.from(files);
                },

                handleDrop(e) {
                    this.dragover = false;
                    this.handleFiles(e.dataTransfer.files);
                },

                removeFile(index) {
                    this.selectedFiles.splice(index, 1);
                },

                async loadModelItems() {
                    if (!this.selectedModelType) {
                        this.modelItems = {};
                        this.selectedModelId = '';
                        return;
                    }

                    try {
                        const response = await fetch('/admin/media/model-items', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify({
                                model_type: this.selectedModelType
                            })
                        });

                        const result = await response.json();
                        if (result.success) {
                            this.modelItems = result.items;
                            this.selectedModelId = '';
                        } else {
                            this.modelItems = {};
                            toast.error(result.message);
                        }
                    } catch (error) {
                        console.error('Failed to load model items:', error);
                        toast.error('{{ admin_trans("error_loading_items") }}');
                    }
                },

                async uploadFiles() {
                    if (this.selectedFiles.length === 0) {
                        toast.warning('{{ admin_trans("please_select_files") }}');
                        return;
                    }

                    if (!this.selectedModelType || !this.selectedModelId) {
                        toast.warning('{{ admin_trans("please_select_type_and_element") }}');
                        return;
                    }

                    this.uploading = true;
                    this.uploadProgress = 0;

                    const formData = new FormData();
                    this.selectedFiles.forEach(file => {
                        formData.append('files[]', file);
                    });
                    formData.append('model_type', this.selectedModelType);
                    formData.append('model_id', this.selectedModelId);
                    formData.append('collection', this.selectedCollection);
                    formData.append('_token', document.querySelector('meta[name="csrf-token"]').content);

                    try {
                        const response = await fetch('/admin/media/upload', {
                            method: 'POST',
                            body: formData
                        });

                        const result = await response.json();

                        if (result.success) {
                            toast.success(result.message);
                            this.resetForm();
                            setTimeout(() => window.location.reload(), 1000);
                        } else {
                            toast.error(result.message);
                        }
                    } catch (error) {
                        toast.error('خطأ في رفع الملفات');
                    } finally {
                        this.uploading = false;
                        this.uploadProgress = 0;
                    }
                },

                resetForm() {
                    this.selectedFiles = [];
                    this.selectedModelType = '';
                    this.selectedModelId = '';
                    this.selectedCollection = 'default';
                    this.modelItems = @json($modelItems);
                },

                formatFileSize(bytes) {
                    if (bytes === 0) return '0 Bytes';
                    const k = 1024;
                    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
                    const i = Math.floor(Math.log(bytes) / Math.log(k));
                    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
                }
            }
        }

        // Media Details Component
        function mediaDetails() {
            return {
                loading: false,
                mediaData: null,

                async loadMedia(id) {
                    this.loading = true;
                    this.mediaData = null;

                    try {
                        const response = await fetch(`/admin/media/${id}`);
                        const result = await response.json();

                        if (result.success) {
                            this.mediaData = result.media;
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_loading_media') }}');
                    } finally {
                        this.loading = false;
                    }
                },

                downloadMedia() {
                    if (this.mediaData) {
                        window.open(`/admin/media/${this.mediaData.id}/download`, '_blank');
                    }
                },

                editMedia() {
                    if (this.mediaData) {
                        window.location.href = `/admin/media/${this.mediaData.id}/edit`;
                    }
                },

                async deleteCurrentMedia() {
                    if (!this.mediaData) return;

                    if (!confirm('{{ admin_trans('confirm_delete_file') }}')) return;

                    try {
                        const response = await fetch(`/admin/media/${this.mediaData.id}`, {
                            method: 'DELETE',
                            headers: {
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            }
                        });

                        const result = await response.json();
                        if (result.success) {
                            toast.success('{{ admin_trans('file_deleted_successfully') }}');
                            this.$dispatch('close-modal', 'media-details-modal');
                            setTimeout(() => window.location.reload(), 1000);
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_deleting_file') }}');
                    }
                }
            }
        }

        // Cleanup Modal Component
        function cleanupModal() {
            return {
                checking: false,
                cleaning: false,
                unusedCount: 0,

                async checkUnused() {
                    this.checking = true;
                    this.unusedCount = 0;

                    try {
                        const response = await fetch('/admin/media/cleanup', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify({
                                execute: false
                            })
                        });

                        const result = await response.json();
                        if (result.success) {
                            this.unusedCount = result.count || 0;
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_checking_files') }}');
                    } finally {
                        this.checking = false;
                    }
                },

                async executeCleanup() {
                    this.cleaning = true;

                    try {
                        const response = await fetch('/admin/media/cleanup', {
                            method: 'POST',
                            headers: {
                                'Content-Type': 'application/json',
                                'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                            },
                            body: JSON.stringify({
                                execute: true
                            })
                        });

                        const result = await response.json();
                        if (result.success) {
                            toast.success(result.message);
                            this.$dispatch('close-modal', 'cleanup-modal');
                            setTimeout(() => window.location.reload(), 1000);
                        }
                    } catch (error) {
                        toast.error('{{ admin_trans('error_cleaning_files') }}');
                    } finally {
                        this.cleaning = false;
                    }
                }
            }
        }

        // Delete Media Function
        async function deleteMedia(id) {
            if (!confirm('{{ admin_trans('confirm_delete_file') }}')) return;

            try {
                const response = await fetch(`/admin/media/${id}`, {
                    method: 'DELETE',
                    headers: {
                        'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                    }
                });

                const result = await response.json();
                if (result.success) {
                    toast.success('{{ admin_trans('file_deleted_successfully') }}');
                    setTimeout(() => window.location.reload(), 1000);
                }
            } catch (error) {
                toast.error('{{ admin_trans('error_deleting_file') }}');
            }
        }

        // Initialize Lucide Icons
        document.addEventListener('DOMContentLoaded', function() {
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }
        });
    </script>
@endpush
