@extends('admin.layouts.app')

@section('title', admin_trans('country_translations'))
@section('page-title', admin_trans('country_translations') . ' - ' . ($country->name ?? $country->code))

@section('content')

<div class="space-y-6">
    
    <!-- Header Card -->
    <div class="card">
        <div class="flex items-center justify-between">
            <div class="flex items-center gap-4">
                <!-- Country Flag -->
                @if($country->flag_url)
                    <img src="{{ $country->flag_url }}" 
                         alt="{{ $country->name }} Flag" 
                         class="w-12 h-8 object-cover rounded border">
                @else
                    <div class="w-12 h-8 bg-gray-200 rounded border flex items-center justify-center">
                        <span class="text-sm">{{ $country->flag_emoji ?? '🏳️' }}</span>
                    </div>
                @endif
                
                <div>
                    <h2 class="text-xl font-bold text-primary">{{ $country->name ?? $country->code }}</h2>
                    <p class="text-sm text-secondary">
                        {{ $country->code }} • {{ $country->iso_code }} 
                        @if($country->dial_code)
                            • {{ $country->dial_code }}
                        @endif
                    </p>
                </div>
            </div>
            
            <x-button 
                variant="outline" 
                icon="arrow-left"
                href="{{ route('admin.countries.index') }}"
            >
                {{ admin_trans('back_to_countries') }}
            </x-button>
        </div>
    </div>

    <!-- Translations Management -->
    <div class="card">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-lg font-bold text-primary">{{ admin_trans('manage_translations') }}</h3>
            <x-badge variant="info">
                {{ count($translations) }}/{{ count($languages) }} {{ admin_trans('languages') }}
            </x-badge>
        </div>

        @if($languages->isEmpty())
            <x-empty-state 
                icon="languages"
                :title="admin_trans('no_languages_available')"
                :description="admin_trans('add_languages_first')"
            >
                <x-button 
                    variant="primary"
                    icon="plus"
                    href="{{ route('admin.languages.index') }}"
                >
                    {{ admin_trans('manage_languages') }}
                </x-button>
            </x-empty-state>
        @else
            <div class="space-y-4">
                @foreach($languages as $language)
                    @php
                        $translation = $translations->get($language->id);
                        $hasTranslation = $translation !== null;
                    @endphp
                    
                    <div class="p-4 border rounded-xl {{ $hasTranslation ? 'bg-green-50 border-green-200' : 'bg-gray-50 border-gray-200' }}">
                        <form method="POST" action="{{ route('admin.countries.store-translation', $country) }}" 
                              class="flex items-center gap-4">
                            @csrf
                            <input type="hidden" name="language_id" value="{{ $language->id }}">
                            
                            <!-- Language Info -->
                            <div class="flex items-center gap-3 min-w-0 flex-shrink-0 w-40">
                                <div class="w-8 h-8 rounded-lg flex items-center justify-center text-white font-bold text-xs"
                                     style="background: {{ $hasTranslation ? 'var(--color-success-500)' : 'var(--color-gray-400)' }};">
                                    {{ strtoupper(substr($language->code, 0, 2)) }}
                                </div>
                                <div class="min-w-0">
                                    <p class="font-semibold text-sm text-primary truncate">{{ $language->name }}</p>
                                    <p class="text-xs text-tertiary truncate">{{ $language->native_name }}</p>
                                </div>
                            </div>
                            
                            <!-- Translation Input -->
                            <div class="flex-1">
                                <input 
                                    type="text" 
                                    name="name" 
                                    value="{{ $translation->name ?? '' }}"
                                    placeholder="{{ admin_trans('enter_country_name_in') }} {{ $language->native_name }}"
                                    class="input w-full"
                                    required
                                >
                            </div>
                            
                            <!-- Action Buttons -->
                            <div class="flex items-center gap-2 flex-shrink-0">
                                <x-button 
                                    type="submit"
                                    variant="{{ $hasTranslation ? 'secondary' : 'primary' }}"
                                    size="sm"
                                    :icon="$hasTranslation ? 'edit' : 'plus'"
                                >
                                    {{ $hasTranslation ? admin_trans('update') : admin_trans('add') }}
                                </x-button>
                                
                                @if($hasTranslation)
                                    <div class="flex items-center gap-1">
                                        <i data-lucide="check-circle" class="w-4 h-4 text-green-600"></i>
                                        <span class="text-xs text-green-600 font-medium">{{ admin_trans('saved') }}</span>
                                    </div>
                                @endif
                            </div>
                        </form>
                    </div>
                @endforeach
            </div>

            <!-- Translation Progress -->
            <div class="mt-6 p-4 bg-blue-50 rounded-xl">
                <div class="flex items-center justify-between mb-2">
                    <span class="text-sm font-medium text-blue-800">{{ admin_trans('translation_progress') }}</span>
                    <span class="text-sm font-bold text-blue-800">{{ count($translations) }}/{{ count($languages) }}</span>
                </div>
                <div class="w-full bg-blue-200 rounded-full h-2">
                    <div class="bg-blue-600 h-2 rounded-full transition-all" 
                         style="width: {{ count($languages) > 0 ? (count($translations) / count($languages)) * 100 : 0 }}%"></div>
                </div>
                @if(count($translations) === count($languages))
                    <div class="flex items-center gap-2 mt-2">
                        <i data-lucide="check-circle" class="w-4 h-4 text-green-600"></i>
                        <span class="text-sm text-green-600 font-medium">{{ admin_trans('all_translations_complete') }}</span>
                    </div>
                @endif
            </div>
        @endif
    </div>

    <!-- Quick Actions -->
    <div class="card">
        <h3 class="text-lg font-bold text-primary mb-4">{{ admin_trans('quick_actions') }}</h3>
        <div class="flex flex-wrap gap-3">
            <x-button 
                variant="outline" 
                icon="edit"
                @click="$dispatch('open-modal', 'edit-country-modal')"
            >
                {{ admin_trans('edit_country_info') }}
            </x-button>
            
            @if($country->iso_code)
                <form method="POST" action="{{ route('admin.countries.update-flag', $country) }}" class="inline">
                    @csrf
                    @method('PATCH')
                    <x-button 
                        type="submit"
                        variant="outline" 
                        icon="refresh-cw"
                    >
                        {{ admin_trans('sync_from_api') }}
                    </x-button>
                </form>
            @endif
            
            <x-button 
                variant="outline" 
                icon="globe"
                href="{{ route('admin.countries.index') }}"
            >
                {{ admin_trans('all_countries') }}
            </x-button>
        </div>
    </div>
</div>

<!-- Quick Edit Country Modal -->
<x-modal name="edit-country-modal" maxWidth="md">
    <div class="p-6">
        <div class="flex items-center justify-between mb-6">
            <h3 class="text-xl font-bold text-primary">{{ admin_trans('quick_edit_country') }}</h3>
            <button @click="$dispatch('close-modal', 'edit-country-modal')" 
                    class="p-2 rounded-lg hover:bg-primary-light transition-colors">
                <i data-lucide="x" class="w-5 h-5 text-tertiary"></i>
            </button>
        </div>

        <form method="POST" action="{{ route('admin.countries.update', $country) }}" class="space-y-4">
            @csrf
            @method('PUT')

            <x-input 
                :label="admin_trans('country_name')"
                name="name"
                value="{{ $country->name }}"
                :required="true"
            />

            <div class="grid grid-cols-2 gap-4">
                <x-input 
                    :label="admin_trans('country_code')"
                    name="code"
                    value="{{ $country->code }}"
                    :required="true"
                />

                <x-input 
                    :label="admin_trans('iso_code')"
                    name="iso_code"
                    value="{{ $country->iso_code }}"
                />
            </div>

            <x-input 
                :label="admin_trans('flag_url')"
                name="flag_url"
                type="url"
                value="{{ $country->flag_url }}"
            />

            <!-- Hidden fields to preserve other data -->
            <input type="hidden" name="iso2" value="{{ $country->iso2 }}">
            <input type="hidden" name="dial_code" value="{{ $country->dial_code }}">
            <input type="hidden" name="currency" value="{{ $country->currency }}">
            <input type="hidden" name="currency_code" value="{{ $country->currency_code }}">
            <input type="hidden" name="currency_symbol" value="{{ $country->currency_symbol }}">
            <input type="hidden" name="order" value="{{ $country->order }}">
            <input type="hidden" name="status" value="{{ $country->status }}">
            <input type="hidden" name="is_default" value="{{ $country->is_default ? '1' : '0' }}">

            <div class="flex gap-3 justify-end pt-4 border-t" style="border-color: var(--color-border);">
                <x-button 
                    variant="outline" 
                    type="button"
                    @click="$dispatch('close-modal', 'edit-country-modal')"
                >
                    {{ admin_trans('cancel') }}
                </x-button>
                <x-button variant="primary" type="submit" icon="check">
                    {{ admin_trans('update') }}
                </x-button>
            </div>
        </form>
    </div>
</x-modal>

@endsection

@push('scripts')
<script>
// Auto-save translations after successful submission
document.addEventListener('DOMContentLoaded', function() {
    // Show success message for successful translations
    @if(session('success'))
        setTimeout(() => {
            location.reload();
        }, 1500);
    @endif
});
</script>
@endpush