@php
    $title = admin_trans('user_growth_chart');
    $description = 'Track user registration growth over the last 12 months';
    $icon = 'trending-up';
    $stats = [
        [
            'label' => 'Total Users',
            'value' => number_format(array_sum(array_column($monthlyData, 'count'))),
            'icon' => 'users',
            'color' => 'primary'
        ],
        [
            'label' => 'This Month',
            'value' => number_format(end($monthlyData)['count']),
            'icon' => 'user-plus',
            'color' => 'success'
        ],
        [
            'label' => 'Average/Month',
            'value' => number_format(array_sum(array_column($monthlyData, 'count')) / count($monthlyData)),
            'icon' => 'activity',
            'color' => 'secondary'
        ],
        [
            'label' => 'Peak Month',
            'value' => max(array_column($monthlyData, 'count')),
            'icon' => 'zap',
            'color' => 'warning'
        ],
    ];
@endphp

@extends('admin.analytics.base')

@section('analytics-content')

<!-- Growth Chart -->
<div class="mb-6">
    <canvas id="growthChart" height="80"></canvas>
</div>

<!-- Monthly Breakdown Table -->
<div class="mt-6">
    <h3 class="text-lg font-bold text-primary mb-4">Monthly Breakdown</h3>
    <x-enhanced-table
        :headers="[
            'month' => [
                'text' => 'Month',
                'align' => 'auto',
            ],
            'users' => [
                'text' => 'New Users',
                'align' => 'center',
            ],
            'trend' => [
                'text' => 'Trend',
                'align' => 'center',
            ]
        ]"
        :responsive="true"
    >
        @foreach($monthlyData as $index => $item)
        <tr class="hover:bg-primary-light transition-colors">
            <x-table-cell type="text">
                <span class="font-medium text-primary">{{ $item['month'] }}</span>
            </x-table-cell>
            <x-table-cell type="center">
                <span class="text-lg font-bold text-primary">{{ number_format($item['count']) }}</span>
            </x-table-cell>
            <x-table-cell type="center">
                @if($index > 0)
                    @php
                        $previous = $monthlyData[$index - 1]['count'];
                        $current = $item['count'];
                        $change = $previous > 0 ? round((($current - $previous) / $previous) * 100, 1) : 0;
                    @endphp
                    <div class="flex items-center justify-center gap-1">
                        @if($change > 0)
                            <i data-lucide="trending-up" class="w-4 h-4 text-success-500"></i>
                            <span class="text-sm font-medium text-success-500">+{{ $change }}%</span>
                        @elseif($change < 0)
                            <i data-lucide="trending-down" class="w-4 h-4 text-danger-500"></i>
                            <span class="text-sm font-medium text-danger-500">{{ $change }}%</span>
                        @else
                            <i data-lucide="minus" class="w-4 h-4 text-secondary"></i>
                            <span class="text-sm text-secondary">0%</span>
                        @endif
                    </div>
                @else
                    <span class="text-sm text-tertiary">-</span>
                @endif
            </x-table-cell>
        </tr>
        @endforeach
    </x-enhanced-table>
</div>

@endsection

@push('scripts')
<script src="https://cdn.jsdelivr.net/npm/chart.js@4"></script>
<script>
document.addEventListener('DOMContentLoaded', function() {
    const ctx = document.getElementById('growthChart').getContext('2d');

    const data = @json($monthlyData);
    const labels = data.map(item => item.month);
    const values = data.map(item => item.count);

    new Chart(ctx, {
        type: 'line',
        data: {
            labels: labels,
            datasets: [{
                label: 'New Users',
                data: values,
                borderColor: '#16CC62',
                backgroundColor: 'rgba(22, 204, 98, 0.1)',
                borderWidth: 3,
                fill: true,
                tension: 0.4,
                pointRadius: 4,
                pointHoverRadius: 6,
                pointBackgroundColor: '#16CC62',
                pointBorderColor: '#fff',
                pointBorderWidth: 2,
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: true,
            plugins: {
                legend: {
                    display: false
                },
                title: {
                    display: true,
                    text: 'User Growth Over Last 12 Months',
                    font: {
                        size: 16,
                        weight: 'bold'
                    }
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        precision: 0
                    }
                }
            }
        }
    });

    // Re-initialize Lucide icons
    lucide.createIcons();
});
</script>
@endpush
