import './bootstrap';
import Alpine from 'alpinejs';
import collapse from '@alpinejs/collapse';
import { Toast } from './components/toast';

// Register Alpine plugins
Alpine.plugin(collapse);

// Initialize Alpine before starting
window.Alpine = Alpine;

// Theme Store - must be registered before Alpine.start()
document.addEventListener('alpine:init', () => {
    Alpine.store('theme', {
        mode: localStorage.getItem('admin_theme') || 'light',
        
        init() {
            this.apply();
        },
        
        toggle() {
            this.mode = this.mode === 'light' ? 'dark' : 'light';
            localStorage.setItem('admin_theme', this.mode);
            this.apply();
        },
        
        apply() {
            if (this.mode === 'dark') {
                document.documentElement.classList.add('dark');
            } else {
                document.documentElement.classList.remove('dark');
            }
        }
    });

    // Sidebar Store
    Alpine.store('sidebar', {
        open: localStorage.getItem('sidebar_open') !== 'false',
        
        toggle() {
            this.open = !this.open;
            localStorage.setItem('sidebar_open', this.open);
        }
    });

    // Language Store
    Alpine.store('language', {
        current: document.documentElement.getAttribute('lang') || 'ar',
        
        toggle() {
            const newLang = this.current === 'ar' ? 'en' : 'ar';
            window.location.href = `/admin/locale/${newLang}`;
        }
    });

    // Modal Helper
    Alpine.magic('modal', () => {
        return {
            open(name) {
                window.dispatchEvent(new CustomEvent('open-modal', { detail: name }));
            },
            close(name) {
                window.dispatchEvent(new CustomEvent('close-modal', { detail: name }));
            }
        };
    });
});

// Initialize Toast System
window.toastManager = new Toast();

// Expose toast helpers globally
window.toast = {
    success: (message, title) => window.toastManager.success(message, title),
    error: (message, title) => window.toastManager.error(message, title),
    warning: (message, title) => window.toastManager.warning(message, title),
    info: (message, title) => window.toastManager.info(message, title),
};

// Handle Laravel flash messages
document.addEventListener('DOMContentLoaded', () => {
    // Success messages
    const successMessage = document.querySelector('meta[name="flash-success"]');
    if (successMessage) {
        toast.success(successMessage.content);
    }

    // Error messages
    const errorMessage = document.querySelector('meta[name="flash-error"]');
    if (errorMessage) {
        toast.error(errorMessage.content);
    }

    // Warning messages
    const warningMessage = document.querySelector('meta[name="flash-warning"]');
    if (warningMessage) {
        toast.warning(warningMessage.content);
    }

    // Info messages
    const infoMessage = document.querySelector('meta[name="flash-info"]');
    if (infoMessage) {
        toast.info(infoMessage.content);
    }

    // Initialize Lucide icons after Alpine loads
    setTimeout(() => {
        if (typeof lucide !== 'undefined') {
            lucide.createIcons();
        }
    }, 100);
});

// Re-initialize Lucide icons on Alpine updates
document.addEventListener('alpine:initialized', () => {
    Alpine.effect(() => {
        setTimeout(() => {
            if (typeof lucide !== 'undefined') {
                lucide.createIcons();
            }
        }, 50);
    });
});

// Start Alpine
Alpine.start();