<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Cache;

class ClarityService
{
    protected $projectId;
    protected $baseUrl = 'https://www.clarity.ms';

    public function __construct()
    {
        $this->projectId = config('services.clarity.project_id');
    }

    /**
     * Track page view for mobile app
     */
    public function trackPageView($data)
    {
        try {
            $payload = [
                'project_id' => $this->projectId,
                'session_id' => $data['session_id'] ?? $this->generateSessionId(),
                'user_id' => $data['user_id'] ?? null,
                'page_url' => $data['page_url'],
                'page_title' => $data['page_title'] ?? '',
                'platform' => $data['platform'] ?? 'mobile',
                'device_type' => $data['device_type'] ?? 'mobile',
                'app_version' => $data['app_version'] ?? '1.0.0',
                'timestamp' => now()->toISOString(),
                'custom_data' => $data['custom_data'] ?? [],
            ];

            // Store locally for batch sending
            $this->storeEvent('pageview', $payload);

            return [
                'success' => true,
                'session_id' => $payload['session_id'],
            ];

        } catch (\Exception $e) {
            Log::error('Clarity tracking error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Track custom event for mobile app
     */
    public function trackEvent($eventName, $data)
    {
        try {
            $payload = [
                'project_id' => $this->projectId,
                'session_id' => $data['session_id'] ?? $this->generateSessionId(),
                'user_id' => $data['user_id'] ?? null,
                'event_name' => $eventName,
                'event_data' => $data['event_data'] ?? [],
                'platform' => $data['platform'] ?? 'mobile',
                'timestamp' => now()->toISOString(),
            ];

            $this->storeEvent('custom_event', $payload);

            return [
                'success' => true,
                'event_id' => uniqid('evt_'),
            ];

        } catch (\Exception $e) {
            Log::error('Clarity event tracking error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Track user interaction (tap, scroll, etc.)
     */
    public function trackInteraction($data)
    {
        try {
            $payload = [
                'project_id' => $this->projectId,
                'session_id' => $data['session_id'],
                'interaction_type' => $data['type'], // tap, scroll, swipe, etc.
                'element_id' => $data['element_id'] ?? null,
                'element_class' => $data['element_class'] ?? null,
                'coordinates' => $data['coordinates'] ?? null,
                'timestamp' => now()->toISOString(),
            ];

            $this->storeEvent('interaction', $payload);

            return ['success' => true];

        } catch (\Exception $e) {
            Log::error('Clarity interaction tracking error: ' . $e->getMessage());
            return ['success' => false];
        }
    }

    /**
     * Get analytics data (if available through API)
     */
    public function getAnalytics($startDate, $endDate, $filters = [])
    {
        try {
            // Note: Clarity doesn't have a public API for retrieving data
            // This would require manual export or third-party tools
            
            $cacheKey = "clarity_analytics_{$startDate}_{$endDate}_" . md5(serialize($filters));
            
            return Cache::remember($cacheKey, 3600, function () use ($startDate, $endDate, $filters) {
                // This would be replaced with actual API call if available
                return [
                    'success' => true,
                    'data' => [
                        'page_views' => $this->getStoredPageViews($startDate, $endDate),
                        'events' => $this->getStoredEvents($startDate, $endDate),
                        'sessions' => $this->getStoredSessions($startDate, $endDate),
                    ],
                    'period' => [
                        'start' => $startDate,
                        'end' => $endDate,
                    ],
                ];
            });

        } catch (\Exception $e) {
            Log::error('Clarity analytics error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }

    /**
     * Generate session ID
     */
    private function generateSessionId()
    {
        return 'session_' . uniqid() . '_' . time();
    }

    /**
     * Store event locally for batch processing
     */
    private function storeEvent($type, $payload)
    {
        $key = "clarity_events_" . date('Y-m-d');
        $events = Cache::get($key, []);
        
        $events[] = [
            'type' => $type,
            'data' => $payload,
            'stored_at' => now()->toISOString(),
        ];

        Cache::put($key, $events, 86400); // Store for 24 hours
    }

    /**
     * Get stored page views (from local cache)
     */
    private function getStoredPageViews($startDate, $endDate)
    {
        $views = [];
        $current = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);

        while ($current <= $end) {
            $key = "clarity_events_" . $current->format('Y-m-d');
            $events = Cache::get($key, []);
            
            foreach ($events as $event) {
                if ($event['type'] === 'pageview') {
                    $views[] = $event['data'];
                }
            }
            
            $current->addDay();
        }

        return $views;
    }

    /**
     * Get stored custom events
     */
    private function getStoredEvents($startDate, $endDate)
    {
        $events = [];
        $current = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);

        while ($current <= $end) {
            $key = "clarity_events_" . $current->format('Y-m-d');
            $dailyEvents = Cache::get($key, []);
            
            foreach ($dailyEvents as $event) {
                if ($event['type'] === 'custom_event') {
                    $events[] = $event['data'];
                }
            }
            
            $current->addDay();
        }

        return $events;
    }

    /**
     * Get session statistics
     */
    private function getStoredSessions($startDate, $endDate)
    {
        $sessions = [];
        $current = \Carbon\Carbon::parse($startDate);
        $end = \Carbon\Carbon::parse($endDate);

        while ($current <= $end) {
            $key = "clarity_events_" . $current->format('Y-m-d');
            $events = Cache::get($key, []);
            
            $sessionIds = collect($events)
                ->pluck('data.session_id')
                ->unique()
                ->values()
                ->toArray();

            $sessions = array_merge($sessions, $sessionIds);
            $current->addDay();
        }

        return [
            'total_sessions' => count(array_unique($sessions)),
            'unique_sessions' => array_unique($sessions),
        ];
    }

    /**
     * Batch send events to Clarity (if API becomes available)
     */
    public function batchSendEvents()
    {
        try {
            $yesterday = now()->subDay()->format('Y-m-d');
            $key = "clarity_events_{$yesterday}";
            $events = Cache::get($key, []);

            if (empty($events)) {
                return ['success' => true, 'message' => 'No events to send'];
            }

            // Here you would send to Clarity API if available
            // For now, just log for debugging
            Log::info('Clarity events to send: ' . count($events));

            // Clear sent events
            Cache::forget($key);

            return [
                'success' => true,
                'events_sent' => count($events),
            ];

        } catch (\Exception $e) {
            Log::error('Clarity batch send error: ' . $e->getMessage());
            return ['success' => false, 'error' => $e->getMessage()];
        }
    }
}