<?php

namespace App\Http\Resources;

use Illuminate\Http\Request;
use Illuminate\Http\Resources\Json\JsonResource;

class OfferResource extends JsonResource
{
    /**
     * Transform the resource into an array.
     *
     * @return array<string, mixed>
     */
    public function toArray(Request $request): array
    {
        $locale = $this->getLocaleFromRequest($request);
        
        return [
            'id' => $this->id,
            'title' => $this->getTitle($locale['locale_id']),
            'description' => $this->getDescription($locale['locale_id']),
            'type' => $this->type,
            'code' => $this->code,
            'promotion_type' => $this->promotion_type,
            'discount_value' => $this->discount_value,
            'regular_price' => $this->regular_price,
            'sale_price' => $this->sale_price,
            'currency' => $this->currency,
            'is_featured' => $this->is_featured,
            'is_exclusive' => $this->is_exclusive,
            'is_free_shipping' => $this->is_free_shipping,
            'is_new_customer' => $this->is_new_customer,
            'expires_at' => $this->expires_at?->toISOString(),
            'is_expired' => $this->is_expired,
            'days_until_expiry' => $this->when(
                $this->expires_at,
                fn() => now()->diffInDays($this->expires_at, false)
            ),
            'image' => $this->getFirstMediaUrl('images'),
            'gallery' => $this->when(
                $this->whenLoaded('media') && $this->hasMedia('images'),
                fn() => $this->getMedia('images')->map(fn($media) => $media->getUrl())
            ),
            'store' => $this->when($this->relationLoaded('countryStore'), [
                'id' => $this->countryStore->id,
                'name' => $this->countryStore->store->getName($locale['locale_id']),
                'logo' => $this->countryStore->store->getFirstMediaUrl('logo'),
            ]),
            'brand' => $this->when($this->relationLoaded('brand') && $this->brand, [
                'id' => $this->brand->id,
                'name' => $this->brand->getName($locale['language_id']),
                'logo' => $this->brand->getFirstMediaUrl('logo'),
            ]),
            'categories' => CategoryResource::collection($this->whenLoaded('categories')),
            'stats' => [
                'favorites_count' => $this->favorites_count,
                'uses_count' => $this->uses_count,
                'views_count' => $this->views_count,
                'popularity' => $this->popularity,
            ],
            'savings' => $this->when($this->regular_price && $this->sale_price, [
                'amount' => $this->saving_amount,
                'percentage' => $this->saving_percentage,
            ]),
            'is_favorited' => $this->when(
                $request->user(), 
                fn() => $request->user()->isSubscribedTo($this->resource, 'favorite')
            ),
            'created_at' => $this->created_at->toISOString(),
        ];
    }

    /**
     * Get locale from request attributes
     */
    private function getLocaleFromRequest(Request $request): array
    {
        return [
            'country_id' => $request->attributes->get('country_id'),
            'language_id' => $request->attributes->get('language_id'),
            'locale_id' => $request->attributes->get('locale_id'),
        ];
    }
}