<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Str;

class Notification extends Model
{
    use HasFactory;

    protected $keyType = 'string';
    public $incrementing = false;

    protected $fillable = [
        'id',
        'type',
        'notifiable_type',
        'notifiable_id',
        'title',
        'message',
        'data',
        'channels',
        'status',
        'priority',
        'read_at',
    ];

    protected $casts = [
        'data' => 'array',
        'channels' => 'array',
        'priority' => 'integer',
        'read_at' => 'datetime',
    ];

    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($notification) {
            if (empty($notification->id)) {
                $notification->id = (string) Str::uuid();
            }
        });
    }

    // Relations
    public function notifiable()
    {
        return $this->morphTo();
    }

    public function deliveries()
    {
        return $this->hasMany(NotificationDelivery::class);
    }

    public function targets()
    {
        return $this->hasMany(NotificationTarget::class);
    }

    // Scopes
    public function scopePending($query)
    {
        return $query->where('status', 'pending');
    }

    public function scopeSent($query)
    {
        return $query->where('status', 'sent');
    }

    public function scopeFailed($query)
    {
        return $query->where('status', 'failed');
    }

    public function scopeRead($query)
    {
        return $query->whereNotNull('read_at');
    }

    public function scopeUnread($query)
    {
        return $query->whereNull('read_at');
    }

    public function scopeByType($query, $type)
    {
        return $query->where('type', $type);
    }

    public function scopeHighPriority($query, $minPriority = 5)
    {
        return $query->where('priority', '>=', $minPriority);
    }

    // Accessors
    public function getIsReadAttribute()
    {
        return !is_null($this->read_at);
    }

    public function getIsPendingAttribute()
    {
        return $this->status === 'pending';
    }

    public function getIsSentAttribute()
    {
        return $this->status === 'sent';
    }

    public function getIsFailedAttribute()
    {
        return $this->status === 'failed';
    }

    // Helper Methods
    public function markAsRead()
    {
        if (is_null($this->read_at)) {
            $this->update(['read_at' => now()]);
        }
    }

    public function markAsUnread()
    {
        if (!is_null($this->read_at)) {
            $this->update(['read_at' => null]);
        }
    }

    public function markAsSent()
    {
        $this->update(['status' => 'sent']);
    }

    public function markAsFailed()
    {
        $this->update(['status' => 'failed']);
    }

    public function addTarget($targetType, $targetId = null)
    {
        return $this->targets()->create([
            'target_type' => $targetType,
            'target_id' => $targetId,
        ]);
    }
}