<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Relations\Pivot;

class CountryStore extends Pivot
{
    use HasFactory;

    protected $table = 'country_store';

    protected $fillable = [
        'country_id',
        'store_id',
        'user_id',
        'status',
        'is_featured',
        'order',
    ];

    protected $casts = [
        'is_featured' => 'boolean',
        'order' => 'integer',
    ];

    // Relations
    public function country()
    {
        return $this->belongsTo(Country::class);
    }

    public function store()
    {
        return $this->belongsTo(Store::class);
    }

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function stats()
    {
        return $this->hasOne(CountryStoreStats::class, 'country_store_id', 'id');
    }

    public function offers()
    {
        return $this->hasMany(Offer::class, 'country_store_id', 'id');
    }

    // ✅ FIX: تحديد الـ foreign keys بشكل صريح
    public function categories()
    {
        return $this->belongsToMany(
            Category::class,
            'category_country_store',      // pivot table name
            'country_store_id',             // foreign key في الـ pivot table
            'category_id'                   // related key في الـ pivot table
        );
    }

    public function subscriptions()
    {
        return $this->morphMany(Subscription::class, 'subscribable');
    }

    // Scopes
    public function scopeActive($query)
    {
        return $query->where('status', 'active');
    }

    public function scopeFeatured($query)
    {
        return $query->where('is_featured', true);
    }

    public function scopeOrdered($query)
    {
        return $query->orderBy('order')->orderBy('id');
    }

    public function scopeByCountry($query, $countryId)
    {
        return $query->where('country_id', $countryId);
    }

    public function scopeByStore($query, $storeId)
    {
        return $query->where('store_id', $storeId);
    }

    // Accessors
    public function getIsActiveAttribute()
    {
        return $this->status === 'active';
    }

    // Helper Methods
    public function getOrCreateStats()
    {
        return $this->stats ?: $this->stats()->create([
            'followers_count' => 0,
            'uses_count' => 0,
            'views_count' => 0,
            'impressions_count' => 0,
            'popularity' => 0,
        ]);
    }

    public function incrementStat($stat, $amount = 1)
    {
        $stats = $this->getOrCreateStats();
        $stats->increment($stat, $amount);
        return $stats;
    }
}