<?php

namespace App\Jobs;

use App\Models\Notification;
use App\Models\NotificationDelivery;
use App\Models\UserDevice;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class SendNotificationJob implements ShouldQueue
{
    use Dispatchable, InteractsWithQueue, Queueable, SerializesModels;

    public $tries = 3;
    public $backoff = [10, 30, 60]; // Retry after 10s, 30s, 60s

    protected $notification;

    public function __construct(Notification $notification)
    {
        $this->notification = $notification;
    }

    public function handle()
    {
        try {
            Log::info('Processing notification job', [
                'notification_id' => $this->notification->id,
                'type' => $this->notification->type,
                'channels' => $this->notification->channels,
            ]);

            // Get all target users
            $targetUsers = $this->getTargetUsers();
            
            if ($targetUsers->isEmpty()) {
                Log::warning('No target users found for notification', [
                    'notification_id' => $this->notification->id,
                ]);
                
                $this->notification->update(['status' => 'failed']);
                return;
            }

            $successCount = 0;
            $failureCount = 0;

            // Process each user
            foreach ($targetUsers as $user) {
                try {
                    $this->sendToUser($user);
                    $successCount++;
                } catch (\Exception $e) {
                    Log::error('Failed to send notification to user', [
                        'notification_id' => $this->notification->id,
                        'user_id' => $user->id,
                        'error' => $e->getMessage(),
                    ]);
                    $failureCount++;
                }
            }

            // Update notification status
            if ($successCount > 0 && $failureCount === 0) {
                $this->notification->update(['status' => 'sent']);
            } elseif ($successCount > 0) {
                $this->notification->update(['status' => 'sent']); // Partial success
            } else {
                $this->notification->update(['status' => 'failed']);
            }

            Log::info('Notification job completed', [
                'notification_id' => $this->notification->id,
                'success_count' => $successCount,
                'failure_count' => $failureCount,
            ]);

        } catch (\Exception $e) {
            Log::error('Notification job failed', [
                'notification_id' => $this->notification->id,
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            $this->notification->update(['status' => 'failed']);
            throw $e;
        }
    }

    protected function getTargetUsers()
    {
        $userIds = collect();

        foreach ($this->notification->targets as $target) {
            $targetUserIds = $target->getTargetUsers()->pluck('id');
            $userIds = $userIds->merge($targetUserIds);
        }

        // Remove duplicates and get users
        return \App\Models\User::whereIn('id', $userIds->unique())->get();
    }

    protected function sendToUser($user)
    {
        foreach ($this->notification->channels as $channel) {
            $delivery = NotificationDelivery::firstOrCreate([
                'notification_id' => $this->notification->id,
                'user_id' => $user->id,
                'channel' => $channel,
            ], [
                'status' => 'pending',
            ]);

            try {
                switch ($channel) {
                    case 'database':
                        $this->sendDatabaseNotification($user, $delivery);
                        break;
                    
                    case 'push':
                        $this->sendPushNotification($user, $delivery);
                        break;
                    
                    case 'email':
                        $this->sendEmailNotification($user, $delivery);
                        break;
                }

                $delivery->markAsSent();

            } catch (\Exception $e) {
                $delivery->markAsFailed();
                throw $e;
            }
        }
    }

    protected function sendDatabaseNotification($user, $delivery)
    {
        // Database notifications are already stored
        // Just mark as sent
        Log::debug('Database notification stored', [
            'user_id' => $user->id,
            'notification_id' => $this->notification->id,
        ]);
    }

    protected function sendPushNotification($user, $delivery)
    {
        // Get user's devices with FCM tokens
        $devices = UserDevice::where('user_id', $user->id)
            ->whereNotNull('fcm_token')
            ->get();

        if ($devices->isEmpty()) {
            Log::info('No FCM tokens found for user', [
                'user_id' => $user->id,
            ]);
            return;
        }

        $fcmTokens = $devices->pluck('fcm_token')->toArray();
        
        // Prepare FCM payload
        $payload = [
            'registration_ids' => $fcmTokens,
            'notification' => [
                'title' => $this->notification->title,
                'body' => $this->notification->message,
                'icon' => $this->notification->data['image_url'] ?? null,
                'sound' => $this->notification->data['sound'] ?? 'default',
                'badge' => $this->notification->data['badge'] ?? null,
            ],
            'data' => [
                'notification_id' => $this->notification->id,
                'type' => $this->notification->type,
                'priority' => $this->notification->priority,
                'action_url' => $this->notification->data['action_url'] ?? null,
                'custom_data' => json_encode($this->notification->data),
            ],
            'priority' => $this->notification->priority >= 8 ? 'high' : 'normal',
        ];

        // Send to FCM
        $response = Http::withHeaders([
            'Authorization' => 'key=' . config('services.fcm.server_key'),
            'Content-Type' => 'application/json',
        ])->post('https://fcm.googleapis.com/fcm/send', $payload);

        if (!$response->successful()) {
            throw new \Exception('FCM request failed: ' . $response->body());
        }

        $fcmResponse = $response->json();
        
        Log::info('FCM notification sent', [
            'user_id' => $user->id,
            'notification_id' => $this->notification->id,
            'success_count' => $fcmResponse['success'] ?? 0,
            'failure_count' => $fcmResponse['failure'] ?? 0,
        ]);

        // Handle invalid tokens
        if (isset($fcmResponse['results'])) {
            foreach ($fcmResponse['results'] as $index => $result) {
                if (isset($result['error'])) {
                    $token = $fcmTokens[$index];
                    
                    // Remove invalid tokens
                    if (in_array($result['error'], ['InvalidRegistration', 'NotRegistered'])) {
                        UserDevice::where('fcm_token', $token)->update(['fcm_token' => null]);
                        Log::info('Removed invalid FCM token', [
                            'token' => substr($token, 0, 20) . '...',
                            'error' => $result['error'],
                        ]);
                    }
                }
            }
        }
    }

    protected function sendEmailNotification($user, $delivery)
    {
        // إذا المستخدم guest أو ما عندوش email، ما نرسلش
        if ($user->is_guest || !$user->email) {
            Log::info('Skipping email notification for guest or user without email', [
                'user_id' => $user->id,
                'is_guest' => $user->is_guest,
                'has_email' => !empty($user->email),
            ]);
            return;
        }

        // يمكن استخدام Laravel Mail هنا
        // Mail::to($user->email)->send(new NotificationMail($this->notification));
        
        Log::info('Email notification sent', [
            'user_id' => $user->id,
            'email' => $user->email,
            'notification_id' => $this->notification->id,
        ]);
    }

    public function failed(\Throwable $exception)
    {
        Log::error('Notification job permanently failed', [
            'notification_id' => $this->notification->id,
            'error' => $exception->getMessage(),
            'attempts' => $this->attempts(),
        ]);

        $this->notification->update(['status' => 'failed']);
    }
}