<?php

namespace App\Http\Controllers\Api;

use App\Models\User;
use App\Models\UserDevice;
use App\Models\Country;
use App\Models\Language;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use libphonenumber\PhoneNumberUtil;
use libphonenumber\PhoneNumberFormat;
use libphonenumber\PhoneNumberType;
use Dedoc\Scramble\Attributes\HeaderParameter;
use Dedoc\Scramble\Attributes\BodyParameter;

/**
 * @tags 04. User Management
 */
class UserController extends ApiController
{
    /**
     * Get User Profile
     * 
     * Get current authenticated user profile with relationships.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function profile(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // Load relationships
        $user->load(['country', 'currentCountry', 'language', 'devices']);

        $locale = $this->getCurrentLocale($request);

        return $this->successResponse([
            'user' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'phone' => $user->phone,
                'country_code' => $user->country_code,
                'e164_number' => $user->e164_number,
                'country_id' => $user->country_id,
                'language_id' => $user->language_id,
                'current_country_id' => $user->current_country_id,
                'role' => $user->role,
                'is_guest' => $user->is_guest,
                'is_verified' => $user->is_verified,
                'is_banned' => $user->is_banned,
                'created_at' => $user->created_at,
                'updated_at' => $user->updated_at,
                
                // Relationships
                'country' => $user->country ? [
                    'id' => $user->country->id,
                    'name' => $user->country->name,
                    'code' => $user->country->code,
                    'flag_emoji' => $user->country->flag_emoji,
                ] : null,
                
                'current_country' => $user->currentCountry ? [
                    'id' => $user->currentCountry->id,
                    'name' => $user->currentCountry->name,
                    'code' => $user->currentCountry->code,
                    'flag_emoji' => $user->currentCountry->flag_emoji,
                ] : null,
                
                'language' => $user->language ? [
                    'id' => $user->language->id,
                    'name' => $user->language->name,
                    'native_name' => $user->language->native_name,
                    'code' => $user->language->code,
                ] : null,
                
                'devices_count' => $user->devices->count(),
                'active_devices_count' => $user->devices->filter(function ($device) {
                    return $device->isActive();
                })->count(),
            ],
            'locale' => $locale,
        ], 'Profile retrieved successfully');
    }

    /**
     * Update User Profile
     * 
     * Update user profile information. Guest users have limited fields.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('name', 'User full name (regular users only)', required: false, type: 'string', example: 'John Doe')]
    #[BodyParameter('email', 'User email address (regular users only)', required: false, type: 'string', example: 'john@example.com')]
    #[BodyParameter('phone', 'User phone number (regular users only)', required: false, type: 'string', example: '01234567890')]
    #[BodyParameter('country_code', 'Country code for phone number (required with phone)', required: false, type: 'string', example: 'EG')]
    #[BodyParameter('country_id', 'Country ID for localization', required: false, type: 'integer', example: 1)]
    #[BodyParameter('language_id', 'Language ID for localization', required: false, type: 'integer', example: 1)]
    public function updateProfile(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // Different validation rules for guest and regular users
        $rules = [
            'country_id' => 'nullable|exists:countries,id',
            'language_id' => 'nullable|exists:languages,id',
        ];

        if (!$user->is_guest) {
            $rules = array_merge($rules, [
                'name' => 'nullable|string|max:255',
                'email' => 'nullable|email|unique:users,email,' . $user->id,
                'phone' => 'nullable|string|max:20',
                'country_code' => 'nullable|string|max:10',
            ]);
        }

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        return $this->executeTransaction(function () use ($request, $user) {
            $data = [];

            // Handle country and language updates for all users
            if ($request->filled('country_id')) {
                $country = Country::find($request->country_id);
                if ($country && $country->is_active) {
                    $data['country_id'] = $country->id;
                    $data['current_country_id'] = $country->id;
                }
            }

            if ($request->filled('language_id')) {
                $language = Language::find($request->language_id);
                if ($language && $language->is_active) {
                    $data['language_id'] = $language->id;
                }
            }

            // Handle profile updates for regular users only
            if (!$user->is_guest) {
                if ($request->filled('name')) {
                    $data['name'] = $request->name;
                }

                if ($request->filled('email')) {
                    $data['email'] = $request->email;
                }

                // Handle phone number with proper formatting
                if ($request->filled('phone')) {
                    $phone = $request->phone;
                    $countryCode = $request->country_code;

                    if (!$countryCode) {
                        return $this->errorResponse('Country code is required when updating phone number', 400);
                    }

                    try {
                        $phoneUtil = PhoneNumberUtil::getInstance();
                        $proto = $phoneUtil->parse($phone, $countryCode);

                        if (!$phoneUtil->isValidNumber($proto)) {
                            return $this->errorResponse('Invalid phone number', 400);
                        }

                        $numberType = $phoneUtil->getNumberType($proto);
                        if (!in_array($numberType, [PhoneNumberType::MOBILE, PhoneNumberType::FIXED_LINE_OR_MOBILE])) {
                            return $this->errorResponse('Phone number must be mobile', 400);
                        }

                        $e164 = $phoneUtil->format($proto, PhoneNumberFormat::E164);
                        $nationalFormat = $phoneUtil->format($proto, PhoneNumberFormat::NATIONAL);

                        // Check if phone number already exists
                        $existingUser = User::where('e164_number', $e164)
                            ->where('id', '!=', $user->id)
                            ->first();

                        if ($existingUser) {
                            return $this->errorResponse('Phone number already exists', 409);
                        }

                        $data['phone'] = trim($nationalFormat);
                        $data['e164_number'] = $e164;
                        $data['country_code'] = $countryCode;

                    } catch (\Exception $e) {
                        return $this->errorResponse('Invalid phone number format', 400);
                    }
                }
            }

            // Update user
            $user->update($data);

            $this->logActivity($user, 'profile_updated', $data);

            $user->refresh();
            $user->load(['country', 'currentCountry', 'language']);

            return [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'country_id' => $user->country_id,
                    'language_id' => $user->language_id,
                    'current_country_id' => $user->current_country_id,
                    'role' => $user->role,
                    'is_guest' => $user->is_guest,
                    'is_verified' => $user->is_verified,
                    'updated_at' => $user->updated_at,
                ],
            ];
        }, 'Profile updated successfully');
    }

    /**
     * Change Password
     * 
     * Change user password. Regular users only.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('old_password', 'Current password', required: true, type: 'string', example: 'oldpassword123')]
    #[BodyParameter('new_password', 'New password (min 6 characters)', required: true, type: 'string', example: 'newpassword123')]
    #[BodyParameter('new_password_confirmation', 'Confirm new password', required: true, type: 'string', example: 'newpassword123')]
    public function changePassword(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        if ($user->is_guest) {
            return $this->errorResponse('Guest users cannot change password', 403);
        }

        $validator = Validator::make($request->all(), [
            'old_password' => 'required|string',
            'new_password' => 'required|string|min:6|confirmed',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        // Verify old password
        if (!Hash::check($request->old_password, $user->password)) {
            return $this->errorResponse('Current password is incorrect', 400);
        }

        // Update password - بدون Hash::make() لأن الـ model هيعمله تلقائياً
        $user->update([
            'password' => $request->new_password,
        ]);

        // Optional: Delete all other tokens لإجبار إعادة تسجيل الدخول من الأجهزة الأخرى
        // $user->tokens()->where('id', '!=', $user->currentAccessToken()->id)->delete();

        $this->logActivity($user, 'password_changed');

        return $this->successResponse(null, 'Password changed successfully');
    }

    /**
     * Upgrade Guest Account
     * 
     * Upgrade guest account to regular user account.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('name', 'User full name', required: true, type: 'string', example: 'John Doe')]
    #[BodyParameter('email', 'User email address (optional if phone provided)', required: false, type: 'string', example: 'john@example.com')]
    #[BodyParameter('phone', 'User phone number (optional if email provided)', required: false, type: 'string', example: '01234567890')]
    #[BodyParameter('country_code', 'Country code for phone number (required with phone)', required: false, type: 'string', example: 'EG')]
    #[BodyParameter('password', 'User password (min 6 characters)', required: true, type: 'string', example: 'password123')]
    #[BodyParameter('password_confirmation', 'Confirm password', required: true, type: 'string', example: 'password123')]
    public function upgradeAccount(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        if (!$user->is_guest) {
            return $this->errorResponse('User is already a regular account', 400);
        }

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'nullable|email|unique:users,email',
            'phone' => 'nullable|string|max:20',
            'country_code' => 'nullable|string|max:10',
            'password' => 'required|string|min:6|confirmed',
        ]);

        // Ensure at least email or phone is provided
        $validator->after(function ($validator) use ($request) {
            $hasEmail = !empty(trim($request->email));
            $hasPhone = !empty(trim($request->phone));
            
            if (!$hasEmail && !$hasPhone) {
                $validator->errors()->add('email', 'Either email or phone number is required');
            }
        });

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        return $this->executeTransaction(function () use ($request, $user) {
            $data = [
                'name' => $request->name,
                'password' => $request->password, // الـ model هيعمل hash
                'is_guest' => false,
            ];

            // Handle email
            if (!empty(trim($request->email))) {
                $data['email'] = trim($request->email);
            }

            // Handle phone number with proper formatting
            if (!empty(trim($request->phone))) {
                $phone = trim($request->phone);
                $countryCode = $request->country_code;

                if (!$countryCode) {
                    return $this->errorResponse('Country code is required when providing phone number', 400);
                }

                try {
                    $phoneUtil = PhoneNumberUtil::getInstance();
                    $proto = $phoneUtil->parse($phone, $countryCode);

                    if (!$phoneUtil->isValidNumber($proto)) {
                        return $this->errorResponse('Invalid phone number', 400);
                    }

                    $numberType = $phoneUtil->getNumberType($proto);
                    if (!in_array($numberType, [PhoneNumberType::MOBILE, PhoneNumberType::FIXED_LINE_OR_MOBILE])) {
                        return $this->errorResponse('Phone number must be mobile', 400);
                    }

                    $e164 = $phoneUtil->format($proto, PhoneNumberFormat::E164);
                    $nationalFormat = $phoneUtil->format($proto, PhoneNumberFormat::NATIONAL);

                    // Check if phone number already exists
                    $existingUser = User::where('e164_number', $e164)
                        ->where('id', '!=', $user->id)
                        ->first();

                    if ($existingUser) {
                        return $this->errorResponse('Phone number already exists', 409);
                    }

                    $data['phone'] = trim($nationalFormat);
                    $data['e164_number'] = $e164;
                    $data['country_code'] = $countryCode;

                } catch (\Exception $e) {
                    return $this->errorResponse('Invalid phone number format', 400);
                }
            }

            // Update user
            $user->update($data);

            // Create new token with user abilities
            $user->tokens()->delete(); // Delete guest tokens
            $token = $user->createToken('user-token', ['user'])->plainTextToken;

            $this->logActivity($user, 'account_upgraded', [
                'from_guest' => true,
                'new_email' => $data['email'] ?? null,
                'new_phone' => $data['phone'] ?? null,
            ]);

            $user->refresh();
            $user->load(['country', 'currentCountry', 'language']);

            return [
                'user' => [
                    'id' => $user->id,
                    'name' => $user->name,
                    'email' => $user->email,
                    'phone' => $user->phone,
                    'country_id' => $user->country_id,
                    'language_id' => $user->language_id,
                    'current_country_id' => $user->current_country_id,
                    'role' => $user->role,
                    'is_guest' => $user->is_guest,
                    'is_verified' => $user->is_verified,
                    'upgraded_at' => $user->updated_at,
                ],
                'token' => $token,
            ];
        }, 'Account upgraded successfully');
    }

    /**
     * Get User Devices
     * 
     * Get list of all registered devices for current user.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function devices(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // Get current token info
        $currentToken = $request->user()->currentAccessToken();
        $currentDeviceId = $this->extractDeviceIdFromToken($currentToken);

        $devices = $user->devices()
            ->orderBy('updated_at', 'desc')
            ->get()
            ->map(function ($device) use ($currentDeviceId, $user) {
                // Check if this device has active tokens
                $activeTokens = $user->tokens()
                    ->where('name', 'like', '%-' . $device->device_id)
                    ->count();

                return [
                    'id' => $device->id,
                    'device_id' => $device->device_id,
                    'platform' => $device->platform,
                    'app_version' => $device->app_version,
                    'ip_country' => $device->ip_country,
                    'last_ip' => $device->last_ip,
                    'last_seen' => $device->updated_at,
                    'created_at' => $device->created_at,
                    'is_current' => $device->device_id === $currentDeviceId,
                    'is_active' => $activeTokens > 0,
                    'active_sessions' => $activeTokens,
                ];
            });

        return $this->successResponse([
            'devices' => $devices,
            'total_devices' => $devices->count(),
            'active_devices' => $devices->where('is_active', true)->count(),
            'current_device' => $devices->where('is_current', true)->first(),
        ], 'Devices retrieved successfully');
    }

    /**
     * Remove Device
     * 
     * Remove specific device from user account.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function removeDevice(Request $request, $deviceId): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $device = $user->devices()->where('device_id', $deviceId)->first();

        if (!$device) {
            return $this->errorResponse('Device not found', 404);
        }

        // Delete all tokens for this specific device
        $deletedTokens = $user->tokens()
            ->where('name', 'like', '%-' . $deviceId)
            ->delete();

        // Remove device
        $device->delete();

        $this->logActivity($user, 'device_removed', [
            'device_id' => $deviceId,
            'platform' => $device->platform,
            'deleted_sessions' => $deletedTokens,
        ]);

        return $this->successResponse([
            'device_id' => $deviceId,
            'deleted_sessions' => $deletedTokens,
        ], 'Device removed successfully');
    }

    /**
     * Logout from Specific Device
     * 
     * Logout from specific device without removing device info.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function logoutDevice(Request $request, $deviceId): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        // Delete all tokens for this specific device
        $deletedTokens = $user->tokens()
            ->where('name', 'like', '%-' . $deviceId)
            ->delete();

        if ($deletedTokens === 0) {
            return $this->errorResponse('No active sessions found for this device', 404);
        }

        $this->logActivity($user, 'device_logged_out', [
            'device_id' => $deviceId,
            'deleted_sessions' => $deletedTokens,
        ]);

        return $this->successResponse([
            'device_id' => $deviceId,
            'logged_out_sessions' => $deletedTokens,
        ], 'Device logged out successfully');
    }

    /**
     * Delete Account
     * 
     * Delete user account permanently with all associated data.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication', required: true, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    #[BodyParameter('password', 'Current password (required for regular users)', required: false, type: 'string', example: 'password123')]
    #[BodyParameter('confirm_deletion', 'Must be true to confirm deletion', required: true, type: 'boolean', example: true)]
    public function deleteAccount(Request $request): JsonResponse
    {
        $user = $request->user();
        
        if (!$user) {
            return $this->errorResponse('User not authenticated', 401);
        }

        $validator = Validator::make($request->all(), [
            'password' => $user->is_guest ? 'nullable' : 'required|string',
            'confirm_deletion' => 'required|boolean|accepted',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        // Verify password for non-guest users
        if (!$user->is_guest && !Hash::check($request->password, $user->password)) {
            return $this->errorResponse('Password is incorrect', 400);
        }

        return $this->executeTransaction(function () use ($user) {
            $userId = $user->id;
            $isGuest = $user->is_guest;

            // Delete all tokens
            $user->tokens()->delete();

            // Delete related data
            $user->devices()->delete();
            
            // Log before deletion
            $this->logActivity($user, 'account_deleted', [
                'user_id' => $userId,
                'is_guest' => $isGuest,
                'deleted_at' => now(),
            ]);

            // Delete user
            $user->delete();

            return [
                'deleted' => true,
                'deleted_at' => now()->toISOString(),
                'user_id' => $userId,
            ];
        }, 'Account deleted successfully');
    }

    /**
     * Extract device ID from token name
     */
    private function extractDeviceIdFromToken($token): ?string
    {
        if (!$token || !$token->name) {
            return null;
        }
        
        // Extract device_id from token name (format: user-token-device123)
        $parts = explode('-', $token->name);
        return count($parts) >= 3 ? end($parts) : null;
    }
}


