<?php

namespace App\Http\Controllers\Api;

use App\Models\CountryStore;
use App\Models\Offer;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Dedoc\Scramble\Attributes\HeaderParameter;

/**
 * @tags 05. Stores
 */
class StoreController extends ApiController
{
    /**
     * Get Stores List
     * 
     * Get all active stores in user's current country with localized names.
     * Stores are filtered by country and displayed in user's preferred language.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function index(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'featured' => 'nullable|boolean',
            'category_id' => 'nullable|exists:categories,id',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:50',
            'search' => 'nullable|string|max:100',
            'sort' => 'nullable|in:name,featured,offers_count,popularity,followers',
            'order' => 'nullable|in:asc,desc',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);
        $perPage = $request->input('per_page', 20);
        $search = $request->input('search');
        $featured = $request->input('featured');
        $categoryId = $request->input('category_id');
        $sort = $request->input('sort', 'featured');
        $order = $request->input('order', 'desc');

        // Build query
        $query = CountryStore::query()
            ->where('country_id', $locale['country_id'])
            ->active()
            ->with([
                'store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'store',
                'stats'
            ])
            ->withCount([
                'offers' => function ($q) {
                    $q->active()->notExpired();
                }
            ]);

        // Apply category filter
        if ($categoryId) {
            $query->whereHas('categories', function ($q) use ($categoryId) {
                $q->where('categories.id', $categoryId);
            });
        }

        // Apply featured filter
        if ($featured !== null) {
            $query->where('is_featured', $featured);
        }

        // Apply search filter
        if ($search) {
            $query->whereHas('store.translations', function ($q) use ($search, $locale) {
                $q->where('locale_id', $locale['locale_id'])
                    ->where(function ($sq) use ($search) {
                        $sq->where('name', 'like', "%{$search}%")
                            ->orWhere('description', 'like', "%{$search}%");
                    });
            });
        }

        // Apply sorting
        switch ($sort) {
            case 'name':
                $query->join('stores', 'country_store.store_id', '=', 'stores.id')
                    ->join('store_translations', function ($join) use ($locale) {
                        $join->on('stores.id', '=', 'store_translations.store_id')
                            ->where('store_translations.locale_id', $locale['locale_id']);
                    })
                    ->orderBy('store_translations.name', $order);
                break;
            case 'offers_count':
                $query->orderBy('offers_count', $order);
                break;
            case 'popularity':
                $query->leftJoin('country_store_stats', 'country_store.id', '=', 'country_store_stats.country_store_id')
                    ->orderBy('country_store_stats.popularity', $order);
                break;
            case 'followers':
                $query->leftJoin('country_store_stats', 'country_store.id', '=', 'country_store_stats.country_store_id')
                    ->orderBy('country_store_stats.followers_count', $order);
                break;
            default: // featured
                $query->orderBy('is_featured', 'desc')
                    ->orderBy('order', 'asc')
                    ->orderBy('id', 'asc');
        }

        $stores = $query->select('country_store.*')->paginate($perPage);

        // Transform data
        $transformedStores = $stores->getCollection()->map(function ($countryStore) use ($locale, $request) {
            $isSubscribed = false;
            if ($request->user()) {
                $isSubscribed = $request->user()->isSubscribedTo($countryStore, 'follow');
            }

            return [
                'id' => $countryStore->id,
                'store_id' => $countryStore->store_id,
                'name' => $countryStore->store->getName($locale['locale_id']),
                'description' => $countryStore->store->getDescription($locale['locale_id']),
                'is_featured' => $countryStore->is_featured,
                'is_active' => $countryStore->is_active,
                'order' => $countryStore->order,
                'logo' => $countryStore->store->getFirstMediaUrl('logo'),
                'offers_count' => $countryStore->offers_count,
                'stats' => [
                    'followers_count' => $countryStore->stats->followers_count ?? 0,
                    'popularity' => $countryStore->stats->popularity ?? 0,
                    'avg_discount' => round($countryStore->stats->avg_discount ?? 0, 2),
                    'max_discount' => round($countryStore->stats->max_discount ?? 0, 2),
                ],
                'is_subscribed' => $isSubscribed,
            ];
        });

        return $this->resourceResponse([
            'stores' => $transformedStores,
            'pagination' => [
                'current_page' => $stores->currentPage(),
                'last_page' => $stores->lastPage(),
                'per_page' => $stores->perPage(),
                'total' => $stores->total(),
                'from' => $stores->firstItem(),
                'to' => $stores->lastItem(),
            ],
            'filters' => [
                'featured' => $featured,
                'category_id' => $categoryId,
                'search' => $search,
                'sort' => $sort,
                'order' => $order,
                'country_id' => $locale['country_id'],
                'locale_id' => $locale['locale_id'],
            ],
        ], 'Stores retrieved successfully');
    }

    /**
     * Get Store Details
     * 
     * Get detailed information about a specific store including statistics and categories.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function show(Request $request, $storeId): JsonResponse
    {
        $locale = $this->getCurrentLocale($request);

        $countryStore = CountryStore::query()
            ->where('id', $storeId)
            ->where('country_id', $locale['country_id'])
            ->active()
            ->with([
                'store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'store',
                'stats',
                'categories.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ])
            ->withCount([
                'offers' => function ($q) {
                    $q->active()->notExpired();
                }
            ])
            ->first();

        if (!$countryStore) {
            return $this->errorResponse('Store not found', 404);
        }

        // Get recent offers
        $recentOffers = $countryStore->offers()
            ->active()
            ->notExpired()
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ])
            ->orderBy('created_at', 'desc')
            ->limit(5)
            ->get();

        // Check if user is subscribed and get subscription details
        $isSubscribed = false;
        $subscriptionId = null;
        if ($request->user()) {
            $subscription = $request->user()->subscriptions()
                ->where('subscribable_type', CountryStore::class)
                ->where('subscribable_id', $storeId)
                ->first();

            if ($subscription) {
                $isSubscribed = true;
                $subscriptionId = $subscription->id;
            }
        }

        return $this->successResponse([
            'store' => [
                'id' => $countryStore->id,
                'store_id' => $countryStore->store_id,
                'name' => $countryStore->store->getName($locale['locale_id']),
                'description' => $countryStore->store->getDescription($locale['locale_id']),
                'is_featured' => $countryStore->is_featured,
                'is_active' => $countryStore->is_active,
                'order' => $countryStore->order,
                'logo' => $countryStore->store->getFirstMediaUrl('logo'),
                'gallery' => $countryStore->store->getMedia('gallery')->map(function ($media) {
                    return $media->getUrl();
                }),
                'offers_count' => $countryStore->offers_count,
                'stats' => [
                    'followers_count' => $countryStore->stats->followers_count ?? 0,
                    'uses_count' => $countryStore->stats->uses_count ?? 0,
                    'views_count' => $countryStore->stats->views_count ?? 0,
                    'popularity' => $countryStore->stats->popularity ?? 0,
                    'avg_discount' => round($countryStore->stats->avg_discount ?? 0, 2),
                    'max_discount' => round($countryStore->stats->max_discount ?? 0, 2),
                ],
                'categories' => $countryStore->categories->map(function ($category) use ($locale) {
                    return [
                        'id' => $category->id,
                        'name' => $category->getName($locale['language_id']),
                        'icon' => $category->getFirstMediaUrl('icon'),
                    ];
                }),
                'is_subscribed' => $isSubscribed,
                'subscription_id' => $subscriptionId, // 🔥 NEW
            ],
            'recent_offers' => $recentOffers->map(function ($offer) use ($locale) {
                return [
                    'id' => $offer->id,
                    'title' => $offer->getTitle($locale['locale_id']),
                    'type' => $offer->type,
                    'discount_value' => $offer->discount_value,
                    'is_featured' => $offer->is_featured,
                    'expires_at' => $offer->expires_at?->toISOString(),
                    'image' => $offer->getFirstMediaUrl('images'),
                    'brand' => $offer->brand ? [
                        'id' => $offer->brand->id,
                        'name' => $offer->brand->getName($locale['language_id']),
                    ] : null,
                ];
            }),
            'locale' => $locale,
        ], 'Store details retrieved successfully');
    }
    /**
     * Get Store Offers
     * 
     * Get all active offers for a specific store.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function offers(Request $request, $storeId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'featured' => 'nullable|boolean',
            'exclusive' => 'nullable|boolean',
            'free_shipping' => 'nullable|boolean',
            'new_customer' => 'nullable|boolean',
            'min_discount' => 'nullable|numeric|min:0|max:100',
            'type' => 'nullable|in:coupon,deal,cashback',
            'promotion_type' => 'nullable|in:percentage,fixed,bogo,gift',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:50',
            'search' => 'nullable|string|max:100',
            'sort' => 'nullable|in:newest,discount,popularity,expiry',
            'order' => 'nullable|in:asc,desc',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);

        // Check if store exists
        $countryStore = CountryStore::where('id', $storeId)
            ->where('country_id', $locale['country_id'])
            ->active()
            ->with(['store'])
            ->first();

        if (!$countryStore) {
            return $this->errorResponse('Store not found', 404);
        }

        $perPage = $request->input('per_page', 20);
        $search = $request->input('search');
        $sort = $request->input('sort', 'newest');
        $order = $request->input('order', 'desc');

        // Build query for offers
        $query = $countryStore->offers()
            ->active()
            ->notExpired()
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                },
                'categories.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ]);

        // Apply filters
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->boolean('featured'));
        }

        if ($request->filled('exclusive')) {
            $query->where('is_exclusive', $request->boolean('exclusive'));
        }

        if ($request->filled('free_shipping')) {
            $query->where('is_free_shipping', $request->boolean('free_shipping'));
        }

        if ($request->filled('new_customer')) {
            $query->where('is_new_customer', $request->boolean('new_customer'));
        }

        if ($request->filled('min_discount')) {
            $query->where('discount_value', '>=', $request->input('min_discount'));
        }

        if ($request->filled('type')) {
            $query->where('type', $request->input('type'));
        }

        if ($request->filled('promotion_type')) {
            $query->where('promotion_type', $request->input('promotion_type'));
        }

        // Apply search filter
        if ($search) {
            $query->whereHas('translations', function ($q) use ($search, $locale) {
                $q->where('locale_id', $locale['locale_id'])
                    ->where(function ($sq) use ($search) {
                        $sq->where('title', 'like', "%{$search}%")
                            ->orWhere('description', 'like', "%{$search}%");
                    });
            });
        }

        // Apply sorting
        switch ($sort) {
            case 'discount':
                $query->orderBy('discount_value', $order);
                break;
            case 'popularity':
                $query->orderBy('popularity', $order);
                break;
            case 'expiry':
                $query->orderByRaw('expires_at IS NULL, expires_at ' . $order);
                break;
            default: // newest
                $query->orderBy('created_at', $order);
        }

        $offers = $query->paginate($perPage);

        // Transform data
        $transformedOffers = $offers->getCollection()->map(function ($offer) use ($locale, $request) {
            $isSubscribed = false;
            if ($request->user()) {
                $isSubscribed = $request->user()->isSubscribedTo($offer, 'favorite');
            }

            return [
                'id' => $offer->id,
                'title' => $offer->getTitle($locale['locale_id']),
                'description' => $offer->getDescription($locale['locale_id']),
                'type' => $offer->type,
                'code' => $offer->code,
                'promotion_type' => $offer->promotion_type,
                'discount_value' => $offer->discount_value,
                'regular_price' => $offer->regular_price,
                'sale_price' => $offer->sale_price,
                'currency' => $offer->currency,
                'is_featured' => $offer->is_featured,
                'is_exclusive' => $offer->is_exclusive,
                'is_free_shipping' => $offer->is_free_shipping,
                'is_new_customer' => $offer->is_new_customer,
                'expires_at' => $offer->expires_at?->toISOString(),
                'is_expired' => $offer->is_expired,
                'image' => $offer->getFirstMediaUrl('images'),
                'brand' => $offer->brand ? [
                    'id' => $offer->brand->id,
                    'name' => $offer->brand->getName($locale['language_id']),
                    'logo' => $offer->brand->getFirstMediaUrl('logo'),
                ] : null,
                'categories' => $offer->categories->map(function ($category) use ($locale) {
                    return [
                        'id' => $category->id,
                        'name' => $category->getName($locale['language_id']),
                    ];
                }),
                'stats' => [
                    'favorites_count' => $offer->favorites_count,
                    'uses_count' => $offer->uses_count,
                    'views_count' => $offer->views_count,
                ],
                'is_favorited' => $isSubscribed,
            ];
        });

        return $this->resourceResponse([
            'store' => [
                'id' => $countryStore->id,
                'name' => $countryStore->store->getName($locale['locale_id']),
                'logo' => $countryStore->store->getFirstMediaUrl('logo'),
            ],
            'offers' => $transformedOffers,
            'pagination' => [
                'current_page' => $offers->currentPage(),
                'last_page' => $offers->lastPage(),
                'per_page' => $offers->perPage(),
                'total' => $offers->total(),
                'from' => $offers->firstItem(),
                'to' => $offers->lastItem(),
            ],
            'filters' => [
                'featured' => $request->input('featured'),
                'exclusive' => $request->input('exclusive'),
                'free_shipping' => $request->input('free_shipping'),
                'new_customer' => $request->input('new_customer'),
                'min_discount' => $request->input('min_discount'),
                'type' => $request->input('type'),
                'promotion_type' => $request->input('promotion_type'),
                'search' => $search,
                'sort' => $sort,
                'order' => $order,
                'store_id' => $storeId,
                'country_id' => $locale['country_id'],
                'locale_id' => $locale['locale_id'],
            ],
        ], 'Store offers retrieved successfully');
    }
}
