<?php

namespace App\Http\Controllers\Api;

use App\Models\Offer;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Dedoc\Scramble\Attributes\HeaderParameter;

/**
 * @tags 07. Offers
 */
class OfferController extends ApiController
{
    /**
     * Get Offers List
     * 
     * Get all active offers in user's current country with advanced filtering.
     * Offers are localized based on user's language and filtered by country.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function index(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'featured' => 'nullable|boolean',
            'exclusive' => 'nullable|boolean',
            'free_shipping' => 'nullable|boolean',
            'new_customer' => 'nullable|boolean',
            'min_discount' => 'nullable|numeric|min:0|max:100',
            'max_discount' => 'nullable|numeric|min:0|max:100',
            'type' => 'nullable|in:coupon,deal,cashback',
            'promotion_type' => 'nullable|in:percentage,fixed,bogo,gift',
            'category_id' => 'nullable|exists:categories,id',
            'brand_id' => 'nullable|exists:brands,id',
            'store_id' => 'nullable|exists:country_store,id',
            'page' => 'nullable|integer|min:1',
            'per_page' => 'nullable|integer|min:1|max:50',
            'search' => 'nullable|string|max:100',
            'sort' => 'nullable|in:newest,oldest,discount_high,discount_low,popularity,expiry_soon,expiry_late',
            'expires_within_days' => 'nullable|integer|min:1|max:365',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);
        $perPage = $request->input('per_page', 20);
        $search = $request->input('search');
        $sort = $request->input('sort', 'newest');

        // Build query
        $query = Offer::query()
            ->active()
            ->notExpired()
            ->whereHas('countryStore', function ($q) use ($locale) {
                $q->where('country_id', $locale['country_id'])
                    ->active();
            })
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store',
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                },
                'categories.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ]);

        // Apply filters
        if ($request->filled('featured')) {
            $query->where('is_featured', $request->boolean('featured'));
        }

        if ($request->filled('exclusive')) {
            $query->where('is_exclusive', $request->boolean('exclusive'));
        }

        if ($request->filled('free_shipping')) {
            $query->where('is_free_shipping', $request->boolean('free_shipping'));
        }

        if ($request->filled('new_customer')) {
            $query->where('is_new_customer', $request->boolean('new_customer'));
        }

        if ($request->filled('min_discount')) {
            $query->where('discount_value', '>=', $request->input('min_discount'));
        }

        if ($request->filled('max_discount')) {
            $query->where('discount_value', '<=', $request->input('max_discount'));
        }

        if ($request->filled('type')) {
            $query->where('type', $request->input('type'));
        }

        if ($request->filled('promotion_type')) {
            $query->where('promotion_type', $request->input('promotion_type'));
        }

        if ($request->filled('category_id')) {
            $query->whereHas('categories', function ($q) use ($request) {
                $q->where('categories.id', $request->input('category_id'));
            });
        }

        if ($request->filled('brand_id')) {
            $query->where('brand_id', $request->input('brand_id'));
        }

        if ($request->filled('store_id')) {
            $query->where('country_store_id', $request->input('store_id'));
        }

        if ($request->filled('expires_within_days')) {
            $days = $request->input('expires_within_days');
            $query->where('expires_at', '<=', now()->addDays($days))
                ->where('expires_at', '>', now());
        }

        // Apply search filter
        if ($search) {
            $query->where(function ($q) use ($search, $locale) {
                $q->whereHas('translations', function ($sq) use ($search, $locale) {
                    $sq->where('locale_id', $locale['locale_id'])
                        ->where(function ($ssq) use ($search) {
                            $ssq->where('title', 'like', "%{$search}%")
                                ->orWhere('description', 'like', "%{$search}%");
                        });
                })
                    ->orWhere('code', 'like', "%{$search}%");
            });
        }

        // Apply sorting
        switch ($sort) {
            case 'oldest':
                $query->orderBy('created_at', 'asc');
                break;
            case 'discount_high':
                $query->orderBy('discount_value', 'desc');
                break;
            case 'discount_low':
                $query->orderBy('discount_value', 'asc');
                break;
            case 'popularity':
                $query->orderBy('popularity', 'desc');
                break;
            case 'expiry_soon':
                $query->orderByRaw('expires_at IS NULL, expires_at ASC');
                break;
            case 'expiry_late':
                $query->orderByRaw('expires_at IS NULL DESC, expires_at DESC');
                break;
            default: // newest
                $query->orderBy('created_at', 'desc');
        }

        $offers = $query->paginate($perPage);

        // Transform data
        $transformedOffers = $offers->getCollection()->map(function ($offer) use ($locale, $request) {
            return $this->transformOffer($offer, $locale, $request);
        });

        return $this->resourceResponse([
            'offers' => $transformedOffers,
            'pagination' => [
                'current_page' => $offers->currentPage(),
                'last_page' => $offers->lastPage(),
                'per_page' => $offers->perPage(),
                'total' => $offers->total(),
                'from' => $offers->firstItem(),
                'to' => $offers->lastItem(),
            ],
            'filters' => [
                'featured' => $request->input('featured'),
                'exclusive' => $request->input('exclusive'),
                'free_shipping' => $request->input('free_shipping'),
                'new_customer' => $request->input('new_customer'),
                'min_discount' => $request->input('min_discount'),
                'max_discount' => $request->input('max_discount'),
                'type' => $request->input('type'),
                'promotion_type' => $request->input('promotion_type'),
                'category_id' => $request->input('category_id'),
                'brand_id' => $request->input('brand_id'),
                'store_id' => $request->input('store_id'),
                'expires_within_days' => $request->input('expires_within_days'),
                'search' => $search,
                'sort' => $sort,
                'country_id' => $locale['country_id'],
                'locale_id' => $locale['locale_id'],
            ],
        ], 'Offers retrieved successfully');
    }

    /**
     * Get Offer Details
     * 
     * Get detailed information about a specific offer including full description and terms.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function show(Request $request, $offerId): JsonResponse
    {
        $locale = $this->getCurrentLocale($request);

        $offer = Offer::query()
            ->where('id', $offerId)
            ->active()
            ->whereHas('countryStore', function ($q) use ($locale) {
                $q->where('country_id', $locale['country_id'])
                    ->active();
            })
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store',
                'countryStore.stats',
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                },
                'categories.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ])
            ->first();

        if (!$offer) {
            return $this->errorResponse('Offer not found or expired', 404);
        }

        // Increment views count
        $offer->incrementViews();

        // Get related offers from same store
        $relatedOffers = Offer::query()
            ->where('country_store_id', $offer->country_store_id)
            ->where('id', '!=', $offer->id)
            ->active()
            ->notExpired()
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ])
            ->orderBy('popularity', 'desc')
            ->limit(5)
            ->get();

        // Check if user is subscribed and get subscription details
        $isSubscribed = false;
        $subscriptionId = null;
        if ($request->user()) {
            $subscription = $request->user()->subscriptions()
                ->where('subscribable_type', Offer::class)
                ->where('subscribable_id', $offerId)
                ->first();

            if ($subscription) {
                $isSubscribed = true;
                $subscriptionId = $subscription->id;
            }
        }

        $transformedOffer = $this->transformOffer($offer, $locale, $request, true);

        // Add subscription_id to the transformed offer
        $transformedOffer['subscription_id'] = $subscriptionId; // 🔥 NEW

        return $this->successResponse([
            'offer' => $transformedOffer,
            'store' => [
                'id' => $offer->countryStore->id,
                'store_id' => $offer->countryStore->store_id,
                'name' => $offer->countryStore->store->getName($locale['locale_id']),
                'description' => $offer->countryStore->store->getDescription($locale['locale_id']),
                'logo' => $offer->countryStore->store->getFirstMediaUrl('logo'),
                'is_featured' => $offer->countryStore->is_featured,
                'stats' => [
                    'followers_count' => $offer->countryStore->stats->followers_count ?? 0,
                    'offers_count' => $offer->countryStore->offers()->active()->notExpired()->count(),
                ],
            ],
            'related_offers' => $relatedOffers->map(function ($relatedOffer) use ($locale, $request) {
                return $this->transformOffer($relatedOffer, $locale, $request);
            }),
            'locale' => $locale,
        ], 'Offer details retrieved successfully');
    }


    /**
     * Track Offer Usage
     * 
     * Track when user uses an offer (clicks on it or copies code).
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function trackUsage(Request $request, $offerId): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'action' => 'required|in:view,click,copy_code,visit_store',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);
        $action = $request->input('action');

        $offer = Offer::query()
            ->where('id', $offerId)
            ->active()
            ->whereHas('countryStore', function ($q) use ($locale) {
                $q->where('country_id', $locale['country_id'])
                    ->active();
            })
            ->first();

        if (!$offer) {
            return $this->errorResponse('Offer not found', 404);
        }

        // Track usage based on action
        switch ($action) {
            case 'view':
                $offer->incrementViews();
                break;
            case 'click':
            case 'copy_code':
            case 'visit_store':
                $offer->incrementUses();
                break;
        }

        // Log activity if user is authenticated
        if ($request->user()) {
            $this->logActivity($request->user(), 'offer_' . $action, [
                'offer_id' => $offer->id,
                'offer_title' => $offer->getTitle($locale['locale_id']),
                'store_id' => $offer->country_store_id,
                'action' => $action,
            ]);
        }

        return $this->successResponse([
            'tracked' => true,
            'action' => $action,
            'offer_id' => $offer->id,
            'stats' => [
                'views_count' => $offer->fresh()->views_count,
                'uses_count' => $offer->fresh()->uses_count,
            ],
        ], 'Offer usage tracked successfully');
    }

    /**
     * Get Featured Offers
     * 
     * Get featured offers in user's country - typically used for homepage.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function featured(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'limit' => 'nullable|integer|min:1|max:20',
            'category_id' => 'nullable|exists:categories,id',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);
        $limit = $request->input('limit', 10);
        $categoryId = $request->input('category_id');

        $query = Offer::query()
            ->active()
            ->notExpired()
            ->featured()
            ->whereHas('countryStore', function ($q) use ($locale) {
                $q->where('country_id', $locale['country_id'])
                    ->active();
            })
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store',
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ]);

        if ($categoryId) {
            $query->whereHas('categories', function ($q) use ($categoryId) {
                $q->where('categories.id', $categoryId);
            });
        }

        $offers = $query->orderBy('popularity', 'desc')
            ->orderBy('created_at', 'desc')
            ->limit($limit)
            ->get();

        $transformedOffers = $offers->map(function ($offer) use ($locale, $request) {
            return $this->transformOffer($offer, $locale, $request);
        });

        return $this->successResponse([
            'featured_offers' => $transformedOffers,
            'total_count' => $offers->count(),
            'filters' => [
                'limit' => $limit,
                'category_id' => $categoryId,
                'country_id' => $locale['country_id'],
                'locale_id' => $locale['locale_id'],
            ],
        ], 'Featured offers retrieved successfully');
    }

    /**
     * Get Expiring Soon Offers
     * 
     * Get offers that are expiring within specified days.
     */
    #[HeaderParameter('Authorization', 'Bearer {token} for authentication (optional)', required: false, type: 'string', example: 'Bearer 1|f832abc...xyz')]
    public function expiringSoon(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'days' => 'nullable|integer|min:1|max:30',
            'limit' => 'nullable|integer|min:1|max:50',
        ]);

        if ($validator->fails()) {
            return $this->validationErrorResponse($validator->errors());
        }

        $locale = $this->getCurrentLocale($request);
        $days = $request->input('days', 7);
        $limit = $request->input('limit', 20);

        $offers = Offer::query()
            ->active()
            ->where('expires_at', '>', now())
            ->where('expires_at', '<=', now()->addDays($days))
            ->whereHas('countryStore', function ($q) use ($locale) {
                $q->where('country_id', $locale['country_id'])
                    ->active();
            })
            ->with([
                'translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store.translations' => function ($q) use ($locale) {
                    $q->where('locale_id', $locale['locale_id']);
                },
                'countryStore.store',
                'brand.translations' => function ($q) use ($locale) {
                    $q->where('language_id', $locale['language_id']);
                }
            ])
            ->orderBy('expires_at', 'asc')
            ->limit($limit)
            ->get();

        $transformedOffers = $offers->map(function ($offer) use ($locale, $request) {
            return $this->transformOffer($offer, $locale, $request);
        });

        return $this->successResponse([
            'expiring_offers' => $transformedOffers,
            'total_count' => $offers->count(),
            'expires_within_days' => $days,
            'country_id' => $locale['country_id'],
            'locale_id' => $locale['locale_id'],
        ], 'Expiring offers retrieved successfully');
    }

    /**
     * Transform offer data for API response
     */
    private function transformOffer($offer, $locale, $request, $includeFullDetails = false)
    {
        $isSubscribed = false;
        if ($request->user()) {
            $isSubscribed = $request->user()->isSubscribedTo($offer, 'favorite');
        }

        $data = [
            'id' => $offer->id,
            'title' => $offer->getTitle($locale['locale_id']),
            'description' => $offer->getDescription($locale['locale_id']),
            'type' => $offer->type,
            'code' => $offer->code,
            'promotion_type' => $offer->promotion_type,
            'discount_value' => $offer->discount_value,
            'regular_price' => $offer->regular_price,
            'sale_price' => $offer->sale_price,
            'currency' => $offer->currency,
            'is_featured' => $offer->is_featured,
            'is_exclusive' => $offer->is_exclusive,
            'is_free_shipping' => $offer->is_free_shipping,
            'is_new_customer' => $offer->is_new_customer,
            'expires_at' => $offer->expires_at?->toISOString(),
            'is_expired' => $offer->is_expired,
            'days_until_expiry' => $offer->expires_at ? now()->diffInDays($offer->expires_at, false) : null,
            'image' => $offer->getFirstMediaUrl('images'),
            'store' => [
                'id' => $offer->countryStore->id,
                'name' => $offer->countryStore->store->getName($locale['locale_id']),
                'logo' => $offer->countryStore->store->getFirstMediaUrl('logo'),
            ],
            'brand' => $offer->brand ? [
                'id' => $offer->brand->id,
                'name' => $offer->brand->getName($locale['language_id']),
                'logo' => $offer->brand->getFirstMediaUrl('logo'),
            ] : null,
            'categories' => $offer->categories->map(function ($category) use ($locale) {
                return [
                    'id' => $category->id,
                    'name' => $category->getName($locale['language_id']),
                    'icon' => $category->getFirstMediaUrl('icon'),
                ];
            }),
            'stats' => [
                'favorites_count' => $offer->favorites_count,
                'uses_count' => $offer->uses_count,
                'views_count' => $offer->views_count,
                'popularity' => $offer->popularity,
            ],
            'is_favorited' => $isSubscribed,
            'created_at' => $offer->created_at->toISOString(),
        ];

        if ($includeFullDetails) {
            $data['saving_amount'] = $offer->saving_amount;
            $data['saving_percentage'] = $offer->saving_percentage;
            $data['gallery'] = $offer->getMedia('images')->map(function ($media) {
                return $media->getUrl();
            });
        }

        return $data;
    }
}
