<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;

class ProfileController extends Controller
{
    /**
     * Show the admin profile page
     */
    public function show()
    {
        $user = Auth::user();
        
        return view('admin.profile.show', compact('user'));
    }

    /**
     * Update admin profile information
     */
    public function updateProfile(Request $request)
    {
        $user = Auth::user();

        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'email' => 'required|email|max:255|unique:users,email,' . $user->id,
        ]);

        if ($validator->fails()) {
            flash_error(admin_trans('validation_failed'));
            return back()->withErrors($validator)->withInput();
        }

        try {
            $user->update([
                'name' => $request->name,
                'email' => $request->email,
            ]);

            flash_success(admin_trans('profile_updated_successfully'));
            return back();

        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_profile'));
            return back()->withInput();
        }
    }

    /**
     * Update admin password
     */
    public function updatePassword(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'current_password' => 'required',
            'password' => ['required', 'confirmed', Password::min(8)],
        ]);

        if ($validator->fails()) {
            flash_error(admin_trans('validation_failed'));
            return back()->withErrors($validator, 'password');
        }

        $user = Auth::user();

        // Check if current password is correct
        if (!Hash::check($request->current_password, $user->password)) {
            flash_error(admin_trans('current_password_incorrect'));
            return back()->withErrors(['current_password' => admin_trans('current_password_incorrect')], 'password');
        }

        try {
            $user->update([
                'password' => Hash::make($request->password),
            ]);

            flash_success(admin_trans('password_updated_successfully'));
            return back();

        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_password'));
            return back();
        }
    }

    /**
     * Get admin profile data (AJAX)
     */
    public function getProfileData()
    {
        $user = Auth::user();

        return response()->json([
            'success' => true,
            'data' => [
                'id' => $user->id,
                'name' => $user->name,
                'email' => $user->email,
                'role' => $user->role,
                'created_at' => $user->created_at->format('Y-m-d H:i:s'),
                'email_verified_at' => $user->email_verified_at ? $user->email_verified_at->format('Y-m-d H:i:s') : null,
                'last_login' => $user->updated_at->format('Y-m-d H:i:s'), // Assuming updated_at tracks last activity
            ]
        ]);
    }

    /**
     * Update admin preferences/settings
     */
    public function updatePreferences(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'language_id' => 'nullable|exists:languages,id',
            'country_id' => 'nullable|exists:countries,id',
        ]);

        if ($validator->fails()) {
            flash_error(admin_trans('validation_failed'));
            return back()->withErrors($validator);
        }

        try {
            $user = Auth::user();
            
            $user->update([
                'language_id' => $request->language_id,
                'country_id' => $request->country_id,
            ]);

            flash_success(admin_trans('preferences_updated_successfully'));
            return back();

        } catch (\Exception $e) {
            flash_error(admin_trans('error_updating_preferences'));
            return back();
        }
    }
}