<?php

namespace App\Http\Controllers\Admin\Analytics;

use App\Http\Controllers\Controller;
use App\Models\User;
use App\Models\Country;
use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class UsersAnalyticsController extends Controller
{
    /**
     * Users by Role
     */
    public function byRole()
    {
        $data = User::select('role', DB::raw('count(*) as total'))
            ->groupBy('role')
            ->get();

        return view('admin.analytics.users.by-role', compact('data'));
    }

    /**
     * Users by Country
     */
    public function byCountry()
    {
        $data = User::select('countries.name as country_name', DB::raw('count(users.id) as total'))
            ->leftJoin('user_locales', 'users.id', '=', 'user_locales.user_id')
            ->leftJoin('locales', 'user_locales.locale_id', '=', 'locales.id')
            ->leftJoin('countries', 'locales.country_id', '=', 'countries.id')
            ->groupBy('countries.id', 'countries.name')
            ->orderByDesc('total')
            ->get();

        return view('admin.analytics.users.by-country', compact('data'));
    }

    /**
     * Users by Language
     */
    public function byLanguage()
    {
        $data = User::select('languages.name as language_name', 'languages.native_name', DB::raw('count(users.id) as total'))
            ->leftJoin('user_locales', 'users.id', '=', 'user_locales.user_id')
            ->leftJoin('locales', 'user_locales.locale_id', '=', 'locales.id')
            ->leftJoin('languages', 'locales.language_id', '=', 'languages.id')
            ->groupBy('languages.id', 'languages.name', 'languages.native_name')
            ->orderByDesc('total')
            ->get();

        return view('admin.analytics.users.by-language', compact('data'));
    }

    /**
     * User Growth Chart (last 12 months)
     */
    public function growthChart()
    {
        $monthlyData = [];
        for ($i = 11; $i >= 0; $i--) {
            $date = Carbon::now()->subMonths($i);
            $count = User::whereYear('created_at', $date->year)
                ->whereMonth('created_at', $date->month)
                ->count();

            $monthlyData[] = [
                'month' => $date->format('M Y'),
                'count' => $count,
            ];
        }

        return view('admin.analytics.users.growth-chart', compact('monthlyData'));
    }

    /**
     * Active Users
     */
    public function active()
    {
        $data = User::whereNull('banned_at')
            ->orderByDesc('created_at')
            ->paginate(20);

        $totalActive = User::whereNull('banned_at')->count();

        return view('admin.analytics.users.active', compact('data', 'totalActive'));
    }

    /**
     * Banned Users
     */
    public function banned()
    {
        $data = User::whereNotNull('banned_at')
            ->orderByDesc('banned_at')
            ->paginate(20);

        $totalBanned = User::whereNotNull('banned_at')->count();

        return view('admin.analytics.users.banned', compact('data', 'totalBanned'));
    }

    /**
     * Guest Users
     */
    public function guests()
    {
        $data = User::where('is_guest', true)
            ->orderByDesc('created_at')
            ->paginate(20);

        $totalGuests = User::where('is_guest', true)->count();

        return view('admin.analytics.users.guests', compact('data', 'totalGuests'));
    }
}
