<?php

namespace App\Http\Controllers\Admin\Analytics;

use App\Http\Controllers\Controller;
use App\Models\Store;
use App\Models\Country;
use App\Models\CountryStore;
use App\Models\Locale;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class StoresAnalyticsController extends Controller
{
    /**
     * Stores by Country
     */
    public function byCountry()
    {
        $data = DB::table('country_store')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->select(
                'countries.name as country_name',
                DB::raw('count(DISTINCT country_store.store_id) as total')
            )
            ->groupBy('countries.id', 'countries.name')
            ->orderByDesc('total')
            ->get();

        return view('admin.analytics.stores.by-country', compact('data'));
    }

    /**
     * Featured Stores
     */
    public function featured()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = DB::table('country_store')
            ->join('stores', 'country_store.store_id', '=', 'stores.id')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->leftJoin('country_store_stats', 'country_store.id', '=', 'country_store_stats.country_store_id')
            ->leftJoin(
                DB::raw('(SELECT country_store_id, COUNT(*) as offers_count FROM offers GROUP BY country_store_id) as offer_counts'),
                'country_store.id',
                '=',
                'offer_counts.country_store_id'
            )
            ->where('country_store.is_featured', true)
            ->where('country_store.status', 'active')
            ->select(
                'stores.id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as name'),
                'countries.name as country_name',
                DB::raw('COALESCE(offer_counts.offers_count, 0) as offers_count'),
                DB::raw('COALESCE(country_store_stats.followers_count, 0) as followers_count')
            )
            ->orderByDesc('offers_count')
            ->paginate(20);

        return view('admin.analytics.stores.featured', compact('data'));
    }

    /**
     * Stores by Followers Count
     */
    public function byFollowers()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = DB::table('country_store')
            ->join('stores', 'country_store.store_id', '=', 'stores.id')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->leftJoin('country_store_stats', 'country_store.id', '=', 'country_store_stats.country_store_id')
            ->where('country_store.status', 'active')
            ->select(
                'stores.id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as name'),
                'countries.name as country_name',
                DB::raw('COALESCE(country_store_stats.followers_count, 0) as followers_count')
            )
            ->orderByDesc('country_store_stats.followers_count')
            ->paginate(20);

        return view('admin.analytics.stores.by-followers', compact('data'));
    }

    /**
     * Stores by Offers Count
     */
    public function byOffersCount()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = DB::table('country_store')
            ->join('stores', 'country_store.store_id', '=', 'stores.id')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->leftJoin(
                DB::raw('(SELECT country_store_id, COUNT(*) as offers_count FROM offers GROUP BY country_store_id) as offer_counts'),
                'country_store.id',
                '=',
                'offer_counts.country_store_id'
            )
            ->select(
                'stores.id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as name'),
                'countries.name as country_name',
                DB::raw('COALESCE(offer_counts.offers_count, 0) as offers_count')
            )
            ->orderByDesc('offers_count')
            ->paginate(20);

        return view('admin.analytics.stores.by-offers-count', compact('data'));
    }

    /**
     * Store Performance Report
     */
    public function performance()
    {
        $totalStores = Store::count();
        $activeStores = CountryStore::where('status', 'active')->distinct('store_id')->count('store_id');
        $featuredStores = CountryStore::where('is_featured', true)->distinct('store_id')->count('store_id');
        $totalFollowers = DB::table('country_store_stats')->sum('followers_count');

        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $topStores = DB::table('country_store')
            ->join('stores', 'country_store.store_id', '=', 'stores.id')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->leftJoin('country_store_stats', 'country_store.id', '=', 'country_store_stats.country_store_id')
            ->leftJoin(
                DB::raw('(SELECT country_store_id, COUNT(*) as offers_count FROM offers GROUP BY country_store_id) as offer_counts'),
                'country_store.id',
                '=',
                'offer_counts.country_store_id'
            )
            ->where('country_store.status', 'active')
            ->select(
                'stores.id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as name'),
                'countries.name as country_name',
                DB::raw('COALESCE(country_store_stats.followers_count, 0) as followers_count'),
                DB::raw('COALESCE(offer_counts.offers_count, 0) as offers_count')
            )
            ->orderByDesc('country_store_stats.followers_count')
            ->limit(10)
            ->get();

        $storesByCountry = DB::table('country_store')
            ->join('countries', 'country_store.country_id', '=', 'countries.id')
            ->select(
                'countries.name as country_name',
                DB::raw('count(DISTINCT country_store.store_id) as total')
            )
            ->groupBy('countries.id', 'countries.name')
            ->orderByDesc('total')
            ->limit(10)
            ->get();

        return view('admin.analytics.stores.performance', compact(
            'totalStores',
            'activeStores',
            'featuredStores',
            'totalFollowers',
            'topStores',
            'storesByCountry'
        ));
    }
}
