<?php

namespace App\Http\Controllers\Admin\Analytics;

use App\Http\Controllers\Controller;
use App\Models\Offer;
use App\Models\Store;
use App\Models\Brand;
use App\Models\Category;
use App\Models\CountryStore;
use App\Models\Locale;
use App\Models\Language;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class OffersAnalyticsController extends Controller
{
    /**
     * Offers by Status
     */
    public function byStatus()
    {
        $data = Offer::select('status', DB::raw('count(*) as total'))
            ->groupBy('status')
            ->get();

        return view('admin.analytics.offers.by-status', compact('data'));
    }

    /**
     * Offers by Type
     */
    public function byType()
    {
        $data = Offer::select('type', DB::raw('count(*) as total'))
            ->groupBy('type')
            ->get();

        return view('admin.analytics.offers.by-type', compact('data'));
    }

    /**
     * Offers by Store
     */
    public function byStore()
    {
        // Get default locale
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = DB::table('offers')
            ->join('country_store', 'offers.country_store_id', '=', 'country_store.id')
            ->join('stores', 'country_store.store_id', '=', 'stores.id')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->select(
                'stores.id as store_id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as store_name'),
                DB::raw('count(offers.id) as total')
            )
            ->groupBy('stores.id', 'store_translations.name')
            ->orderByDesc('total')
            ->paginate(20);

        return view('admin.analytics.offers.by-store', compact('data'));
    }

    /**
     * Offers by Brand
     */
    public function byBrand()
    {
        // Get default language
        $defaultLanguage = Language::where('is_default', true)->first();
        $languageId = $defaultLanguage ? $defaultLanguage->id : 1;

        $data = DB::table('offers')
            ->leftJoin('brands', 'offers.brand_id', '=', 'brands.id')
            ->leftJoin('brand_translations', function($join) use ($languageId) {
                $join->on('brands.id', '=', 'brand_translations.brand_id')
                     ->where('brand_translations.language_id', '=', $languageId);
            })
            ->select(
                DB::raw('COALESCE(brands.id, 0) as brand_id'),
                DB::raw('COALESCE(brand_translations.name, "No Brand") as brand_name'),
                DB::raw('count(offers.id) as total')
            )
            ->groupBy('brands.id', 'brand_translations.name')
            ->orderByDesc('total')
            ->paginate(20);

        return view('admin.analytics.offers.by-brand', compact('data'));
    }

    /**
     * Offers by Category
     */
    public function byCategory()
    {
        // Get default language
        $defaultLanguage = Language::where('is_default', true)->first();
        $languageId = $defaultLanguage ? $defaultLanguage->id : 1;

        $data = DB::table('offers')
            ->join('category_offer', 'offers.id', '=', 'category_offer.offer_id')
            ->join('categories', 'category_offer.category_id', '=', 'categories.id')
            ->leftJoin('category_translations', function($join) use ($languageId) {
                $join->on('categories.id', '=', 'category_translations.category_id')
                     ->where('category_translations.language_id', '=', $languageId);
            })
            ->select(
                'categories.id as category_id',
                DB::raw('COALESCE(category_translations.name, CONCAT("Category #", categories.id)) as category_name'),
                DB::raw('count(offers.id) as total')
            )
            ->groupBy('categories.id', 'category_translations.name')
            ->orderByDesc('total')
            ->paginate(20);

        return view('admin.analytics.offers.by-category', compact('data'));
    }

    /**
     * Offers Expiring Soon (within next 7 days)
     */
    public function expiringSoon()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = Offer::with(['countryStore.store.translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'categories.translations' => function($q) {
                $defaultLang = Language::where('is_default', true)->first();
                $q->where('language_id', $defaultLang ? $defaultLang->id : 1);
            }])
            ->where('expires_at', '>=', now())
            ->where('expires_at', '<=', now()->addDays(7))
            ->where('status', 'active')
            ->orderBy('expires_at', 'asc')
            ->paginate(20);

        return view('admin.analytics.offers.expiring-soon', compact('data', 'localeId'));
    }

    /**
     * Most Popular Offers
     */
    public function mostPopular()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = Offer::with(['countryStore.store.translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'categories.translations' => function($q) {
                $defaultLang = Language::where('is_default', true)->first();
                $q->where('language_id', $defaultLang ? $defaultLang->id : 1);
            }])
            ->where('status', 'active')
            ->orderByDesc('views_count')
            ->orderByDesc('uses_count')
            ->paginate(20);

        return view('admin.analytics.offers.most-popular', compact('data', 'localeId'));
    }

    /**
     * Least Popular Offers
     */
    public function leastPopular()
    {
        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $data = Offer::with(['countryStore.store.translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'translations' => function($q) use ($localeId) {
                $q->where('locale_id', $localeId);
            }, 'categories.translations' => function($q) {
                $defaultLang = Language::where('is_default', true)->first();
                $q->where('language_id', $defaultLang ? $defaultLang->id : 1);
            }])
            ->where('status', 'active')
            ->orderBy('views_count', 'asc')
            ->orderBy('uses_count', 'asc')
            ->paginate(20);

        return view('admin.analytics.offers.least-popular', compact('data', 'localeId'));
    }

    /**
     * Offers Usage Report
     */
    public function usageReport()
    {
        $totalOffers = Offer::count();
        $activeOffers = Offer::where('status', 'active')->count();
        $expiredOffers = Offer::where('expires_at', '<', now())->count();
        $totalViews = Offer::sum('views_count');
        $totalUses = Offer::sum('uses_count');

        $defaultLocale = Locale::where('is_default', true)->first();
        $localeId = $defaultLocale ? $defaultLocale->id : 1;

        $topStores = DB::table('stores')
            ->leftJoin('store_translations', function($join) use ($localeId) {
                $join->on('stores.id', '=', 'store_translations.store_id')
                     ->where('store_translations.locale_id', '=', $localeId);
            })
            ->leftJoin('country_store', 'stores.id', '=', 'country_store.store_id')
            ->leftJoin('offers', 'country_store.id', '=', 'offers.country_store_id')
            ->select(
                'stores.id',
                DB::raw('COALESCE(store_translations.name, CONCAT("Store #", stores.id)) as name'),
                DB::raw('count(DISTINCT offers.id) as offers_count')
            )
            ->groupBy('stores.id', 'store_translations.name')
            ->orderByDesc('offers_count')
            ->limit(10)
            ->get();

        return view('admin.analytics.offers.usage-report', compact(
            'totalOffers',
            'activeOffers',
            'expiredOffers',
            'totalViews',
            'totalUses',
            'topStores'
        ));
    }
}
