<?php

if (!function_exists('setting')) {
    /**
     * Get setting value
     * 
     * @param string $key Format: 'group.key' or 'group.key.country_id'
     * @param mixed $default
     * @return mixed
     */
    function setting($key, $default = null)
    {
        $parts = explode('.', $key);
        $group = $parts[0] ?? null;
        $settingKey = $parts[1] ?? null;
        $countryId = $parts[2] ?? current_filtered_country_id();

        if (!$group || !$settingKey) {
            return $default;
        }

        return \App\Models\Setting::getValue($group, $settingKey, $default, $countryId);
    }
}

if (!function_exists('app_name')) {
    /**
     * Get application name from settings or .env
     * 
     * @return string
     */
    function app_name()
    {
        // First try from settings database
        $settingName = setting('general.app_name');
        
        // Fallback to .env file
        if (empty($settingName)) {
            $settingName = config('app.name', 'Qansah');
        }
        
        return $settingName;
    }
}

if (!function_exists('app_logo')) {
    /**
     * Get application logo from settings
     * 
     * @return string
     */
    function app_logo()
    {
        $logo = setting('general.site_logo');
        
        // Fallback to default logo if not set
        if (empty($logo)) {
            $logo = '/img/logo.svg';
        }
        
        return $logo;
    }
}

if (!function_exists('app_favicon')) {
    /**
     * Get application favicon from settings
     * 
     * @return string
     */
    function app_favicon()
    {
        $favicon = setting('general.site_favicon');
        
        // Fallback to default favicon if not set
        if (empty($favicon)) {
            $favicon = '/img/favicon.ico';
        }
        
        return $favicon;
    }
}

if (!function_exists('app_version')) {
    /**
     * Get application version from settings or .env
     * 
     * @return string
     */
    function app_version()
    {
        // First try from settings database
        $settingVersion = setting('general.app_version');
        
        // Fallback to .env file
        if (empty($settingVersion)) {
            $settingVersion = config('app.api_version', env('API_VERSION', '1.0.0'));
        }
        
        return $settingVersion;
    }
}

if (!function_exists('seo_title')) {
    /**
     * Generate SEO title for current country
     * 
     * @param string $pageTitle
     * @param int|null $countryId
     * @return string
     */
    function seo_title($pageTitle = '', $countryId = null)
    {
        $template = setting('seo.meta_title_template', '%s - ' . app_name(), $countryId);
        
        if (empty($pageTitle)) {
            return str_replace(' - %s', '', $template);
        }
        
        return str_replace('%s', $pageTitle, $template);
    }
}

if (!function_exists('seo_description')) {
    /**
     * Get SEO description for current country
     * 
     * @param int|null $countryId
     * @return string
     */
    function seo_description($countryId = null)
    {
        return setting('seo.meta_description_template', 'Discover the best deals and coupons', $countryId);
    }
}

if (!function_exists('seo_keywords')) {
    /**
     * Get SEO keywords for current country
     * 
     * @param int|null $countryId
     * @return string
     */
    function seo_keywords($countryId = null)
    {
        return setting('seo.meta_keywords', 'coupons,deals,discounts,savings', $countryId);
    }
}

if (!function_exists('google_analytics_id')) {
    /**
     * Get Google Analytics ID
     * 
     * @return string
     */
    function google_analytics_id()
    {
        return setting('seo.google_analytics_id', '');
    }
}

if (!function_exists('facebook_pixel_id')) {
    /**
     * Get Facebook Pixel ID
     * 
     * @return string
     */
    function facebook_pixel_id()
    {
        return setting('seo.facebook_pixel_id', '');
    }
}

if (!function_exists('google_tag_manager_id')) {
    /**
     * Get Google Tag Manager ID
     * 
     * @return string
     */
    function google_tag_manager_id()
    {
        return setting('seo.google_tag_manager', '');
    }
}

if (!function_exists('is_maintenance_mode')) {
    /**
     * Check if app is in maintenance mode
     * 
     * @return bool
     */
    function is_maintenance_mode()
    {
        return setting('general.maintenance_mode', false);
    }
}

if (!function_exists('maintenance_message')) {
    /**
     * Get maintenance mode message
     * 
     * @return string
     */
    function maintenance_message()
    {
        return setting('general.maintenance_message', 'الموقع تحت الصيانة، سنعود قريباً');
    }
}

if (!function_exists('app_description')) {
    /**
     * Get app description
     * 
     * @param int|null $countryId
     * @return string
     */
    function app_description($countryId = null)
    {
        return setting('general.app_description', 'أفضل منصة للكوبونات والعروض', $countryId);
    }
}

if (!function_exists('social_urls')) {
    /**
     * Get all social media URLs
     * 
     * @return array
     */
    function social_urls()
    {
        return [
            'facebook' => setting('social.facebook_url', ''),
            'twitter' => setting('social.twitter_url', ''),
            'instagram' => setting('social.instagram_url', ''),
            'linkedin' => setting('social.linkedin_url', ''),
            'youtube' => setting('social.youtube_url', ''),
        ];
    }
}